/*\file Ll2xy.c
 *\brief: lat/long to x/y coordinate mex module.
 */
#include "./Ll2xy.h"

void mexFunction( int nlhs, mxArray* plhs[], int nrhs, const mxArray* prhs[]){

	int i,verbose=1;

	/*input: */
	double  *lat=NULL,*lon=NULL;
	int     nlat,nlon,ncoord;
	int     sgn;

	Options* options=NULL;
	double   cm=0.,sp=0.;

	/* output: */
	double  *x=NULL,*y=NULL;
	int     iret=0;

	/*Boot module: */
	MODULEBOOT();

	/*checks on arguments on the matlab side: */
	if (nlhs > NLHS) {
		Ll2xyUsage();
		_error_("Ll2xy usage error");
	}
	if (nrhs < NRHS) {
		Ll2xyUsage();
		_error_("Ll2xy usage error");
	}

	/*Input datasets: */
	if (verbose) printf("Fetching inputs:\n");
	FetchData(&lat,&nlat,LAT_IN);
	if (verbose)
		if   (nlat == 1) printf("  lat=%g\n",lat[0]);
		else             printf("  lat=[%d values]\n",nlat);
//	for (i=0; i<nlat; i++) printf("  lat[%d]=%g\n",i,lat[i]);
	FetchData(&lon,&nlon,LON_IN);
	if (verbose)
		if   (nlon == 1) printf("  lon=%g\n",lon[0]);
		else             printf("  lon=[%d values]\n",nlon);
//	for (i=0; i<nlon; i++) printf("  lon[%d]=%g\n",i,lon[i]);
	FetchData(&sgn,SGN_IN);
	if (verbose) printf("  sgn=%d\n",sgn);

	if (verbose) printf("Parsing options:\n");
	options=new Options(NRHS,nrhs,prhs);
	if (options->Size()) for(i=0;i<options->Size();i++) ((Option*)options->GetObjectByOffset(i))->DeepEcho();
	/*  defaults are in x-layer, so don't duplicate them here, and only use user values if both have been specified  */
	if (options->GetOption("central_meridian") || options->GetOption("standard_parallel")) {
		options->Get(&cm,"central_meridian");
		if (verbose) printf("  cm=%g\n",cm);
		options->Get(&sp,"standard_parallel");
		if (verbose) printf("  sp=%g\n",sp);
	}

	/*some checks*/
	if (verbose) printf("Checking inputs:\n");

	if   (nlat != nlon) _error_("Must have same number of lat[%d] and lon[%d] coordinates.",nlat,nlon);
	else                ncoord=nlat;
	if (sgn != +1 && sgn != -1) _error_("Hemisphere sgn=%d must be +1 (north) or -1 (south).",sgn);
	if (fabs(cm)      > 180.) _error_("Central meridian cm=%g must be between -180 (west) and +180 (east) degrees.",cm);
	if (sp < 0. || sp >  90.) _error_("Standard parallel sp=%g must be between 0 and 90 degrees (in specified hemisphere).",sp);

	x=(double *)xmalloc(ncoord*sizeof(double));
	y=(double *)xmalloc(ncoord*sizeof(double));

	/* Run core computations: */
	if (verbose) printf("Calling core:\n");
	if (options->GetOption("central_meridian") && options->GetOption("standard_parallel"))
		iret=Ll2xyx(x,y,
					lat,lon,ncoord,
					sgn,cm,sp);
	else
		iret=Ll2xyx(x,y,
					lat,lon,ncoord,
					sgn);
	if (verbose) printf("  iret=%d\n",iret);

	/*Write data: */
	WriteData(X_OUT,x,ncoord);
	WriteData(Y_OUT,y,ncoord);

	/*Clean-up*/
	delete options;

	/*end module: */
	MODULEEND();
}

void Ll2xyUsage(void)
{
	_printf_(true,"Ll2xy - lat/long to x/y coordinate transformation module:\n");
	_printf_(true,"\n");
	_printf_(true,"   This module transforms lat/long to x/y coordinates.\n");
	_printf_(true,"\n");
	_printf_(true,"   Usage:\n");
	_printf_(true,"      [x,y]=Ll2xy(lat,lon,sgn,'param name',param,...);\n");
	_printf_(true,"\n");
	_printf_(true,"      lat         latitude coordinates (double vector)\n");
	_printf_(true,"      lon         longitude coordinates (double vector)\n");
	_printf_(true,"      sgn         sign for hemisphere (double, +1 (north) or -1 (south))\n");
	_printf_(true,"\n");
	_printf_(true,"      central_meridian     central meridian (double, optional, but must specify with sp)\n");
	_printf_(true,"      standard_parallel    standard parallel (double, optional, but must specify with cm)\n");
	_printf_(true,"\n");
	_printf_(true,"      x           x coordinates (double vector)\n");
	_printf_(true,"      y           y coordinates (double vector)\n");
	_printf_(true,"\n");
	_printf_(true,"   Examples:\n");
	_printf_(true,"      [x,y]=Ll2xy(lat,lon, 1);\n");
	_printf_(true,"      [x,y]=Ll2xy(lat,lon, 1,'central_meridian',45,'standard_parallel',70);\n");
	_printf_(true,"      [x,y]=Ll2xy(lat,lon,-1,'central_meridian', 0,'standard_parallel',71);\n");
	_printf_(true,"\n");
}

