function md=parametercontrolB(md,varargin),
%PARAMETERCONTROLB - parameterization for control method on B
%
%   It is possible to specify the number of steps, values for the
%   minimum and maximum values of B, the 
%   kind of cm_responses to use or the the optscal.
%   
%   Usage:
%       md=parametercontrolB(md,varargin)
%
%   Example:
%      md=parametercontrolB(md)
%      md=parametercontrolB(md,'nsteps',20,'cm_responses',0)
%      md=parametercontrolB(md,'cm_min',10,'cm_max',10^8,'cm_jump',0.99,'maxiter',20)
%      md=parametercontrolB(md,eps_cm',10^-4,'optscal',[10^7 10^8])
%
%   See also  PARAMETERCONTROLDRAG

%process options
options=pairoptions(varargin{:});

%control type
md.control_type=RheologyBbarEnum;

%weights
weights=getfieldvalue(options,'weights',ones(md.numberofnodes,1));
if (length(weights)~=md.numberofnodes)
	md.weights=ones(md.numberofnodes,1);
else
	md.weights=weights;
end

%nsteps
nsteps=getfieldvalue(options,'nsteps',100);
if (length(nsteps)~=1 | nsteps<=0 | floor(nsteps)~=nsteps)
	md.nsteps=100;
else
	md.nsteps=nsteps;
end


%cm_min
cm_min=getfieldvalue(options,'cm_min',paterson(273.15+5)*ones(md.numberofnodes,1));
if (length(cm_min)==1)
	md.cm_min=cm_min*ones(md.numberofnodes,1);
elseif (length(cm_min)==md.numberofnodes)
	md.cm_min=cm_min;
else
	md.cm_min=cm_min;
end

%cm_max
cm_max=getfieldvalue(options,'cm_max',paterson(273.15-70)*ones(md.numberofnodes,1));
if (length(cm_max)==1)
	md.cm_max=cm_max*ones(md.numberofnodes,1);
elseif (length(cm_max)==md.numberofnodes)
	md.cm_max=cm_max;
else
	md.cm_max=cm_max;
end

%eps_cm
eps_cm=getfieldvalue(options,'eps_cm',NaN);
if (length(eps_cm)~=1 | eps_cm<0 )
	md.eps_cm=NaN;
else
	md.eps_cm=eps_cm;
end

%maxiter
maxiter=getfieldvalue(options,'maxiter',10*ones(md.nsteps,1));
if (any(maxiter<0) | any(floor(maxiter)~=maxiter))
	md.maxiter=10*ones(md.nsteps,1);
else
	md.maxiter=repmat(maxiter(:),md.nsteps,1);
	md.maxiter(md.nsteps+1:end)=[];
end

%cm_jump
cm_jump=getfieldvalue(options,'cm_jump',0.9*ones(md.nsteps,1));
if ~isreal(cm_jump)
	md.cm_jump=0.9*ones(md.nsteps,1);
else
	md.cm_jump=repmat(cm_jump(:),md.nsteps,1);
	md.cm_jump(md.nsteps+1:end)=[];
end

%cm_responses
found=0;
if exist(options,'cm_responses'),
	cm_responses=getfieldvalue(options,'cm_responses');
	if ~any(~ismember(cm_responses,[ SurfaceAbsVelMisfitEnum, SurfaceRelVelMisfitEnum, SurfaceLogVelMisfitEnum, SurfaceLogVxVyMisfitEnum, SurfaceAverageVelMisfitEnum ]));
		md.cm_responses=repmat(cm_responses(:),md.nsteps,1);
		md.cm_responses(md.nsteps+1:end)=[];
		found=1;
	end
end
if ~found
	third=ceil(md.nsteps/3);
	md.cm_responses=[...
		SurfaceLogVelMisfitEnum*ones(third,1);...
		SurfaceAbsVelMisfitEnum*ones(third,1);...
		repmat([SurfaceAbsVelMisfitEnum;SurfaceAbsVelMisfitEnum;SurfaceLogVelMisfitEnum;SurfaceAbsVelMisfitEnum],third,1)...
		];
	md.cm_responses(md.nsteps+1:end)=[];
end

%optscal
found=0;
if exist(options,'optscal'),
	optscal=getfieldvalue(options,'optscal');
	if ~any(optscal<0),
		md.optscal=repmat(optscal(:),md.nsteps,1);
		md.optscal(md.nsteps+1:end)=[];
		found=1;
	end
end
if ~found
	third=ceil(md.nsteps/3);
	md.optscal=[2*10^8*ones(3,1);10^8*ones(third-3,1);10^7*ones(2*third,1);];
	md.optscal(md.nsteps+1:end)=[];
end
