function marshall(md)
%MARSHALL - output JPL-package compatible binary file from @model md, for certain solution type.
%
%   The routine creates a JPL-package compatible binary file from @model md
%   This binary file will be used for parallel runs in JPL-package
%
%   Usage:
%      marshall(md)

%some checks on list of arguments
if ((nargin~=3) & (nargout~=0))
	help marshall;
	error('marshall error message');
end

disp(['marshalling file ' md.name '.bin']);

%open file for binary writing
fid=fopen([ md.name '.bin'],'wb');
if fid==-1,
	error(['marshall error message: could not open ' [md.name '.bin'],' file for binary writing']);
end

WriteData(fid,md.dim,'Integer','dim');
WriteData(fid,md.numberofnodes,'Integer','numberofnodes');
WriteData(fid,md.numberofelements,'Integer','numberofelements');
WriteData(fid,md.x,'Mat','x');
WriteData(fid,md.y,'Mat','y');
WriteData(fid,md.z,'Mat','z');
WriteData(fid,md.elements,'Mat','elements');
WriteData(fid,md.elementconnectivity,'Mat','elementconnectivity');
WriteData(fid,md.elements_type,'Mat','elements_type');
WriteData(fid,md.vertices_type,'Mat','vertices_type');
WriteData(fid,md.nodeonhutter,'Mat','nodeonhutter');
WriteData(fid,md.nodeonmacayeal,'Mat','nodeonmacayeal');
if md.dim==3,
	WriteData(fid,md.numberofelements2d,'Integer','numberofelements2d');
	WriteData(fid,md.numberofnodes2d,'Integer','numberofnodes2d');
	WriteData(fid,md.elements2d,'Mat','elements2d');
	WriteData(fid,md.numlayers,'Integer','numlayers');
	WriteData(fid,md.nodeonpattyn,'Mat','nodeonpattyn');
end
WriteData(fid,md.upperelements,'Mat','upperelements');
WriteData(fid,md.lowerelements,'Mat','lowerelements');
WriteData(fid,md.elementonbed,'Mat','elementonbed');
WriteData(fid,md.elementonsurface,'Mat','elementonsurface');
WriteData(fid,md.nodeonbed,'Mat','nodeonbed');
WriteData(fid,md.nodeonsurface,'Mat','nodeonsurface');
WriteData(fid,md.nodeonstokes,'Mat','nodeonstokes');
WriteData(fid,md.borderstokes,'Mat','borderstokes');

WriteData(fid,md.thickness,'Mat','thickness');
WriteData(fid,md.thickness_coeff,'Mat','thickness_coeff');
WriteData(fid,md.surface,'Mat','surface');
WriteData(fid,md.bed,'Mat','bed');

WriteData(fid,md.vx_obs,'Mat','vx_obs');
WriteData(fid,md.vy_obs,'Mat','vy_obs');
WriteData(fid,md.thickness_obs,'Mat','thickness_obs');

WriteData(fid,md.vx,'Mat','vx');
WriteData(fid,md.vy,'Mat','vy');
WriteData(fid,md.vz,'Mat','vz');
WriteData(fid,md.pressure,'Mat','pressure');
WriteData(fid,md.temperature,'Mat','temperature');
WriteData(fid,md.waterfraction,'Mat','waterfraction');

WriteData(fid,md.drag_type,'Integer','drag_type');
WriteData(fid,md.drag_coefficient,'Mat','drag_coefficient');
WriteData(fid,md.drag_p,'Mat','drag_p');
WriteData(fid,md.drag_q,'Mat','drag_q');

WriteData(fid,md.elementoniceshelf,'Mat','elementoniceshelf');
WriteData(fid,md.elementonwater,'Mat','elementonwater');
WriteData(fid,md.nodeonicesheet,'Mat','nodeonicesheet');
WriteData(fid,md.nodeoniceshelf,'Mat','nodeoniceshelf');
WriteData(fid,md.nodeonwater,'Mat','nodeonwater');

WriteData(fid,md.spcvelocity,'Mat','spcvelocity');
WriteData(fid,md.spctemperature,'Mat','spctemperature');
WriteData(fid,md.spcthickness,'Mat','spcthickness');
WriteData(fid,md.spcwatercolumn,'Mat','spcwatercolumn');
WriteData(fid,md.diagnostic_ref,'Mat','diagnostic_ref');

WriteData(fid,md.pressureload,'Mat','pressureload');
WriteData(fid,md.edges,'Mat','edges');

WriteData(fid,md.geothermalflux,'Mat','geothermalflux');
WriteData(fid,md.surface_accumulation_rate,'Mat','surface_accumulation_rate');
WriteData(fid,md.surface_ablation_rate,'Mat','surface_ablation_rate');
WriteData(fid,md.surface_mass_balance,'Mat','surface_mass_balance');
WriteData(fid,md.gl_melting_rate,'Scalar','gl_melting_rate');
WriteData(fid,md.basal_melting_rate,'Mat','basal_melting_rate');
WriteData(fid,md.basal_melting_rate_correction_apply,'Integer','basal_melting_rate_correction_apply');

%deal with forcings
if ~isempty(md.forcings),
	forcingnames=fieldnames(md.forcings);
	numforcings=length(forcingnames);
	forcingtypes=zeros(numforcings,1);
	for i=1:numforcings,
		switch (forcingnames{i})
			case 'surface_accumulation_rate'
				forcingtypes(i)=SurfaceAccumulationRateEnum;
			case 'surface_ablation_rate'
				forcingtypes(i)=SurfaceAblationRateEnum;
			case 'surface_mass_balance'
				forcingtypes(i)=SurfaceMassBalanceEnum;
			case 'basal_melting_rate'
				forcingtypes(i)=BasalMeltingRateEnum;
			otherwise
				error(['forcing ' forcingnames{i} ' not supported yet']);
		end
	end
	WriteData(fid,numforcings,'Integer','numforcings');
	WriteData(fid,forcingtypes,'Mat','forcingtypes');

	for i=1:numforcings,
		forcing=md.forcings.(forcingnames{i});
		forcingname=EnumToString(forcingtypes(i));

		WriteData(fid,size(forcing,2),'Integer',['forcing_' forcingname '_num_time_steps']);
		WriteData(fid,forcing(end,:),'Mat',['forcing_' forcingname '_time_steps']);
		for j=1:size(forcing,2),
			WriteData(fid,forcing(1:end-1,j),'Mat',['forcing_' forcingname '_' num2str(j)]);
		end
	end
else
	WriteData(fid,0,'Integer','numforcings');
end

if md.basal_melting_rate_correction_apply,
	WriteData(fid,md.basal_melting_rate_correction,'Mat','basal_melting_rate_correction');
end
WriteData(fid,md.dhdt,'Mat','dhdt');
WriteData(fid,md.watercolumn,'Mat','watercolumn');

%Transient
WriteData(fid,md.isdiagnostic,'Integer','isdiagnostic');
WriteData(fid,md.isprognostic,'Integer','isprognostic');
WriteData(fid,md.isthermal,'Integer','isthermal');

%Get materials
WriteData(fid,md.rho_water,'Scalar','rho_water');
WriteData(fid,md.rho_ice,'Scalar','rho_ice');
WriteData(fid,md.g,'Scalar','g');
WriteData(fid,md.rheology_B,'Mat','rheology_B');
WriteData(fid,md.rheology_n,'Mat','rheology_n');
WriteData(fid,md.rheology_law,'Integer','rheology_law');

%Control methods
WriteData(fid,md.control_analysis,'Integer','control_analysis');
WriteData(fid,numel(md.control_type),'Integer','num_control_type');
WriteData(fid,size(md.cm_responses,2),'Integer','num_cm_responses');
WriteData(fid,md.control_type,'Mat','control_type');

%Write solution parameters
WriteData(fid,md.cm_responses,'Mat','cm_responses');
WriteData(fid,md.weights,'Mat','weights');
WriteData(fid,md.cm_jump,'Mat','cm_jump');
WriteData(fid,md.yts,'Scalar','yts');
WriteData(fid,md.meanvel,'Scalar','meanvel');
WriteData(fid,md.epsvel,'Scalar','epsvel');
WriteData(fid,VerboseToBinary(md.verbose),'Integer','verbose_binary');
WriteData(fid,md.output_frequency,'Integer','output_frequency');
WriteData(fid,md.artificial_diffusivity,'Integer','artificial_diffusivity');
WriteData(fid,md.prognostic_DG,'Integer','prognostic_DG');
WriteData(fid,md.nsteps,'Integer','nsteps');
WriteData(fid,md.eps_cm,'Scalar','eps_cm');
WriteData(fid,md.tolx,'Scalar','tolx');
WriteData(fid,md.maxiter,'Mat','maxiter');
WriteData(fid,md.cm_min,'Mat','cm_min');
WriteData(fid,md.cm_max,'Mat','cm_max');
WriteData(fid,md.cm_gradient,'Integer','cm_gradient');
WriteData(fid,md.eps_res,'Scalar','eps_res');
WriteData(fid,md.eps_rel,'Scalar','eps_rel');
WriteData(fid,md.eps_abs,'Scalar','eps_abs');
WriteData(fid,md.max_nonlinear_iterations,'Scalar','max_nonlinear_iterations');
WriteData(fid,md.dt,'Scalar','dt');
WriteData(fid,md.ndt,'Scalar','ndt');
WriteData(fid,md.time_adapt,'Integer','time_adapt');
WriteData(fid,md.cfl_coefficient,'Scalar','cfl_coefficient');
WriteData(fid,md.hydrostatic_adjustment,'Integer','hydrostatic_adjustment');
WriteData(fid,md.penalty_offset,'Scalar','penalty_offset');
WriteData(fid,md.penalty_melting,'Scalar','penalty_melting');
WriteData(fid,md.penalty_lock,'Integer','penalty_lock');
WriteData(fid,md.sparsity,'Scalar','sparsity');
WriteData(fid,md.connectivity,'Integer','connectivity');
WriteData(fid,md.lowmem,'Integer','lowmem');
WriteData(fid,md.optscal,'Mat','optscal');
WriteData(fid,md.viscosity_overshoot,'Scalar','viscosity_overshoot');
WriteData(fid,md.stokesreconditioning,'Scalar','stokesreconditioning');
WriteData(fid,md.shelf_dampening,'Integer','shelf_dampening');
WriteData(fid,md.waitonlock,'Scalar','waitonlock');
WriteData(fid,md.kff,'Integer','kff');

%Thermal parameters
WriteData(fid,md.beta,'Scalar','beta');
WriteData(fid,md.meltingpoint,'Scalar','meltingpoint');
WriteData(fid,md.referencetemperature,'Scalar','referencetemperature');
WriteData(fid,md.latentheat,'Scalar','latentheat');
WriteData(fid,md.heatcapacity,'Scalar','heatcapacity');
WriteData(fid,md.thermalconductivity,'Scalar','thermalconductivity');
WriteData(fid,md.min_thermal_constraints,'Integer','min_thermal_constraints');
WriteData(fid,md.min_mechanical_constraints,'Integer','min_mechanical_constraints');
WriteData(fid,md.mixed_layer_capacity,'Scalar','mixed_layer_capacity');
WriteData(fid,md.thermal_exchange_velocity,'Scalar','thermal_exchange_velocity');
WriteData(fid,md.stabilize_constraints,'Integer','stabilize_constraints');

%Hydrology parameters
WriteData(fid,md.hydro_kn,'Scalar','hydro_kn');
WriteData(fid,md.hydro_p,'Scalar','hydro_p');
WriteData(fid,md.hydro_q,'Scalar','hydro_q');
WriteData(fid,md.hydro_CR,'Scalar','hydro_CR');
WriteData(fid,md.hydro_n,'Scalar','hydro_n');

%elements type
WriteData(fid,md.ishutter,'Integer','ishutter');
WriteData(fid,md.ismacayealpattyn,'Integer','ismacayealpattyn');
WriteData(fid,md.isstokes,'Integer','isstokes');

%grounding line migration: 
WriteData(fid,md.gl_migration,'Integer','gl_migration');
if(md.gl_migration~=NoneEnum),
	WriteData(fid,md.bathymetry,'Mat','bathymetry');
end

%Rifts
WriteData(fid,md.riftinfo,'Mat','riftinfo');

%parameter output.
md.numoutput=length(md.parameteroutput);
WriteData(fid,md.numoutput,'Integer','numoutput');
if(md.numoutput),
	for i=1:md.numoutput,
		WriteData(fid,md.parameteroutput{i},'String',['parameteroutput_' num2str(i-1)]);
	end
end

%Qmu: the rest will be handle by qmumarshall
WriteData(fid,md.qmu_analysis,'Integer','qmu_analysis');
WriteData(fid,md.name,'String','name');

%Get penalties
WriteData(fid,md.penalties,'Mat','penalties');

%input and output file names
WriteData(fid,md.inputfilename,'String','inputfilename');
WriteData(fid,md.outputfilename,'String','outputfilename');

%i/o: 
WriteData(fid,md.io_gather,'Integer','io_gather');


%close file
st=fclose(fid);
if st==-1,
	error(['marshall error message: could not close file ' [md.name '.bin']]);
end
