/*!\file:  objectivefunctionC
 * \brief  objective function that returns a misfit, for a certain parameter.
 */ 

/*include files: {{{1*/
#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "../toolkits/toolkits.h"
#include "../objects/objects.h"
#include "../shared/shared.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../solvers/solvers.h"
#include "./solutions.h"
#include "../modules/modules.h"
#include "../include/include.h"
/*}}}*/

double objectivefunctionC(double search_scalar,OptArgs* optargs){

	int i;  
	
	/*output: */
	double J=0,Jplus;
	
	/*parameters: */
	int        num_responses;
	int        solution_type,analysis_type;
	bool       isstokes       = false;
	bool       conserve_loads = true;
	int       *responses      = NULL;
	FemModel  *femmodel       = NULL;

	/*Recover finite element model: */
	femmodel=optargs->femmodel;

	/*Recover parameters: */
	femmodel->parameters->FindParam(&num_responses,NumResponsesEnum);
	femmodel->parameters->FindParam(&responses,NULL,NULL,StepResponsesEnum);
	femmodel->parameters->FindParam(&isstokes,IsStokesEnum);
	femmodel->parameters->FindParam(&analysis_type,AnalysisTypeEnum);
	femmodel->parameters->FindParam(&solution_type,SolutionTypeEnum);

	/*set analysis type to compute velocity: */
	if (solution_type==SteadystateSolutionEnum || solution_type==DiagnosticSolutionEnum){
		femmodel->SetCurrentConfiguration(DiagnosticHorizAnalysisEnum);
	}
	else if (solution_type==BalancethicknessSolutionEnum){
		femmodel->SetCurrentConfiguration(BalancethicknessAnalysisEnum);
	}
	else{
		_error_("Solution %s not implemented yet",EnumToStringx(solution_type));
	}

	/*update parameter according to scalar: */ //false means: do not save control
	InputControlUpdatex(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,search_scalar,false);

	/*Run diagnostic with updated inputs: */
	if (solution_type==SteadystateSolutionEnum){
		diagnostic_core(femmodel);	//We need a 3D velocity!! (vz is required for the next thermal run)
	}
	else if (solution_type==DiagnosticSolutionEnum){
		solver_nonlinear(femmodel,conserve_loads); 
	}
	else if (solution_type==BalancethicknessSolutionEnum){
		solver_linear(femmodel); 
	}
	else{
		_error_("Solution %s not implemented yet",EnumToStringx(solution_type));
	}

	/*Compute misfit for this velocity field.*/
	for(int i=0;i<num_responses;i++){
		CostFunctionx(&Jplus, femmodel->elements,femmodel->nodes, femmodel->vertices,femmodel->loads, femmodel->materials, femmodel->parameters,responses[i]);
		J+=Jplus;
	}

	/*Free ressources:*/
	xfree((void**)&responses);
	return J;
}
