%PFE class definition
%
%   Usage:
%      cluster=pfe();
%      cluster=pfe('np',3);
%      cluster=pfe('np',3,'login','username');

classdef pfe
    properties (SetAccess=public)  
		 % {{{1
		 name='pfe'
		 login='';
		 numnodes=20;
		 cpuspernode=8; 
		 port=1025;
		 queue='long';
		 time=12*60;
		 processor='neh';
		 codepath='';
		 executionpath='';
		 interactive=0;
		 bbftp=0;
		 numstreams=8;
		 hyperthreading=0;
	 end
	 properties (SetAccess=private) 
		 np=20*8;
		 % }}}
	 end
	 methods
		 function cluster=pfe(varargin) % {{{1

			 %initialize cluster using default settings if provided
			 if (exist('pfe_settings')==2), pfe_settings; end

			 %use provided options to change fields
			 options=pairoptions(varargin{:});
			 for i=1:size(options.list,1),
				 fieldname=options.list{i,1};
				 fieldvalue=options.list{i,2};
				 if ismember(fieldname,properties('pfe')),
					 cluster.(fieldname)=fieldvalue;
				 else
					 disp(['''' fieldname ''' is not a property of cluster pfe']);
				 end
			 end
		 end
		 %}}}
		 function disp(cluster) % {{{1
			 %  display the object
			 disp(sprintf('class ''%s'' object ''%s'' = ',class(cluster),inputname(1)));
			 disp(sprintf('    name: %s',cluster.name));
			 disp(sprintf('    login: %s',cluster.login));
			 disp(sprintf('    port: %i',cluster.port));
			 disp(sprintf('    numnodes: %i',cluster.numnodes));
			 disp(sprintf('    cpuspernode: %i',cluster.cpuspernode));
			 disp(sprintf('    np: %i',cluster.cpuspernode*cluster.numnodes));
			 disp(sprintf('    queue: %s',cluster.queue));
			 disp(sprintf('    time: %i',cluster.time));
			 disp(sprintf('    processor: %s',cluster.processor));
			 disp(sprintf('    codepath: %s',cluster.codepath));
			 disp(sprintf('    executionpath: %s',cluster.executionpath));
			 disp(sprintf('    interactive: %i',cluster.interactive));
			 disp(sprintf('    hyperthreading: %i',cluster.hyperthreading));
		 end
		 %}}}
		 function IsConsistent(cluster) % {{{1

			 available_queues={'long'};
			 queue_requirements_time=[7200];
			 queue_requirements_np=[2048];

			 QueueRequirements(available_queues,queue_requirements_time,queue_requirements_np,cluster.queue,cluster.np,cluster.time)

			 %now, check cluster.cpuspernode according to processor type
			 if (strcmpi(cluster.processor,'har') | strcmpi(cluster.processor,'neh')),
				 if cluster.hyperthreading,
					 if ((cluster.cpuspernode>16 ) | (cluster.cpuspernode<1)),
						 error('IsConsistent error message: cpuspernode should be between 1 and 16 for ''neh'' and ''har'' processors in hyperthreading mode');
					 end
				 else
					 if ((cluster.cpuspernode>8 ) | (cluster.cpuspernode<1)),
						 error('IsConsistent error message: cpuspernode should be between 1 and 8 for ''neh'' and ''har'' processors');
					 end
				 end
			 elseif strcmpi(cluster.processor,'wes'),
				 if cluster.hyperthreading,
					 if ((cluster.cpuspernode>24 ) | (cluster.cpuspernode<1)),
						 error('IsConsistent error message: cpuspernode should be between 1 and 24 for ''wes'' processors in hyperthreading mode');
					 end
				 else
					 if ((cluster.cpuspernode>12 ) | (cluster.cpuspernode<1)),
						 error('IsConsistent error message: cpuspernode should be between 1 and 12 for ''wes'' processors');
					 end
				 end
			 else
				 error('IsConsistent error message: unknown processor type, should be ''neh'',''wes'' or ''har''');
			 end

			 %Miscelaneous
			 if isempty(cluster.login), error('IsConsistent error message: login empty'); end
			 if isempty(cluster.codepath), error('IsConsistent error message: codepath empty'); end
			 if isempty(cluster.executionpath), error('IsConsistent error message: executionpath empty'); end

		 end
		 %}}}
		 function BuildQueueScript(cluster,md) % {{{1

			 %retrieve parameters 
			 modelname=md.name; 
			 solution_type=md.solution_type; 
			 mem_debug=md.mem_debug;

			 %compute number of processors
			 cluster.np=cluster.numnodes*cluster.cpuspernode;

			 %open file for writing: 
			 fid=fopen([modelname '.queue'],'w');

			 fprintf(fid,'#PBS -S /bin/bash\n');
%			 fprintf(fid,'#PBS -N %s\n',modelname);
			 fprintf(fid,'#PBS -l select=%i:ncpus=%i:model=%s\n',cluster.numnodes,cluster.cpuspernode,cluster.processor);
			 fprintf(fid,'#PBS -l walltime=%i\n',cluster.time*60); %walltime is in seconds.
			 fprintf(fid,'#PBS -W group_list=s1010\n');
			 fprintf(fid,'#PBS -m e\n');
			 fprintf(fid,'#PBS -o %s.outlog \n',modelname);
			 fprintf(fid,'#PBS -e %s.errlog \n\n',modelname);

			 fprintf(fid,'. /usr/share/modules/init/bash\n\n');

			 fprintf(fid,'module load comp-intel/11.1.046\n');
			 fprintf(fid,'module load mpi/mpt.1.25\n');
			 fprintf(fid,'module load math/intel_mkl_64_10.0.011\n\n');

			 fprintf(fid,'export PATH="$PATH:."\n\n');
			 fprintf(fid,'export MPI_GROUP_MAX=64\n\n');

			 fprintf(fid,'cd $PBS_O_WORKDIR\n\n');

			 fprintf(fid,'mpiexec -verbose -np %i %s/issm.exe %s $PBS_O_WORKDIR %s.bin %s.petsc %s.outbin %s.lock\n',cluster.np,cluster.codepath,EnumToString(solution_type),modelname,modelname,modelname,modelname);

			 if ~md.io_gather,
				 %concatenate the output files:
				 fprintf(fid,'cat %s.outbin.* > %s.outbin',modelname,modelname);
			 end

			 %close file
			 fclose(fid);


			 %in interactive mode, create a run file, and errlog and outlog file
			 if cluster.interactive,
				 fid=fopen([modelname '.run'],'w');
				 fprintf(fid,'mpiexec -verbose -np %i %s/issm.exe %s $PBS_O_WORKDIR %s.bin %s.petsc %s.outbin %s.lock\n',cluster.np,cluster.codepath,EnumToString(solution_type),modelname,modelname,modelname,modelname);
				 if ~md.io_gather,
					 %concatenate the output files:
					 fprintf(fid,'cat %s.outbin.* > %s.outbin',modelname,modelname);
				 end
				 fclose(fid);
				 fid=fopen([modelname '.errlog'],'w');
				 fclose(fid);
				 fid=fopen([modelname '.outlog'],'w');
				 fclose(fid);
			 end
		 end %}}}
		 function LaunchQueueJob(cluster,md,options)% {{{1
			 
			 %lauch command, to be executed via ssh
			 if ~cluster.interactive, 
				launchcommand=['cd ' cluster.executionpath ' && rm -rf ./' md.runtimename ' && mkdir ' md.runtimename ...
			                ' && cd ' md.runtimename ' && mv ../' md.runtimename '.tar.gz ./ && tar -zxf ' md.runtimename '.tar.gz  && qsub ' md.name '.queue '];
			else
				launchcommand=['cd ' cluster.executionpath '/Interactive' num2str(cluster.interactive) ' && tar -zxf ' md.runtimename '.tar.gz'];
			end

			if ~strcmpi(options.batch,'yes'),
				
				%compress the files into one zip.
				compressstring=['tar -zcf ' md.runtimename '.tar.gz ' md.name '.bin ' md.name '.queue '  md.name '.petsc '];
				if md.qmu_analysis,
					compressstring=[compressstring md.name '.qmu.in '];
				end
				if cluster.interactive,
					compressstring=[compressstring md.name '.run ' md.name '.errlog ' md.name '.outlog '];
				end
				system(compressstring);
				
				disp('uploading input file and queueing script');
				if cluster.interactive,
					directory=[cluster.executionpath '/Interactive' num2str(cluster.interactive)];
				else 
					directory=cluster.executionpath;
				end
				
				if ~cluster.bbftp,
					issmscpout(cluster.name,directory,cluster.login,cluster.port,{[md.runtimename '.tar.gz']});
				else
					issmbbftpout(cluster.name,directory,cluster.login,cluster.port,cluster.numstreams,{[md.runtimename '.tar.gz']});
				end
				
				disp('launching solution sequence on remote cluster');
				issmssh(cluster.name,cluster.login,cluster.port,launchcommand);

			else
				disp('batch mode requested: not launching job interactively');
				disp('launch solution sequence on remote cluster by hand');
			end
		 end
		 %}}}
		 function Download(cluster,md)% {{{1

			%some check
			if isempty(md.runtimename),
				if ~cluster.interactive,
					error('pfe Download error message: supply runtime name for results to be loaded!');
				end
			end

			%Figure out the  directory where all the files are in: 
			if ~cluster.interactive,
				directory=[cluster.executionpath '/' md.runtimename '/'];
			else
				directory=[cluster.executionpath '/Interactive' num2str(cluster.interactive) '/'];
			end

			%What packages are we picking up from remote cluster
			if ~cluster.interactive,
				packages={[md.name '.outlog'],[md.name '.errlog']};
			else
				packages={};
			end
			if md.qmu_analysis,
				packages{end+1}=[md.name '.qmu.err'];
				packages{end+1}=[md.name '.qmu.out'];
				if isfield(md.qmu_params,'tabular_graphics_data'),
					if md.qmu_params.tabular_graphics_data==true,
						packages{end+1}='dakota_tabular.dat'; 
					end
				end
			else
				packages{end+1}=[md.name '.outbin'];
			end

			%copy files from cluster to present directory
			if ~cluster.bbftp,
				issmscpin(cluster.name, cluster.login, cluster.port, directory, packages);
			else
				issmbbftpin(cluster.name, cluster.login, cluster.port, cluster.numstreams, directory, packages);
			end

		end %}}}
	end
end
