#!/bin/bash
#This bash script calls the nightlyrun.m matlab file to run our nightly test decks. 
#It then processes the results and sends an email to the Ice developpers.

#some functions
function timer() #{{{1
{
	if [[ $# -eq 0 ]]; then
		echo $(date '+%s')
	else
		local  stime=$1
		etime=$(date '+%s')

		if [[ -z "$stime" ]]; then stime=$etime; fi

		dt=$((etime - stime))
		ds=$((dt % 60))
		dm=$(((dt / 60) % 60))
		dh=$((dt / 3600))
		printf '%d:%02d:%02d' $dh $dm $ds
	fi
} #}}}
function todaydate() #{{{1
{
	suffix=`date | awk '{printf("%s-%s-%s  %s",$2,$3,$6,$4);}'`			 
	echo $suffix;			 
} #}}}
function host_name() #{{{1
{
	#return host name depending on the OS
	if [ "$1" = "wineric" ] 
	then
		HOST_NAME=`hostname`;
	else
		HOST_NAME=`hostname -s`;
	fi
	echo $HOST_NAME;
} #}}}

#Get configuration
#Source config file{{{1
if [ $# -ne 1 ];
then
	#no config file specified: exit
	echo "no config file specified. Exiting..." >&2 # Error message to stderr.
	exit 1
fi
if [ ! -f "$1" ]
then
	echo "File $1 not found!" >&2   # Error message to stderr.
	exit 1
fi 
source $1;
#}}}
#Export ISSM_* variables{{{1
export ISSM_DIR
export ISSM_TIER
export ISSM_ARCH
#}}}
#Initialize variables {{{1
TODAY=$(todaydate);
HOST_NAME=$(host_name $MACHINE);
START_TIME=$(timer);
ISSM_RELEASE=$(basename $(echo $REPOSITORY));
USER=$(whoami);
INIT_PATH=$(pwd);
#}}}

#Lauch installation
#Checkout/update/none ISSM if requested (ISSM_INSTALLATION){{{1
if [ "$ISSM_INSTALLATION" == "checkout" ]
then

	#Erase previous code and Fetch the new one
	rm -rf $ISSM_RELEASE
	svn checkout $REPOSITORY

elif [ "$ISSM_INSTALLATION" == "update" ]
then

	#only update ISSM
	cd $ISSM_TIER
	svn update

elif [ "$ISSM_INSTALLATION" == "none" ]
then

	#do nothing
	echo "Skipping ISSM installation"

else

	#Error message
	echo "ISSM_INSTALLATION supported values are: checkout, update and none. Exiting..." >&2 # Error message to stderr.
	exit 1

fi
#}}}
#Create cluster's settings{{{1
#create simpler  cluster.rc file, with only the cluster we are interested in.
cd $ISSM_TIER/test/NightlyRun/
cat << END > $HOST_NAME"_settings.m"
cluster.login='$USER';
cluster.codepath='$ISSM_TIER/bin';
cluster.executionpath='$EXECUTION_PATH';
END
#}}}
#Source environment variables with new matlab path {{{1
cd $ISSM_TIER/etc
source environment.sh MATLAB_DIR=$MATLAB_PATH
#}}}
#install/copy/none external packages    (ISSM_EXTERNALPACKAGES){{{1
if [ "$ISSM_EXTERNALPACKAGES" == "install" ]
then
	cd $ISSM_TIER/externalpackages
	for ep in $EXTERNALPACKAGES
	do 
		cd $ep
		case $ep in
			"petsc" )  cp configs/3.1-p7/$MACHINE/* .  ;;
			"mpich2" ) cp configs/1.0.2/$MACHINE/*  .  ;;
			*)         cp configs/$MACHINE/* .         ;;
		esac
		./install.sh $NUMCPUS_INSTALL
		cd ..
	done

	#3: install automake and autoconf
	cd $ISSM_TIER/devpackages
	make
	cd ..

elif [ "$ISSM_EXTERNALPACKAGES" == "copy" ]
then

	#erase externapackages, and link with externalpackages_dir
	cd $ISSM_TIER
	rm -rf externalpackages
	cp -Rf $EXTERNALPACKAGESDIR ./

	rm -rf devpackages
	cp -Rf $DEVPACKAGESDIR ./

elif [ "$ISSM_EXTERNALPACKAGES" == "none" ]
then

	#Do nothing
	echo "Skipping external packages installation"

else

	#Error message
	echo "ISSM_EXTERNALPACKAGES supported values are: install, copy and none. Exiting..." >&2 # Error message to stderr.
	exit 1

fi
#}}}
#ISSM compilation yes/no                (ISSM_COMPILATION) {{{1
if [ "$ISSM_COMPILATION" == "yes" ]
then

	cd $ISSM_TIER
	make clean
	make distclean
	sh scripts/automakererun.sh
	sh configs/$MACHINE/configure.sh #switch to petsc3 to debug

	#4: compile and install ISSM
	if [ "$MACHINE" = "wineric" ] 
	then
		cd $ISSM_TIER/src/c
		./intel-compile.sh
		cd $ISSM_TIER/src/mex
		make install
		cd $ISSM_TIER
	else
		make -j $NUMCPUS_INSTALL
		make -j $NUMCPUS_INSTALL install
	fi

elif [ "$ISSM_COMPILATION" == "no" ]
then

	#do nothing
	echo "Skipping ISSM compilation"

else

	#Error message
	echo "ISSM_COMPILATION supported values are: yes and no. Exiting..." >&2 # Error message to stderr.
	exit 1

fi
#}}}

#Prepare run
#create softlink to startup {{{1
cd $ISSM_TIER/test/NightlyRun/
ln -s $ISSM_TIER/startup.m .
#}}}
#Create nightlylog directory and info.log {{{1
#put installation elapsed time in info.log
INSTALL_TIME=$(timer)
ELAPSED_INSTALL=$(timer $START_TIME)
rm -rf $ISSM_TIER/nightlylog
mkdir  $ISSM_TIER/nightlylog
cat << END > $ISSM_TIER/nightlylog/info.log
today:     $(echo $TODAY)
user:      $(echo $USER)
host:      $(echo $HOST_NAME)
OS:        $(echo $MACHINE)
release:   $(echo $ISSM_RELEASE)
init_path: $(echo $INIT_PATH)
elapsed_install: $(echo $ELAPSED_INSTALL)
END
#}}}
#check NUMCPUS_RUN options {{{1
if [ "$NUMCPUS_RUN" = "" ]
then
	echo "NUMCPUS_RUN option not found, defaulting to NUMCPUS_RUN = 1"
	NUMCPUS_RUN=1
fi
#}}}

#Run tests
#Launch all tests on different cpus {{{1
for (( i=1;i<=$NUMCPUS_RUN;i++ ))
do
	#Launch matlab and the nightly run script
	cat > $ISSM_TIER/nightlylog/matlab_run$i.m << EOF
	warning off %necessary to avoid a info.log of several Go for parallel runs
	try,
	cd $ISSM_TIER/test/NightlyRun
	startup;
	$(if [ "$NROPTIONS" = ""  ]
	then
		echo "runme('output','nightly','rank',$i,'numprocs',$NUMCPUS_RUN);"
	else
		echo "runme($NROPTIONS,'output','nightly','rank',$i,'numprocs',$NUMCPUS_RUN);"
	fi
	)
	catch me,
		%An error occured, get report and exit
		message=getReport(me)
		directory=strsplit(pwd,'/');
		fid=fopen([ISSM_TIER '/nightlylog/matlaberror.log'], 'at');
		fprintf(fid,'\nMatlab error occured in: %s\n\n',directory{end});
		fprintf(fid,'%s',message);
		fclose(fid);
	end
	disp('MATLABEXITEDCORRECTLY');
	exit
EOF

	#Start run from nightlylog directory
	cd $ISSM_TIER/nightlylog/

	#Start test
	MATLAB_VERSION="7.6" #7.2,7.4,7.6 and 7.8
	/usr/local/pkgs/matlab-$MATLAB_VERSION/bin/matlab  -nojvm -nosplash  -r matlab_run$i -logfile matlab_log$i.log &
done

#wait until matlab closes
wait
#}}}
#concatenate all reports {{{1
cd $ISSM_TIER/nightlylog/
mv matlab_log1.log  matlab_log.log
for (( i=2;i<=$NUMCPUS_RUN;i++ ))
do
	cat matlab_log.log matlab_log$i.log > matlab_log.log.bak
	mv matlab_log.log.bak matlab_log.log
done
#}}}
#Complete info.log {{{1
if [ $(cat matlab_log.log | grep "MATLABEXITEDCORRECTLY" | wc -l) -eq $NUMCPUS_RUN ]
then
	MATLABCRASH=0
else
	MATLABCRASH=1
fi
ELAPSED_RUN=$(timer $INSTALL_TIME)
ELAPSED_TOTAL=$(timer $START_TIME)
cat << END >>  $ISSM_TIER/nightlylog/info.log
elapsed_run:   $(echo $ELAPSED_RUN)
elapsed_total: $(echo $ELAPSED_TOTAL)
matlab_crash:  $(echo $MATLABCRASH)
END
#}}}

#Send Report
#Build html report {{{1
cd $ISSM_TIER/nightlylog/
sh ../scripts/report.sh
echo "html report located in $ISSM_TIER/nightlylog/report.html"
#}}}
#send mail if requested                  (SKIPMAIL) {{{1
if [ "$SKIPMAIL" != "yes" ]
then
	echo "sending report..."
	source $ISSM_TIER/cron/mailinglist
	for i in `echo $MAILINGLIST`; do

		if [ "$MACHINE" = "wineric" ]
		then
			email -html -f "ISSM Nightly run $EMAIL_ADRESS" -s "Nightly runs of $ISSM_RELEASE , configuration: $MACHINE, host: $HOST_NAME, user: $USER. " $i < $ISSM_TIER/test/Verification/NightlyRun/report.html
		else 
			if [ "$MACHINE" = "astrid" ]
			then
cat - $ISSM_TIER/nightlylog/report.html <<HERE | /usr/lib/sendmail  -oi -t
From: "ISSM Nightly run" <$EMAIL_ADRESS>
To: $i
Subject: Nightly runs of $ISSM_RELEASE, configuration: $MACHINE, host: $HOST_NAME, user: $USER.
Mime-Version: 1.0
Content-Type: text/html
HERE
		else
cat - $ISSM_TIER/nightlylog/report.html <<HERE | /usr/sbin/sendmail  -oi -t
From: "ISSM Nightly run" <$EMAIL_ADRESS>
To: $i
Subject: Nightly runs of $ISSM_RELEASE, configuration: $MACHINE, host: $HOST_NAME, user: $USER.
Mime-Version: 1.0
Content-Type: text/html
HERE
			fi
		fi
	done
fi
#}}}
