/*!\file: transient_2d_core.cpp
 * \brief: core of the transient_2d solution 
 */ 

#undef __FUNCT__ 
#define __FUNCT__ "transient_2d_core"

#include "../toolkits/toolkits.h"
#include "../objects/objects.h"
#include "../shared/shared.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "./parallel.h"
#include "../issm.h"

void transient_core_2d(DataSet* results,FemModel* fems, ParameterInputs* inputs){

	extern int my_rank;

	/*fem models: */
	FemModel* fem_dh=NULL;
	FemModel* fem_dv=NULL;
	FemModel* fem_dhu=NULL;
	FemModel* fem_ds=NULL;
	FemModel* fem_sl=NULL;
	FemModel* fem_p=NULL;

	/*output: */
	Result* result=NULL;

	/*solutions: */
	int step;
	double time;
	
	Vec u_g=NULL;
	double* u_g_serial=NULL;
	Vec p_g=NULL;
	double* p_g_serial=NULL;
	Vec h_g=NULL;
	Vec h_g_intermediary=NULL;
	double* h_g_serial=NULL;
	Vec s_g=NULL;
	double* s_g_serial=NULL;
	Vec b_g=NULL;
	double* b_g_serial=NULL;


	/*intermediary: */
	double finaltime;
	double dt;
	double yts;
	DataSet* diagnostic_results=NULL;
	DataSet* prognostic_results=NULL;


	/*flags: */
	int debug=0;
	int numberofnodes;

	/*dof recovery: */
	int dof01[2]={0,1};
	int dof2[1]={2};
	int dof012[3]={0,1,2};
	int dof3[1]={3};

	/*recover fem models: */
	fem_dh=fems+0;
	fem_dv=fems+1;
	fem_ds=fems+2;
	fem_dhu=fems+3;
	fem_sl=fems+4;
	fem_p=fems+5;

	//first recover parameters common to all solutions
	fem_dh->parameters->FindParam((void*)&debug,"debug");
	fem_dh->parameters->FindParam((void*)&finaltime,"ndt");
	fem_dh->parameters->FindParam((void*)&dt,"dt");
	fem_dh->parameters->FindParam((void*)&yts,"yts");
	fem_dh->parameters->FindParam((void*)&numberofnodes,"numberofnodes");

	/*initialize: */
	step=1;
	time=0;
	
	fem_p->parameters->FindParam((void*)&u_g_serial,"u_g");
	u_g=SerialToVec(u_g_serial,2*numberofnodes); xfree((void**)&u_g_serial);

	fem_p->parameters->FindParam((void*)&p_g_serial,"p_g");
	p_g=SerialToVec(p_g_serial,1*numberofnodes); xfree((void**)&p_g_serial);

	fem_p->parameters->FindParam((void*)&h_g_serial,"h_g");
	h_g=SerialToVec(h_g_serial,1*numberofnodes); xfree((void**)&h_g_serial);

	fem_p->parameters->FindParam((void*)&s_g_serial,"s_g");
	s_g=SerialToVec(s_g_serial,1*numberofnodes); xfree((void**)&s_g_serial);

	fem_p->parameters->FindParam((void*)&b_g_serial,"b_g");
	b_g=SerialToVec(b_g_serial,1*numberofnodes); xfree((void**)&b_g_serial);

	/*Add all these initial vectors in the results!: */
	result=new Result(results->Size()+1,time,step,"u_g",u_g); results->AddObject(result);
	result=new Result(results->Size()+1,time,step,"p_g",p_g); results->AddObject(result);
	result=new Result(results->Size()+1,time,step,"h_g",h_g); results->AddObject(result);
	result=new Result(results->Size()+1,time,step,"s_g",s_g); results->AddObject(result);
	result=new Result(results->Size()+1,time,step,"b_g",b_g); results->AddObject(result);

	while(time<finaltime+dt){ //make sure we run up to finaltime.
	
		if(debug)_printf_("%s%g%s%i%s%g\n","time [yr]: ",time/yts,"    iteration number: ",step,"/",floor(finaltime/dt));

		step+=1;
		time+=dt;

		//update inputs
		inputs->Add("thickness",h_g,1,numberofnodes);
		inputs->Add("surface",s_g,1,numberofnodes);
		inputs->Add("bed",b_g,1,numberofnodes);
		inputs->Add("velocity",u_g,2,numberofnodes);

		//Deal with velocities.
		diagnostic_results=new DataSet(ResultsEnum()); 
		diagnostic_core(diagnostic_results,fems, inputs);

		//extract u_g and p_g from diagnostic_results, and erase diagnostic_results;
		diagnostic_results->FindResult(&u_g,"u_g");
		diagnostic_results->FindResult(&p_g,"p_g");
		delete diagnostic_results;

		//compute new thickness
		_printf_("%s\n","computing new thickness");
		
		inputs->Add("velocity",u_g,2,numberofnodes);
		prognostic_results=new DataSet(ResultsEnum()); prognostic_core(prognostic_results,fem_p, inputs);
	
		//extract h_g prognostic_results, and erase prognostic_results;
		prognostic_results->FindResult(&h_g_intermediary,"h_g");
		delete prognostic_results;

		//update surface and bed using the new thickness
		_printf_("   updating geometry\n");
		UpdateGeometryx(&h_g,&b_g,&s_g, 
				fem_p->elements, fem_p->nodes,fem_p->loads, fem_p->materials, 
				h_g_intermediary,b_g,s_g); 
		VecFree(&h_g_intermediary);

		//plug into results.
		result=new Result(results->Size()+1,time,step,"u_g",u_g); results->AddObject(result);
		result=new Result(results->Size()+1,time,step,"p_g",p_g); results->AddObject(result);
		result=new Result(results->Size()+1,time,step,"h_g",h_g); results->AddObject(result);
		result=new Result(results->Size()+1,time,step,"s_g",s_g); results->AddObject(result);
		result=new Result(results->Size()+1,time,step,"b_g",b_g); results->AddObject(result);
	}

	/*Free ressources:*/
	xfree((void**)&u_g_serial);
	xfree((void**)&p_g_serial);
	xfree((void**)&h_g_serial);
	xfree((void**)&s_g_serial);
	xfree((void**)&b_g_serial);
}
