function md=SetIceShelfBC(md,varargin)
%SETICESHELFBC - Create the boundary conditions for diagnostic and thermal models for a  Ice Shelf with Ice Front
%
%   Neumann BC are used on the ice front (an ANRGUS contour around the ice front
%   must be given in input)
%   Dirichlet BC are used elsewhere for diagnostic
%
%   Usage:
%      md=SetIceShelfBC(md,varargin)
%
%   Example:
%      md=SetIceShelfBC(md);
%      md=SetIceShelfBC(md,'Front.exp');
%
%   See also: SETICESHEETBC, SETMARINEICESHEETBC

%node on Dirichlet (boundary and ~icefront)
if nargin==2,
	icefrontfile=varargin{1};
	if ~exist(icefrontfile), error(['SetIceShelfBC error message: ice front file ' icefrontfile ' not found']); end
	nodeinsideicefront=ContourToMesh(md.elements,md.x,md.y,icefrontfile,'node',2);
	nodeonicefront=double(md.nodeonboundary & nodeinsideicefront);
elseif nargin==1,
	nodeonicefront=zeros(md.numberofnodes,1);
else
	help SetIceShelfBC
	error('bad usage');
end
pos=find(md.nodeonboundary & ~nodeonicefront);
md.spcvelocity=zeros(md.numberofnodes,6);
md.spcvelocity(pos,1:3)=1;
md.diagnostic_ref=NaN*ones(md.numberofnodes,6);

%Dirichlet Values
if (length(md.vx_obs)==md.numberofnodes & length(md.vy_obs)==md.numberofnodes)
	disp('      boundary conditions for diagnostic model: spc set as observed velocities');
	md.spcvelocity(pos,4:5)=[md.vx_obs(pos) md.vy_obs(pos)]; %zeros for vz
else
	disp('      boundary conditions for diagnostic model: spc set as zero');
end

%segment on Ice Front
%segment on Neumann (Ice Front)
pos=find(nodeonicefront(md.segments(:,1)) | nodeonicefront(md.segments(:,2)));
if (md.dim==2)
	pressureload=md.segments(pos,:);
elseif md.dim==3
	pressureload_layer1=[md.segments(pos,1:2)  md.segments(pos,2)+md.numberofnodes2d  md.segments(pos,1)+md.numberofnodes2d  md.segments(pos,3)];
	pressureload=[];
	for i=1:md.numlayers-1,
		pressureload=[pressureload ;pressureload_layer1(:,1:4)+(i-1)*md.numberofnodes2d pressureload_layer1(:,5)+(i-1)*md.numberofelements2d ];
	end
end

%Add water or air enum depending on the element
pressureload=[pressureload WaterEnum*md.elementoniceshelf(pressureload(:,end))];

%plug onto model
md.pressureload=pressureload;

%Create zeros basal_melting_rate, surface_ablation_rate, surface_accumulation_rate
% and surface_mass_balance if not specified
if isnan(md.surface_accumulation_rate),
	md.surface_accumulation_rate=zeros(md.numberofnodes,1);
	md.forcings.surface_accumulation_rate=zeros(md.numberofnodes+1,1);
	disp('      no surface_accumulation_rate specified: values set as zero');
end
if isnan(md.surface_ablation_rate),
	md.surface_ablation_rate=zeros(md.numberofnodes,1);
	md.forcings.surface_ablation_rate=zeros(md.numberofnodes+1,1);
	disp('      no surface_ablation_rate specified: values set as zero');
end
if isnan(md.surface_mass_balance),
	md.surface_mass_balance=zeros(md.numberofnodes,1);
	md.forcings.surface_mass_balance=zeros(md.numberofnodes+1,1);
	disp('      no surface_mass_balance specified: values set as zero');
end
if isnan(md.basal_melting_rate),
	md.basal_melting_rate=zeros(md.numberofnodes,1);
	disp('      no basal_melting_rate specified: values set as zero');
end
if isnan(md.dhdt),
	md.dhdt=zeros(md.numberofnodes,1);
	disp('      no dhdt specified: values set as zero');
end

md.spcthickness=zeros(md.numberofnodes,2);

if (length(md.observed_temperature)==md.numberofnodes),
	md.spctemperature=[md.nodeonsurface md.observed_temperature]; %impose observed temperature on surface
	if (length(md.geothermalflux)~=md.numberofnodes),
		md.geothermalflux=zeros(md.numberofnodes,1);
	end
else
	disp('      no thermal boundary conditions created: no observed temperature found');
end
