/*!\file PentaVertexForcing.c
 * \brief: implementation of the PentaVertexForcing object
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include <string.h>
#include "../objects.h"
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "../../shared/shared.h"
#include "../../Container/Container.h"
#include "../../include/include.h"

/*PentaVertexForcing constructors and destructor*/
/*FUNCTION PentaVertexForcing::PentaVertexForcing(){{{1*/
PentaVertexForcing::PentaVertexForcing(){
	enum_type=NoneEnum;
	values=NULL;
	numtimesteps=0;
	parameters=NULL;
	return;
}
/*}}}*/
/*FUNCTION PentaVertexForcing::PentaVertexForcing(int in_enum_type,double* time0values,double time0,int numtimesteps,Parameters* parameters){{{1*/
PentaVertexForcing::PentaVertexForcing(int in_enum_type,double* time0values,double time0,int in_numtimesteps,Parameters* parameters)
	:TriaRef(1)
{

	/*Set TriaRef*/
	this->SetElementType(P1Enum,0);
	this->element_type=P1Enum;

	/*Set Enum*/
	enum_type=in_enum_type;

	/*Allocate values and timesteps, and set first time step values: */
	this->numtimesteps=in_numtimesteps;
	this->values=(double*)xmalloc(this->numtimesteps*6*sizeof(double));
	this->timesteps=(double*)xmalloc(this->numtimesteps*sizeof(double));
	for(int i=0;i<6;i++) this->values[i]=time0values[i];
	this->timesteps[0]=time0;

	this->parameters=parameters;

}
/*}}}*/
/*FUNCTION PentaVertexForcing::PentaVertexForcing(int in_enum_type,double* in_values,double* in_time,int numtimesteps,Parameters* parameters){{{1*/
PentaVertexForcing::PentaVertexForcing(int in_enum_type,double* in_values,double* in_time,int in_numtimesteps,Parameters* parameters)
	:TriaRef(1)
{

	/*Set TriaRef*/
	this->SetElementType(P1Enum,0);
	this->element_type=P1Enum;

	/*Set Enum*/
	enum_type=in_enum_type;

	/*Allocate values and timesteps, and copy: */
	this->numtimesteps=in_numtimesteps;
	this->values=(double*)xmalloc(this->numtimesteps*6*sizeof(double));
	this->timesteps=(double*)xmalloc(this->numtimesteps*sizeof(double));
	
	memcpy(this->values,in_values,numtimesteps*6*sizeof(double));
	memcpy(this->timesteps,in_time,numtimesteps*sizeof(double));

	this->parameters=parameters;

}
/*}}}*/
/*FUNCTION PentaVertexForcing::~PentaVertexForcing(){{{1*/
PentaVertexForcing::~PentaVertexForcing(){
	xfree((void**)&this->values);
	xfree((void**)&this->timesteps);
	parameters=NULL;
	return;
}
/*}}}*/
/*FUNCTION PentaVertexForcing::AddTimeValues(double* values,int step,double time);{{{1*/
void PentaVertexForcing::AddTimeValues(double* values,int step,double time){

	/*insert values at time step: */
	if (step<0)_error_("timestep should start at least at time 0!");
	if (step>numtimesteps-1)_error_("timestep cannot be more than number of time steps");

	/*go ahead and plug: */
	this->timesteps[step]=time;
	for(int i=0;i<6;i++) this->values[6*step+i]=values[i];
}
/*}}}*/

/*Object virtual functions definitions:*/
/*FUNCTION PentaVertexForcing::Echo {{{1*/
void PentaVertexForcing::Echo(void){
	this->DeepEcho();
}
/*}}}*/
/*FUNCTION PentaVertexForcing::DeepEcho{{{1*/
void PentaVertexForcing::DeepEcho(void){

	int i;
	
	printf("PentaVertexForcing:\n");
	printf("   enum: %i (%s)\n",this->enum_type,EnumToStringx(this->enum_type));
	printf("   numtimesteps: %i\n",this->numtimesteps);
	for(i=0;i<this->numtimesteps;i++){
		printf("   time: %g  values: [%g %g %g %g %g %g]\n",this->timesteps[i],this->values[6*i+0],this->values[6*i+1],this->values[6*i+2],this->values[6*i+3],this->values[6*i+4],this->values[6*i+5]);
	}
}
/*}}}*/
/*FUNCTION PentaVertexForcing::Id{{{1*/
int    PentaVertexForcing::Id(void){ return -1; }
/*}}}*/
/*FUNCTION PentaVertexForcing::MyRank{{{1*/
int    PentaVertexForcing::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}
/*}}}*/
/*FUNCTION PentaVertexForcing::Marshall{{{1*/
void  PentaVertexForcing::Marshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   enum_value=0;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum value of PentaVertexForcing: */
	enum_value=PentaVertexForcingEnum;
	
	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_value,sizeof(enum_value));marshalled_dataset+=sizeof(enum_value);
	
	/*marshall PentaVertexForcing data: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);
	memcpy(marshalled_dataset,&numtimesteps,sizeof(numtimesteps));marshalled_dataset+=sizeof(numtimesteps);
	memcpy(marshalled_dataset,values,numtimesteps*6*sizeof(double));marshalled_dataset+=numtimesteps*6*sizeof(double);
	memcpy(marshalled_dataset,timesteps,numtimesteps*sizeof(double));marshalled_dataset+=numtimesteps*sizeof(double);

	*pmarshalled_dataset=marshalled_dataset;
}
/*}}}*/
/*FUNCTION PentaVertexForcing::MarshallSize{{{1*/
int   PentaVertexForcing::MarshallSize(){
	
	return 
		+sizeof(enum_type)+
		+sizeof(numtimesteps)+
		6*numtimesteps*sizeof(double)+
		numtimesteps*sizeof(double)+
		+sizeof(int); //sizeof(int) for enum value
}
/*}}}*/
/*FUNCTION PentaVertexForcing::Demarshall{{{1*/
void  PentaVertexForcing::Demarshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   i;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum type, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/
	memcpy(&enum_type,marshalled_dataset,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);
	memcpy(&numtimesteps,marshalled_dataset,sizeof(numtimesteps));marshalled_dataset+=sizeof(numtimesteps);
	/*allocate: */
	timesteps=(double*)xmalloc(numtimesteps*sizeof(double));
	values=(double*)xmalloc(numtimesteps*6*sizeof(double));
	
	memcpy(values,marshalled_dataset,6*numtimesteps*sizeof(double));marshalled_dataset+=6*numtimesteps*sizeof(double);
	memcpy(timesteps,marshalled_dataset,numtimesteps*sizeof(double));marshalled_dataset+=numtimesteps*sizeof(double);

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION PentaVertexForcing::Enum{{{1*/
int PentaVertexForcing::Enum(void){

	return PentaVertexForcingEnum;

}
/*}}}*/
/*FUNCTION PentaVertexForcing::copy{{{1*/
Object* PentaVertexForcing::copy() {
	
	return new PentaVertexForcing(this->enum_type,this->values,this->timesteps,this->numtimesteps,this->parameters);

}
/*}}}*/
	
/*PentaVertexForcing management*/
/*FUNCTION PentaVertexForcing::EnumType{{{1*/
int PentaVertexForcing::EnumType(void){

	return this->enum_type;

}
/*}}}*/
/*FUNCTION PentaVertexForcing::SpawnTriaInput{{{1*/
Input* PentaVertexForcing::SpawnTriaInput(int* indices){

	/*output*/
	TriaVertexForcing* outforcing=NULL;
	double* triavalues=NULL;
	int i,j;
	
	/*Build tria values*/
	triavalues=(double*)xmalloc(this->numtimesteps*3*sizeof(double));
	for(i=0;i<3;i++)for(j=0;j<this->numtimesteps;j++) triavalues[3*j+i]=this->values[6*j+indices[i]];

	/*Create new Tria forcing (copy of current forcing)*/
	outforcing=new TriaVertexForcing(this->enum_type,triavalues,this->timesteps,this->numtimesteps,this->parameters);

	/*Assign output*/
	return outforcing;
}
/*}}}*/
/*FUNCTION PentaVertexForcing::SpawnResult{{{1*/
ElementResult* PentaVertexForcing::SpawnResult(int step, double time){

	double pentavalues[6];

	/*Ok, we want to spawn a PentaVertexElementResult. We have the time, just get 
	 *the correct values at the three nodes: */
	this->GetTimeValues(&pentavalues[0],time);

	return new PentaVertexElementResult(this->enum_type,&pentavalues[0],step,time);

}
/*}}}*/

/*Object functions*/
/*FUNCTION PentaVertexForcing::GetParameterValue(double* pvalue,GaussTria* gauss){{{1*/
void PentaVertexForcing::GetParameterValue(double* pvalue,GaussTria* gauss){
	
	double time;
	double pentavalues[6];

	/*First, recover current time from parameters: */
	parameters->FindParam(&time,TimeEnum);

	/*Retrieve interpolated values for this time step: */
	this->GetTimeValues(&pentavalues[0],time);

	/*Call TriaRef function*/
	TriaRef::GetParameterValue(pvalue,&pentavalues[0],gauss);
}
/*}}}*/
/*FUNCTION PentaVertexForcing::GetParameterDerivativeValue(double* p, double* xyz_list, GaussTria* gauss){{{1*/
void PentaVertexForcing::GetParameterDerivativeValue(double* p, double* xyz_list, GaussTria* gauss){

	double time;
	double triavalues[3];

	/*First, recover current time from parameters: */
	parameters->FindParam(&time,TimeEnum);

	/*Retrieve interpolated values for this time step: */
	this->GetTimeValues(&triavalues[0],time);

	/*Call TriaRef function*/
	TriaRef::GetParameterDerivativeValue(p,&triavalues[0],xyz_list,gauss);
}
/*}}}*/
/*FUNCTION PentaVertexForcing::ChangeEnum{{{1*/
void PentaVertexForcing::ChangeEnum(int newenumtype){
	this->enum_type=newenumtype;
}
/*}}}*/
/*FUNCTION PentaVertexForcing::GetParameterAverage{{{1*/
void PentaVertexForcing::GetParameterAverage(double* pvalue){
	
	double time;
	double pentavalues[6];

	/*First, recover current time from parameters: */
	parameters->FindParam(&time,TimeEnum);

	/*Retrieve interpolated values for this time step: */
	this->GetTimeValues(&pentavalues[0],time);

	*pvalue=1./6.*(pentavalues[0]+pentavalues[1]+pentavalues[2]+pentavalues[3]+pentavalues[4]+pentavalues[5]);
}
/*}}}*/

/*Intermediary*/
/*FUNCTION PentaVertexForcing::SquareMin{{{1*/
void PentaVertexForcing::SquareMin(double* psquaremin, bool process_units,Parameters* parameters){

	int i;
	const int numnodes=6;
	double valuescopy[numnodes];
	double squaremin;
	double time;

	/*First, recover current time from parameters: */
	parameters->FindParam(&time,TimeEnum);

	/*Retrieve interpolated values for this time step: */
	this->GetTimeValues(&valuescopy[0],time);

	/*Process units if requested: */
	if(process_units)UnitConversion(&valuescopy[0],numnodes,IuToExtEnum,enum_type,parameters);

	/*Now, figure out minimum of valuescopy: */
	squaremin=pow(valuescopy[0],2);
	for(i=1;i<numnodes;i++){
		if(pow(valuescopy[i],2)<squaremin)squaremin=pow(valuescopy[i],2);
	}
	/*Assign output pointers:*/
	*psquaremin=squaremin;
}
/*}}}*/
/*FUNCTION PentaVertexForcing::InfinityNorm{{{1*/
double PentaVertexForcing::InfinityNorm(void){

	/*Output*/
	double norm=0;
	const int numnodes=6;
	double time;
	double pentavalues[6];

	/*First, recover current time from parameters: */
	parameters->FindParam(&time,TimeEnum);

	/*Retrieve interpolated values for this time step: */
	this->GetTimeValues(&pentavalues[0],time);

	for(int i=0;i<numnodes;i++) if(fabs(pentavalues[i])>norm) norm=fabs(pentavalues[i]);
	return norm;
}
/*}}}*/
/*FUNCTION PentaVertexForcing::Max{{{1*/
double PentaVertexForcing::Max(void){

	const int numnodes=6;
	double    max;
	double time;
	double pentavalues[6];

	/*First, recover current time from parameters: */
	parameters->FindParam(&time,TimeEnum);

	/*Retrieve interpolated values for this time step: */
	this->GetTimeValues(&pentavalues[0],time);
	
	max=pentavalues[0];

	for(int i=1;i<numnodes;i++){
		if(pentavalues[i]>max) max=pentavalues[i];
	}
	return max;
}
/*}}}*/
/*FUNCTION PentaVertexForcing::MaxAbs{{{1*/
double PentaVertexForcing::MaxAbs(void){

	const int numnodes=6;
	double    max;
	double time;
	double pentavalues[6];

	/*First, recover current time from parameters: */
	parameters->FindParam(&time,TimeEnum);

	/*Retrieve interpolated values for this time step: */
	this->GetTimeValues(&pentavalues[0],time);
	
	
	max=fabs(pentavalues[0]);

	for(int i=1;i<numnodes;i++){
		if(fabs(pentavalues[i])>max) max=fabs(pentavalues[i]);
	}
	return max;
}
/*}}}*/
/*FUNCTION PentaVertexForcing::Min{{{1*/
double PentaVertexForcing::Min(void){

	const int numnodes=6;
	double    min;
	double time;
	double pentavalues[6];

	/*First, recover current time from parameters: */
	parameters->FindParam(&time,TimeEnum);

	/*Retrieve interpolated values for this time step: */
	this->GetTimeValues(&pentavalues[0],time);
	
	min=pentavalues[0];

	for(int i=1;i<numnodes;i++){
		if(pentavalues[i]<min) min=pentavalues[i];
	}
	return min;
}
/*}}}*/
/*FUNCTION PentaVertexForcing::MinAbs{{{1*/
double PentaVertexForcing::MinAbs(void){

	const int numnodes=6;
	double    min;
	double time;
	double pentavalues[6];

	/*First, recover current time from parameters: */
	parameters->FindParam(&time,TimeEnum);

	/*Retrieve interpolated values for this time step: */
	this->GetTimeValues(&pentavalues[0],time);
	
	min=fabs(pentavalues[0]);

	for(int i=1;i<numnodes;i++){
		if(fabs(pentavalues[i])<min) min=fabs(pentavalues[i]);
	}
	return min;
}
/*}}}*/
/*FUNCTION PentaVertexForcing::GetVectorFromInputs{{{1*/
void PentaVertexForcing::GetVectorFromInputs(Vec vector,int* doflist){

	const int numvertices=6;
	double time;
	double pentavalues[6];

	/*First, recover current time from parameters: */
	parameters->FindParam(&time,TimeEnum);

	/*Retrieve interpolated values for this time step: */
	this->GetTimeValues(&pentavalues[0],time);

	VecSetValues(vector,numvertices,doflist,(const double*)&pentavalues[0],INSERT_VALUES);

} /*}}}*/
/*FUNCTION PentaVertexForcing::GetTimeValues{{{1*/
void PentaVertexForcing::GetTimeValues(double* outvalues,double intime){

	int    i,j;
	double deltat;
	double alpha;
	bool   found=false;

	/*Ok, we have the time, go through the timesteps, and figure out which interval we 
	 *fall within. Then interpolate the values on this interval: */
	if(intime<this->timesteps[0]){
		/*get values for the first time: */
		for(i=0;i<6;i++) outvalues[i]=this->values[6*0+i];
		found=true;
	}
	else if(intime>this->timesteps[this->numtimesteps-1]){
		/*get values for the last time: */
		for(i=0;i<6;i++) outvalues[i]=this->values[6*(this->numtimesteps-1)+i];
		found=true;
	}
	else{
		/*Find which interval we fall within: */
		for(i=0;i<this->numtimesteps;i++){
			if(intime==this->timesteps[i]){
				/*We are right on one step time: */
				for(j=0;j<6;j++) outvalues[j]=this->values[6*i+j];
				found=true;
				break; //we are done with the time interpolation.
			}
			else{
				if(this->timesteps[i]<intime && intime<this->timesteps[i+1]){
					/*ok, we have the interval ]i:i+1[. Interpolate linearly for now: */
					deltat=this->timesteps[i+1]-this->timesteps[i];
					alpha=(intime-this->timesteps[i])/deltat;
					for(j=0;j<6;j++) outvalues[j]=this->values[6*i+j]+alpha*(this->values[6*(i+1)+j]-this->values[6*i+j]);
					found=true;
					break;
				}
				else continue; //keep looking on the next interval
			}
		}
	}
	if(!found)_error_("did not find time interval on which to interpolate forcing values!");
	
}
/*}}}*/
/*FUNCTION PentaVertexForcing::Configure{{{1*/
void PentaVertexForcing::Configure(Parameters* parameters){
	printf("Configuring %p\n",parameters);
	this->parameters=parameters;
}
/*}}}*/
