/*!\file: CreateDataSets
 * \brief general driver for creating all datasets that make a finite element iomodel
 */ 

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "../../objects/objects.h"
#include "../../shared/shared.h"
#include "../../include/include.h"
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "./ModelProcessorx.h"


void CreateDataSets(Elements** pelements,Nodes** pnodes, Vertices** pvertices, Materials** pmaterials, Constraints** pconstraints, Loads** ploads,Parameters** pparameters,IoModel* iomodel,FILE* iomodel_handle,const int solution_type,const int analysis_type,const int nummodels,int analysis_counter){

	bool       continuous = true;
	Elements  *elements   = NULL;
	Materials *materials  = NULL;
	Parameters *parameters  = NULL;
			
	/*Create elements, vertices and materials, independent of analysis_type: */
	CreateElementsVerticesAndMaterials(pelements, pvertices, pmaterials, iomodel,iomodel_handle,nummodels);

	/*Recover elements and materials, for future update: */
	elements=*pelements;
	materials=*pmaterials;

	/*Now, branch onto analysis dependent model generation: */
	switch(analysis_type){
		case  DiagnosticHorizAnalysisEnum:
			CreateNodesDiagnosticHoriz(pnodes, iomodel,iomodel_handle);
			CreateConstraintsDiagnosticHoriz(pconstraints,iomodel,iomodel_handle);
			CreateLoadsDiagnosticHoriz(ploads,iomodel,iomodel_handle);
			UpdateElementsDiagnosticHoriz(elements,iomodel,iomodel_handle,analysis_counter,analysis_type);
			break;
		
		case  DiagnosticVertAnalysisEnum:
			CreateNodesDiagnosticVert(pnodes, iomodel,iomodel_handle);
			CreateConstraintsDiagnosticVert(pconstraints,iomodel,iomodel_handle);
			CreateLoadsDiagnosticVert(ploads,iomodel,iomodel_handle);
			UpdateElementsDiagnosticVert(elements,iomodel,iomodel_handle,analysis_counter,analysis_type);
			break;
	
		case DiagnosticHutterAnalysisEnum:
			CreateNodesDiagnosticHutter(pnodes, iomodel,iomodel_handle);
			CreateConstraintsDiagnosticHutter(pconstraints,iomodel,iomodel_handle);
			CreateLoadsDiagnosticHutter(ploads,iomodel,iomodel_handle);
			UpdateElementsDiagnosticHutter(elements,iomodel,iomodel_handle,analysis_counter,analysis_type);
			break;

		case BedSlopeAnalysisEnum:
			CreateNodesBedSlope(pnodes, iomodel,iomodel_handle);
			CreateConstraintsBedSlope(pconstraints,iomodel,iomodel_handle);
			CreateLoadsBedSlope(ploads,iomodel,iomodel_handle);
			UpdateElementsBedSlope(elements,iomodel,iomodel_handle,analysis_counter,analysis_type);
			break;

		case SurfaceSlopeAnalysisEnum:
			CreateNodesSurfaceSlope(pnodes, iomodel,iomodel_handle);
			CreateConstraintsSurfaceSlope(pconstraints,iomodel,iomodel_handle);
			CreateLoadsSurfaceSlope(ploads,iomodel,iomodel_handle);
			UpdateElementsSurfaceSlope(elements,iomodel,iomodel_handle,analysis_counter,analysis_type);
			break;

		case ThermalAnalysisEnum:
			CreateNodesThermal(pnodes, iomodel,iomodel_handle);
			CreateConstraintsThermal(pconstraints,iomodel,iomodel_handle);
			CreateLoadsThermal(ploads,iomodel,iomodel_handle);
			UpdateElementsThermal(elements,iomodel,iomodel_handle,analysis_counter,analysis_type);
			break;
		
		case HydrologyAnalysisEnum:
			CreateNodesHydrology(pnodes, iomodel,iomodel_handle);
			CreateConstraintsHydrology(pconstraints,iomodel,iomodel_handle);
			CreateLoadsHydrology(ploads,iomodel,iomodel_handle);
			UpdateElementsHydrology(elements,iomodel,iomodel_handle,analysis_counter,analysis_type);
			break;

		case MeltingAnalysisEnum:
			CreateNodesMelting(pnodes, iomodel,iomodel_handle);
			CreateConstraintsMelting(pconstraints,iomodel,iomodel_handle);
			CreateLoadsMelting(ploads,iomodel,iomodel_handle);
			UpdateElementsMelting(elements,iomodel,iomodel_handle,analysis_counter,analysis_type);
			break;

		case PrognosticAnalysisEnum:
			CreateNodesPrognostic(pnodes, iomodel,iomodel_handle);
			CreateConstraintsPrognostic(pconstraints,iomodel,iomodel_handle);
			CreateLoadsPrognostic(ploads,iomodel,iomodel_handle);
			UpdateElementsPrognostic(elements,iomodel,iomodel_handle,analysis_counter,analysis_type);
			break;

		case BalancethicknessAnalysisEnum:
			CreateNodesBalancethickness(pnodes, iomodel,iomodel_handle);
			CreateConstraintsBalancethickness(pconstraints,iomodel,iomodel_handle);
			CreateLoadsBalancethickness(ploads,iomodel,iomodel_handle);
			UpdateElementsBalancethickness(elements,iomodel,iomodel_handle,analysis_counter,analysis_type);
			break;

		case BalancevelocitiesAnalysisEnum:
			CreateNodesBalancevelocities(pnodes, iomodel,iomodel_handle);
			CreateConstraintsBalancevelocities(pconstraints,iomodel,iomodel_handle);
			CreateLoadsBalancevelocities(ploads,iomodel,iomodel_handle);
			UpdateElementsBalancevelocities(elements,iomodel,iomodel_handle,analysis_counter,analysis_type);
			break;
		default:
			_error_("%s%s%s"," analysis_type: ",EnumToStringx(analysis_type)," not supported yet!");
	}

	/*Update Elements and Materials For Control methods*/
	UpdateElementsAndMaterialsControl(elements,materials,iomodel,iomodel_handle);

	/*Generate objects that are not dependent on any analysis_type: */
	CreateParameters(pparameters,iomodel,iomodel_handle,solution_type,analysis_type,analysis_counter);

	/*Update Elements in case we are running a transient solution: */
	parameters=*pparameters;
	if(analysis_counter==(nummodels-1)&& (solution_type==Transient2DSolutionEnum || solution_type==Transient3DSolutionEnum)){
		UpdateElementsTransient(elements,parameters,iomodel,iomodel_handle,analysis_counter,analysis_type);
	}

	/*Sort datasets: */
	SortDataSets(pelements,pnodes,pvertices, ploads, pmaterials, pconstraints, pparameters);

	/*Update counters, because we have created more nodes, loads and constraints, and ids for objects created in next call to CreateDataSets
	 * will need to start at the end of the updated counters: */
	UpdateCounters(iomodel,pnodes,ploads,pconstraints);

}
