/*!\file TriaVertexInput.c
 * \brief: implementation of the TriaVertexInput object
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include <string.h>
#include "../objects.h"
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "../../shared/shared.h"
#include "../../Container/Container.h"
#include "../../include/include.h"

/*TriaVertexInput constructors and destructor*/
/*FUNCTION TriaVertexInput::TriaVertexInput(){{{1*/
TriaVertexInput::TriaVertexInput(){
	return;
}
/*}}}*/
/*FUNCTION TriaVertexInput::TriaVertexInput(int in_enum_type,double* values){{{1*/
TriaVertexInput::TriaVertexInput(int in_enum_type,double* in_values)
	:TriaRef(1)
{

	/*Set TriaRef*/
	this->SetElementType(P1Enum,0);
	this->element_type=P1Enum;

	/*Set Enum*/
	enum_type=in_enum_type;

	/*Set values*/
	values[0]=in_values[0];
	values[1]=in_values[1];
	values[2]=in_values[2];
}
/*}}}*/
/*FUNCTION TriaVertexInput::~TriaVertexInput(){{{1*/
TriaVertexInput::~TriaVertexInput(){
	return;
}
/*}}}*/

/*Object virtual functions definitions:*/
/*FUNCTION TriaVertexInput::Echo {{{1*/
void TriaVertexInput::Echo(void){
	this->DeepEcho();
}
/*}}}*/
/*FUNCTION TriaVertexInput::DeepEcho{{{1*/
void TriaVertexInput::DeepEcho(void){

	printf("TriaVertexInput:\n");
	printf("   enum: %i (%s)\n",this->enum_type,EnumToStringx(this->enum_type));
	printf("   values: [%g %g %g]\n",this->values[0],this->values[1],this->values[2]);
}
/*}}}*/
/*FUNCTION TriaVertexInput::Id{{{1*/
int    TriaVertexInput::Id(void){ return -1; }
/*}}}*/
/*FUNCTION TriaVertexInput::MyRank{{{1*/
int    TriaVertexInput::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}
/*}}}*/
/*FUNCTION TriaVertexInput::Marshall{{{1*/
void  TriaVertexInput::Marshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   enum_value=0;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum value of TriaVertexInput: */
	enum_value=TriaVertexInputEnum;
	
	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_value,sizeof(enum_value));marshalled_dataset+=sizeof(enum_value);
	
	/*marshall TriaVertexInput data: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);
	memcpy(marshalled_dataset,&values,sizeof(values));marshalled_dataset+=sizeof(values);

	*pmarshalled_dataset=marshalled_dataset;
}
/*}}}*/
/*FUNCTION TriaVertexInput::MarshallSize{{{1*/
int   TriaVertexInput::MarshallSize(){
	
	return sizeof(values)+
		+sizeof(enum_type)+
		+sizeof(int); //sizeof(int) for enum value
}
/*}}}*/
/*FUNCTION TriaVertexInput::Demarshall{{{1*/
void  TriaVertexInput::Demarshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   i;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum type, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/
	memcpy(&enum_type,marshalled_dataset,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);
	memcpy(&values,marshalled_dataset,sizeof(values));marshalled_dataset+=sizeof(values);

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION TriaVertexInput::Enum{{{1*/
int TriaVertexInput::Enum(void){

	return TriaVertexInputEnum;

}
/*}}}*/
/*FUNCTION TriaVertexInput::copy{{{1*/
Object* TriaVertexInput::copy() {
	
	return new TriaVertexInput(this->enum_type,this->values);

}
/*}}}*/
	
/*TriaVertexInput management*/
/*FUNCTION TriaVertexInput::EnumType{{{1*/
int TriaVertexInput::EnumType(void){

	return this->enum_type;

}
/*}}}*/
/*FUNCTION TriaVertexInput::SpawnTriaInput{{{1*/
Input* TriaVertexInput::SpawnTriaInput(int* indices){

	/*output*/
	TriaVertexInput* outinput=NULL;

	/*Create new Tria input (copy of current input)*/
	outinput=new TriaVertexInput(this->enum_type,&this->values[0]);

	/*Assign output*/
	return outinput;

}
/*}}}*/
/*FUNCTION TriaVertexInput::SpawnResult{{{1*/
ElementResult* TriaVertexInput::SpawnResult(int step, double time){

	return new TriaVertexElementResult(this->enum_type,this->values,step,time);

}
/*}}}*/

/*Object functions*/
/*FUNCTION TriaVertexInput::GetParameterValue(double* pvalue,GaussTria* gauss){{{1*/
void TriaVertexInput::GetParameterValue(double* pvalue,GaussTria* gauss){

	/*Call TriaRef function*/
	TriaRef::GetParameterValue(pvalue,&values[0],gauss);

}
/*}}}*/
/*FUNCTION TriaVertexInput::GetParameterDerivativeValue(double* p, double* xyz_list, GaussTria* gauss){{{1*/
void TriaVertexInput::GetParameterDerivativeValue(double* p, double* xyz_list, GaussTria* gauss){

	/*Call TriaRef function*/
	TriaRef::GetParameterDerivativeValue(p,&values[0],xyz_list,gauss);
}
/*}}}*/
/*FUNCTION TriaVertexInput::GetVxStrainRate2d{{{1*/
void TriaVertexInput::GetVxStrainRate2d(double* epsilonvx,double* xyz_list, GaussTria* gauss){

	/*Intermediary*/
	int       i;
	const int numnodes=3;
	double B[3][NDOF2*numnodes];
	double velocity[3][NDOF2];

	/*Get B matrix: */
	GetBMacAyeal(&B[0][0], xyz_list, gauss);

	/*Here, we are computing the strain rate of (vx,0)*/
	for(i=0;i<3;i++){
		velocity[i][0]=this->values[i];
		velocity[i][1]=0.0;
	}
	/*Get epsilon(vx) = B*velocity*/
	MatrixMultiply( &B[0][0],3,NDOF2*numnodes,0,
				&velocity[0][0],NDOF2*numnodes,1,0,
				epsilonvx,0);
}
/*}}}*/
/*FUNCTION TriaVertexInput::GetVyStrainRate2d{{{1*/
void TriaVertexInput::GetVyStrainRate2d(double* epsilonvy,double* xyz_list, GaussTria* gauss){

	/*Intermediary*/
	int       i;
	const int numnodes=3;
	double B[3][NDOF2*numnodes];
	double velocity[3][NDOF2];

	/*Get B matrix: */
	GetBMacAyeal(&B[0][0], xyz_list, gauss);

	/*Here, we are computing the strain rate of (0,vy)*/
	for(i=0;i<3;i++){
		velocity[i][0]=0.0;
		velocity[i][1]=this->values[i];
	}
	/*Get epsilon(vy) = B*velocity*/
	MatrixMultiply( &B[0][0],3,NDOF2*numnodes,0,
				&velocity[0][0],NDOF2*numnodes,1,0,
				epsilonvy,0);
}
/*}}}*/
/*FUNCTION TriaVertexInput::ChangeEnum{{{1*/
void TriaVertexInput::ChangeEnum(int newenumtype){
	this->enum_type=newenumtype;
}
/*}}}*/
/*FUNCTION TriaVertexInput::GetParameterAverage{{{1*/
void TriaVertexInput::GetParameterAverage(double* pvalue){
	*pvalue=1./3.*(values[0]+values[1]+values[2]);
}
/*}}}*/

/*Intermediary*/
/*FUNCTION TriaVertexInput::SquareMin{{{1*/
void TriaVertexInput::SquareMin(double* psquaremin, bool process_units,Parameters* parameters){

	int i;
	const int numnodes=3;
	double valuescopy[numnodes];
	double squaremin;

	/*First,  copy values, to process units if requested: */
	for(i=0;i<numnodes;i++)valuescopy[i]=this->values[i];

	/*Process units if requested: */
	if(process_units)UnitConversion(&valuescopy[0],numnodes,IuToExtEnum,enum_type,parameters);

	/*Now, figure out minimum of valuescopy: */
	squaremin=pow(valuescopy[0],2);
	for(i=1;i<numnodes;i++){
		if(pow(valuescopy[i],2)<squaremin)squaremin=pow(valuescopy[i],2);
	}
	/*Assign output pointers:*/
	*psquaremin=squaremin;
}
/*}}}*/
/*FUNCTION TriaVertexInput::ContrainMin{{{1*/
void TriaVertexInput::ConstrainMin(double minimum){
	
	int i;
	const int numnodes=3;

	for(i=0;i<numnodes;i++) if (values[i]<minimum) values[i]=minimum;
}
/*}}}*/
/*FUNCTION TriaVertexInput::InfinityNorm{{{1*/
double TriaVertexInput::InfinityNorm(void){

	/*Output*/
	double norm=0;
	const int numnodes=3;

	for(int i=0;i<numnodes;i++) if(fabs(values[i])>norm) norm=fabs(values[i]);
	return norm;
}
/*}}}*/
/*FUNCTION TriaVertexInput::Max{{{1*/
double TriaVertexInput::Max(void){

	const int numnodes=3;
	double    max=values[0];

	for(int i=1;i<numnodes;i++){
		if(values[i]>max) max=values[i];
	}
	return max;
}
/*}}}*/
/*FUNCTION TriaVertexInput::MaxAbs{{{1*/
double TriaVertexInput::MaxAbs(void){

	const int numnodes=3;
	double    max=fabs(values[0]);

	for(int i=1;i<numnodes;i++){
		if(fabs(values[i])>max) max=fabs(values[i]);
	}
	return max;
}
/*}}}*/
/*FUNCTION TriaVertexInput::Min{{{1*/
double TriaVertexInput::Min(void){

	const int numnodes=3;
	double    min=values[0];

	for(int i=1;i<numnodes;i++){
		if(values[i]<min) min=values[i];
	}
	return min;
}
/*}}}*/
/*FUNCTION TriaVertexInput::MinAbs{{{1*/
double TriaVertexInput::MinAbs(void){

	const int numnodes=3;
	double    min=fabs(values[0]);

	for(int i=1;i<numnodes;i++){
		if(fabs(values[i])<min) min=fabs(values[i]);
	}
	return min;
}
/*}}}*/
/*FUNCTION TriaVertexInput::Scale{{{1*/
void TriaVertexInput::Scale(double scale_factor){
	
	int i;
	const int numnodes=3;

	for(i=0;i<numnodes;i++)values[i]=values[i]*scale_factor;
}
/*}}}*/
/*FUNCTION TriaVertexInput::ArtificialNoise{{{1*/
void TriaVertexInput::ArtificialNoise(double min,double max){

	int i;
	const int numnodes=3;
	double noise;

	/*Compute random number between bounds:
	 * rand() outputs an integer in [0 RAND_MAX]
	 * (double)rand()/RAND_MAX is in [0 1]
	 */
	 noise=min+(max-min)*(double)rand()/RAND_MAX;

	for(i=0;i<numnodes;i++)values[i]=values[i]+noise;
}
/*}}}*/
/*FUNCTION TriaVertexInput::AXPY{{{1*/
void TriaVertexInput::AXPY(Input* xinput,double scalar){

	int i;
	const int numnodes=3;
	TriaVertexInput*  xtriavertexinput=NULL;

	/*xinput is of the same type, so cast it: */
	xtriavertexinput=(TriaVertexInput*)xinput;

	/*Carry out the AXPY operation depending on type:*/
	switch(xinput->Enum()){

		case TriaVertexInputEnum :
			for(i=0;i<numnodes;i++)this->values[i]=this->values[i]+scalar*xtriavertexinput->values[i];
			return;

		default :
			_error_("not implemented yet");
	}

}
/*}}}*/
/*FUNCTION TriaVertexInput::Constrain{{{1*/
void TriaVertexInput::Constrain(double cm_min, double cm_max){

	int i;
	const int numnodes=3;
		
	if(!isnan(cm_min)) for(i=0;i<numnodes;i++)if (this->values[i]<cm_min)this->values[i]=cm_min;
	if(!isnan(cm_max)) for(i=0;i<numnodes;i++)if (this->values[i]>cm_max)this->values[i]=cm_max;

}
/*}}}*/
/*FUNCTION TriaVertexInput::GetVectorFromInputs{{{1*/
void TriaVertexInput::GetVectorFromInputs(Vec vector,int* doflist){

	const int numvertices=3;
	VecSetValues(vector,numvertices,doflist,(const double*)this->values,INSERT_VALUES);

} /*}}}*/
/*FUNCTION TriaVertexInput::GetValuesPtr{{{1*/
void TriaVertexInput::GetValuesPtr(double** pvalues,int* pnum_values){

	*pvalues=this->values;
	if(pnum_values)*pnum_values=3;

}
/*}}}*/
/*FUNCTION TriaVertexInput::PointwiseMin{{{1*/
Input* TriaVertexInput::PointwiseMin(Input* inputB){

	/*Ouput*/
	TriaVertexInput* outinput=NULL;

	/*Intermediaries*/
	int               i;
	TriaVertexInput *xinputB     = NULL;
	int               B_numvalues;
	const int         numnodes    = 3;
	double            minvalues[numnodes];

	/*Check that inputB is of the same type*/
	if (inputB->Enum()!=TriaVertexInputEnum) _error_("Operation not permitted because inputB is of type %s",EnumToStringx(inputB->Enum()));
	xinputB=(TriaVertexInput*)inputB;

	/*Create point wise min*/
	for(i=0;i<numnodes;i++){
		if(this->values[i] > xinputB->values[i]) minvalues[i]=xinputB->values[i];
		else minvalues[i]=this->values[i];
	}

	/*Create new Tria vertex input (copy of current input)*/
	outinput=new TriaVertexInput(this->enum_type,&minvalues[0]);

	/*Return output pointer*/
	return outinput;

}
/*}}}*/
/*FUNCTION TriaVertexInput::PointwiseMax{{{1*/
Input* TriaVertexInput::PointwiseMax(Input* inputB){

	/*Ouput*/
	TriaVertexInput* outinput=NULL;

	/*Intermediaries*/
	int               i;
	TriaVertexInput *xinputB     = NULL;
	int               B_numvalues;
	const int         numnodes    = 3;
	double            maxvalues[numnodes];

	/*Check that inputB is of the same type*/
	if (inputB->Enum()!=TriaVertexInputEnum) _error_("Operation not permitted because inputB is of type %s",EnumToStringx(inputB->Enum()));
	xinputB=(TriaVertexInput*)inputB;

	/*Create point wise max*/
	for(i=0;i<numnodes;i++){
		if(this->values[i] < xinputB->values[i]) maxvalues[i]=xinputB->values[i];
		else maxvalues[i]=this->values[i];
	}

	/*Create new Tria vertex input (copy of current input)*/
	outinput=new TriaVertexInput(this->enum_type,&maxvalues[0]);

	/*Return output pointer*/
	return outinput;

}
/*}}}*/
/*FUNCTION TriaVertexInput::Configure{{{1*/
void TriaVertexInput::Configure(Parameters* parameters){
	/*do nothing: */
}
/*}}}*/
