function femmodel=transient3d_core(femmodel)
%TRANSIENT3D_CORE - core of transient 2d solution
%
%   Usage:
%      femmodel=transient3d_core(femmodel)

	%recover parameters common to all solutions
	ndt=femmodel.parameters.Ndt;
	dt=femmodel.parameters.Dt;
	yts=femmodel.parameters.Yts;
	control_analysis=femmodel.parameters.ControlAnalysis;
	solution_type=femmodel.parameters.SolutionType;
	output_frequency=femmodel.parameters.OutputFrequency;
	time_adapt=femmodel.parameters.TimeAdapt;
	isdiagnostic=femmodel.parameters.IsDiagnostic;
	isprognostic=femmodel.parameters.IsPrognostic;
	isthermal=femmodel.parameters.IsThermal;

	%Initialize
	time=0;
	step=0;

	%Loop through time
	while (time < ndt - (yts*eps)),

		%Increment
		if(time_adapt),
			dt=TimeAdapt(femmodel.elements,femmodel.nodes,femmodel.vertices,femmodel.loads,femmodel.materials,femmodel.parameters);
			if(time+dt>ndt), dt=ndt-time; end
			femmodel.parameters.Dt=dt;
		end
		step=step+1;
		time=time+dt;

		issmprintf(VerboseSolution,'\n%s%g%s%i%s%g\n','time [yr] ',time,' iteration number: ',step,'/',floor(ndt/dt));

		if (isthermal)
			issmprintf(VerboseSolution,'\n%s',['   computing temperature']);
			femmodel=thermal_core_step(femmodel); 
		end

		if (isdiagnostic)
			issmprintf(VerboseSolution,'\n%s',['   computing new velocities']);
			femmodel=diagnostic_core(femmodel); 
		end

		if (isprognostic)
			issmprintf(VerboseSolution,'\n%s',['   computing new thickness']);
			femmodel=prognostic_core(femmodel); 
			issmprintf(VerboseSolution,'\n%s',['   updating vertices position']);
			[femmodel.elements,femmodel.nodes,femmodel.vertices,femmodel.loads,femmodel.materials,femmodel.parameters]=...
				UpdateVertexPositions(femmodel.elements,femmodel.nodes,femmodel.vertices,femmodel.loads,femmodel.materials,femmodel.parameters);
		end

		if (solution_type==Transient3DSolutionEnum & ~control_analysis & (mod(step,output_frequency)==0 | time==ndt)),
			issmprintf(VerboseSolution,'\n%s',['   saving results']);
			femmodel.elements=InputToResult(femmodel.elements,femmodel.nodes,femmodel.vertices,femmodel.loads,femmodel.materials,femmodel.parameters,VxEnum,step,time);
			femmodel.elements=InputToResult(femmodel.elements,femmodel.nodes,femmodel.vertices,femmodel.loads,femmodel.materials,femmodel.parameters,VyEnum,step,time);
			femmodel.elements=InputToResult(femmodel.elements,femmodel.nodes,femmodel.vertices,femmodel.loads,femmodel.materials,femmodel.parameters,VzEnum,step,time);
			femmodel.elements=InputToResult(femmodel.elements,femmodel.nodes,femmodel.vertices,femmodel.loads,femmodel.materials,femmodel.parameters,VelEnum,step,time);
			femmodel.elements=InputToResult(femmodel.elements,femmodel.nodes,femmodel.vertices,femmodel.loads,femmodel.materials,femmodel.parameters,PressureEnum,step,time);
			femmodel.elements=InputToResult(femmodel.elements,femmodel.nodes,femmodel.vertices,femmodel.loads,femmodel.materials,femmodel.parameters,ThicknessEnum,step,time);
			femmodel.elements=InputToResult(femmodel.elements,femmodel.nodes,femmodel.vertices,femmodel.loads,femmodel.materials,femmodel.parameters,SurfaceEnum,step,time);
			femmodel.elements=InputToResult(femmodel.elements,femmodel.nodes,femmodel.vertices,femmodel.loads,femmodel.materials,femmodel.parameters,BedEnum,step,time);
			femmodel.elements=InputToResult(femmodel.elements,femmodel.nodes,femmodel.vertices,femmodel.loads,femmodel.materials,femmodel.parameters,TemperatureEnum,step,time);
			femmodel.elements=InputToResult(femmodel.elements,femmodel.nodes,femmodel.vertices,femmodel.loads,femmodel.materials,femmodel.parameters,MeltingRateEnum,step,time);
		end
	end
end %end of function
