function md=YamsCall(md,field,hmin,hmax,gradation,epsilon),
%YAMSCALL - call yams
%
%   build a metric using the Hessian of the given field
%   call Yams and the output mesh is plugged onto the model
%   -hmin = minimum edge length (m)
%   -hmax = maximum edge length (m)
%   -gradation = maximum edge length gradation between 2 elements
%   -epsilon = average error on each element (m/yr)
%
%   Usage:
%      md=YamsCall(md,field,hmin,hmax,gradation,epsilon);
%
%   Example:
%      md=YamsCall(md,md.vel_obs,1500,10^8,1.3,0.9);

%2d geometric parameter (do not change)
scale=2/9; 

%Compute Hessian
t1=clock; fprintf('%s','      computing Hessian...');
hessian=ComputeHessian(md.elements,md.x,md.y,field,'node');
t2=clock;fprintf('%s\n',[' done (' num2str(etime(t2,t1)) ' seconds)']);

%Compute metric
t1=clock; fprintf('%s','      computing metric...');
if length(md.nodeonwater)==md.numberofnodes,
	pos=find(md.nodeonwater);
else
	pos=[];
end
metric=ComputeMetric(hessian,scale,epsilon,hmin,hmax,pos);
t2=clock;fprintf('%s\n',[' done (' num2str(etime(t2,t1)) ' seconds)']);

%write files
t1=clock; fprintf('%s','      writing initial mesh files...');
save -ascii carre0.met  metric

fid=fopen('carre0.mesh','w');

%initialiation
fprintf(fid,'\n%s\n%i\n','MeshVersionFormatted',1);

%dimension
fprintf(fid,'\n%s\n%i\n','Dimension',2);

%Vertices
fprintf(fid,'\n%s\n%i\n\n','Vertices',md.numberofnodes);
fprintf(fid,'%8g %8g %i\n',[md.x md.y zeros(md.numberofnodes,1)]');

%Triangles
fprintf(fid,'\n\n%s\n%i\n\n','Triangles',md.numberofelements);
fprintf(fid,'%i %i %i %i\n',[md.elements zeros(md.numberofelements,1)]');
numberofelements1=md.numberofelements;
	
%Deal with rifts
if md.numrifts,
	
	%we have the list of triangles that make up the rift. keep those triangles around during refinement.
	triangles=[];
	for i=1:md.numrifts,
		triangles=[triangles md.rifts(i).segments(:,3)'];
	end

	fprintf(fid,'\n\n%s\n%i\n\n','RequiredTriangles',length(triangles));
	fprintf(fid,'%i\n',triangles);
end

%close
fclose(fid);
t2=clock;fprintf('%s\n',[' done (' num2str(etime(t2,t1)) ' seconds)']);

%call yams
fprintf('%s\n','      call Yams...');
if ispc
	%windows
	system(['yams2-win -O 1 -v -0 -ecp -hgrad ' num2str(gradation)  ' carre0 carre1']);
elseif ismac
	%Macosx
	system(['yams2-osx -O 1 -v -0 -ecp -hgrad ' num2str(gradation)  ' carre0 carre1']);
else
	%Linux
	system(['yams2-linux -O 1 -v -0 -ecp -hgrad ' num2str(gradation)  ' carre0 carre1']);
end

%plug new mesh
t1=clock; fprintf('\n%s','      reading final mesh files...');
Tria=load('carre1.tria');
Coor=load('carre1.coor');
md.x=Coor(:,1);
md.y=Coor(:,2);
md.z=zeros(size(Coor,1),1);
md.elements=Tria;
md.numberofnodes=size(Coor,1);
md.numberofelements=size(Tria,1);
numberofelements2=md.numberofelements;
t2=clock;fprintf('%s\n\n',[' done (' num2str(etime(t2,t1)) ' seconds)']);

%display number of elements
fprintf('\n%s %i','      inital number of elements:',numberofelements1);
fprintf('\n%s %i\n\n','      new    number of elements:',numberofelements2);

%clean up:
system('rm carre0.mesh carre0.met carre1.tria carre1.coor carre1.meshb');
