function md=SetMarineIceSheetBC(md,varargin)
%SETICEMARINESHEETBC - Create the boundary conditions for diagnostic and thermal models for a  Marine Ice Sheet with Ice Front
%
%   Neumann BC are used on the ice front (an ARGUS contour around the ice front
%   can be given in input, or it will be deduced as oniceshelf & onboundary)
%   Dirichlet BC are used elsewhere for diagnostic
%
%   Usage:
%      md=SetMarineIceSheetBC(md,icefrontfile)
%      md=SetMarineIceSheetBC(md)
%
%   Example:
%      md=SetMarineIceSheetBC(md,'Front.exp')
%      md=SetMarineIceSheetBC(md)
%
%   See also: SETICESHELFBC, SETMARINEICESHEETBC

%node on Dirichlet (boundary and ~icefront)
if nargin==2,
	%User provided Front.exp, use it
	icefrontfile=varargin{1};
	if ~exist(icefrontfile)
		error(['SetMarineIceSheetBC error message: ice front file ' icefrontfile ' not found']);
	end
	nodeinsideicefront=ContourToMesh(md.elements,md.x,md.y,icefrontfile,'node',2);
	nodeonicefront=double(md.nodeonboundary & nodeinsideicefront);
else
	%Guess where the ice front is
	nodeoniceshelf=zeros(md.numberofnodes,1);
	nodeoniceshelf(md.elements(find(md.elementoniceshelf),:))=1;
	nodeonicefront=double(md.nodeonboundary & nodeoniceshelf);
end
pos=find(md.nodeonboundary & ~nodeonicefront);
if isempty(pos),
	warning('SetMarineIceSheetBC warning: ice front all around the glacier, no dirichlet found. Dirichlet must be added manually')
end
md.spcvelocity=zeros(md.numberofnodes,6);
md.spcvelocity(pos,1:3)=1;
md.diagnostic_ref=NaN*ones(md.numberofnodes,6);

%Dirichlet Values
if (length(md.vx_obs)==md.numberofnodes & length(md.vy_obs)==md.numberofnodes)
	disp('      boundary conditions for diagnostic model: spc set as observed velocities');
	md.spcvelocity(pos,4:5)=[md.vx_obs(pos) md.vy_obs(pos)]; %zeros for vz
else
	disp('      boundary conditions for diagnostic model: spc set as zero');
end

md.spcwatercolumn=zeros(md.numberofnodes,2);
pos=find(md.nodeonboundary); 
md.spcwatercolumn(pos,1)=1;

%segment on Neumann (Ice Front)
pos=find(nodeonicefront(md.segments(:,1)) | nodeonicefront(md.segments(:,2)));
if (md.dim==2)
	pressureload=md.segments(pos,:);
elseif md.dim==3
	pressureload_layer1=[md.segments(pos,1:2)  md.segments(pos,2)+md.numberofnodes2d  md.segments(pos,1)+md.numberofnodes2d  md.segments(pos,3)];
	pressureload=[];
	for i=1:md.numlayers-1,
		pressureload=[pressureload ;pressureload_layer1(:,1:4)+(i-1)*md.numberofnodes2d pressureload_layer1(:,5)+(i-1)*md.numberofelements2d ];
	end
end

%Add water or air enum depending on the element
pressureload=[pressureload WaterEnum*md.elementoniceshelf(pressureload(:,end))+AirEnum*md.elementonicesheet(pressureload(:,end))];

%plug onto model
md.pressureload=pressureload;

%Create zeros melting_rate and accumulation_rate if not specified
if isnan(md.accumulation_rate),
	md.accumulation_rate=zeros(md.numberofnodes,1);
	disp('      no accumulation_rate specified: values set as zero');
end
if isnan(md.melting_rate),
	md.melting_rate=zeros(md.numberofnodes,1);
	disp('      no melting_rate specified: values set as zero');
end
if isnan(md.dhdt),
	md.dhdt=zeros(md.numberofnodes,1);
	disp('      no dhdt specified: values set as zero');
end

md.spcthickness=zeros(md.numberofnodes,2);

if (length(md.observed_temperature)==md.numberofnodes),
	md.spctemperature=[md.nodeonsurface md.observed_temperature]; %impose observed temperature on surface
	if (length(md.geothermalflux)~=md.numberofnodes),
		md.geothermalflux=zeros(md.numberofnodes,1);
		md.geothermalflux(find(md.nodeonicesheet))=50*10^-3; %50mW/m2
	end
else
	disp('      no thermal boundary conditions created: no observed temperature found');
end
