function plot_overlay(md,data,options,plotlines,plotcols,i)
%PLOT_OVERLAY - superimpose radar image to a given field
%
%   Usage:
%      plot_overlay(md,options,plotlines,plotcols,i)
%
%   See also: PLOTMODEL

%process mesh and data
[x y z elements is2d]=processmesh(md,[],options);
if strcmpi(data,'none'),
	radaronly=1;
	data=NaN*ones(md.numberofnodes,1);
	datatype=1;
else
	radaronly=0;
	[data datatype]=processdata(md,data,options);
end

%check is2d
if ~is2d, 
	error('buildoverlay error message: overlay not supported for 3d meshes, project on a layer');
end
if datatype==3,
	error('buildoverlay error message: overlay not supported for quiver plots');
end

%Get xlim and ylim (used to extract radar image)
xlim=getfieldvalue(options,'xlim',[min(x) max(x)]);
ylim=getfieldvalue(options,'ylim',[min(y) max(y)]);
xlim2=xlim/getfieldvalue(options,'unit',1);
ylim2=ylim/getfieldvalue(options,'unit',1);

%radar power
if ~any(isnan(md.sarxm)) & ~any(isnan(md.sarym)) & ~any(isnan(md.sarpwr)),
	disp('plot_overlay info: the radar image held by the model is being used');
else
	disp('Extracting radar image...');
	options=addfielddefault(options,'xlim',xlim2);
	options=addfielddefault(options,'ylim',ylim2);
	md=radarpower(md,options);
end

%InterpFromMeshToGrid
cornereast =min(xlim); cornernorth=max(ylim);
xspacing=(max(xlim)-min(xlim))/(length(md.sarxm));
yspacing=(max(ylim)-min(ylim))/(length(md.sarym));
nlines=length(md.sarym);
ncols =length(md.sarxm);
disp('Interpolating data on grid...');
[x_m y_m data_grid]=InterpFromMeshToGrid(elements,x,y,data,cornereast,cornernorth,xspacing,yspacing,nlines,ncols,NaN);

%Process data_grid
if exist(options,'caxis'),
	caxis_opt=getfieldvalue(options,'caxis');
	data_grid(find(data_grid<caxis_opt(1)))=caxis_opt(1);
	data_grid(find(data_grid>caxis_opt(2)))=caxis_opt(2);
	data_min=caxis_opt(1);
	data_max=caxis_opt(2);
else
	data_min=min(data_grid(:));
	data_max=max(data_grid(:));
end
data_nan=find(isnan(data_grid));

%Generate HSV image
contrast=getfieldvalue(options,'contrast',1);  
transparency=getfieldvalue(options,'alpha',1);
data_grid(data_nan)=data_min; 

colorm=getfieldvalue(options,'colormap','Rignot');
if strcmpi(colorm,'Rignot'),
	%hue (H)
	h_data=(data_grid-data_min)/(data_max-data_min);
	if radaronly,
		h_data(:)=0;
	end
	%saturation (S)
	s_data=max(min((0.1+h_data).^(1/transparency),1),0);
elseif strcmpi(colorm,'Seroussi'),
	%hue (H)
	h_data=1-(data_grid-data_min)/(data_max-data_min)*0.7;
	%h_data=(data_grid-data_min)/(data_max-data_min)*2/3;
	if radaronly,
		h_data(:)=0;
	end
	%saturation (S)
	s_data=max(min((0.1+h_data).^(1/transparency),1),0);
elseif strcmpi(colorm,'redblue')
	data_mean=data_min+(data_max-data_min)/2;
	%hue (H)
	%h_data=0.7*ones(size(data_grid));
	%h_data(find(data_grid>data_mean))=1;
	h_data=1*ones(size(data_grid));
	h_data(find(data_grid>data_mean))=0.7;
	%saturation (S)
	s_data=max(min(abs(data_grid-data_mean)/(data_max-data_mean) ,1),0);
else
	error('colormap not supported yet. (''Rignot'' and ''redblue'' are the only cupported colormaps)');
end

%Saturation is 0 in NaNs
s_data(data_nan)=0;
%intensity (V)
radar=(md.sarpwr).^(contrast);
v_data=radar/max(radar(:)); %use radar power as intensity

%Transform HSV to RGB
image_hsv=zeros(size(data_grid,1),size(data_grid,2),3);
image_hsv(:,:,1)=h_data;
image_hsv(:,:,2)=s_data;
image_hsv(:,:,3)=v_data;
image_rgb=hsv2rgb(image_hsv);

%Select plot area 
subplot(plotlines,plotcols,i);
%uncomment following lines to have more space
%P=get(gca,'pos');
%P(3)=P(3)+0.07;
%P(4)=P(4)+0.07;
%%P(2)=P(2)+0.08;
%P(1)=P(1)-0.03;
%set(gca,'pos',P);

%Plot: 
imagesc(md.sarxm*getfieldvalue(options,'unit',1),md.sarym*getfieldvalue(options,'unit',1),image_rgb);set(gca,'YDir','normal');

%last step: mesh overlay?
if exist(options,'edgecolor'),
	hold on
	A=elements(:,1); B=elements(:,2); C=elements(:,3); 
	trisurf([A B C],x,y,z,'EdgeColor',getfieldvalue(options,'edgecolor'),'FaceColor','none')
end

%Apply options, without colorbar and without grid
options=changefieldvalue(options,'colormap',colorm);           %We used an HSV colorbar
if ~isnan(data_min),
	options=changefieldvalue(options,'caxis',[data_min data_max]); %force caxis so that the colorbar is ready
end
options=addfielddefault(options,'axis','equal off');           %default axis
applyoptions(md,data,options);
