function md2=meshadaptation(md,field,scale,epsilon,threshold)
%MESHADAPTATION - remesh a model for a given field
%
%   remesh a model for a given field, use the hessian computation to spread the error on the element
%   scale: 1.2 large scale leads to higher refining.
%   epsilon: .5  large epsilon will refine fewer elements
%   threshold: value under which no refinement is done.
%
%   Usage:
%      md2=meshadaptation(md,field,scale,epsilon,threshold)
%
%   Example:
%      md2=meshadaptation(md,md.vel,1.2,0.5,1);

%some checks
if ~(md.dim==2)
	error('meshadaptation error message: adaptation for 3d meshes not implemented yet')
end
if length(field)~=md.numberofnodes
	error('meshadaptation error message: input field length shoud be numberofnodes')
end

disp(sprintf('      metric computation') )

%initialization
index=md.elements;
numberofnodes=md.numberofnodes;
numberofelements=md.numberofelements;
gradx=zeros(numberofnodes,1);
grady=zeros(numberofnodes,1);
metric=zeros(numberofelements,1);

%take the threshold, a characteristic length, and get the area equivalent threshold:
threshold=threshold^2/2;

%build some usefull variables
field=field(:);
line=index(:);
summation=1/3*ones(3,1);
linesize=3*numberofelements;

%get areas and  nodal functions coefficients N(x,y)=alpha x + beta y + gamma 
[alpha beta]=GetNodalFunctionsCoeff(index,md.x,md.y);
areas=GetAreas(index,md.x,md.y);

%Compute gradient for each element
grad_elx=sum(field(index).*alpha,2); 
grad_ely=sum(field(index).*beta,2);

%update weights that holds the volume of all the element holding the node i
weights=sparse(line,ones(linesize,1),repmat(areas,3,1),numberofnodes,1);

%Compute gradient for each node (average of the elements around)
gradx=sparse(line,ones(linesize,1),repmat(areas.*grad_elx,3,1),numberofnodes,1);
grady=sparse(line,ones(linesize,1),repmat(areas.*grad_ely,3,1),numberofnodes,1);
gradx=gradx./weights;
grady=grady./weights;

%Compute hessian for each element
hessian=[sum(gradx(index).*alpha,2) sum(grady(index).*beta,2) sum(grady(index).*alpha,2)];

%Compute the new metric
hmin=min(sqrt(areas))/scale;
hmax=max(sqrt(areas));

%first, find the eigen values of eah line of H=[hessian(i,1) hessian(i,3); hessian(i,3)  hessian(i,2)]
a=hessian(:,1); b=hessian(:,3); d=hessian(:,2);
lambda1=0.5*((a+d)+sqrt(4*b.^2+(a-d).^2));
lambda2=0.5*((a+d)-sqrt(4*b.^2+(a-d).^2));

%Modify the eigen values to control the shape of the elements
lambda1=min(max(abs(lambda1)/epsilon,1/hmax^2),1/hmin^2);
lambda2=min(max(abs(lambda2)/epsilon,1/hmax^2),1/hmin^2);

%find the corresponding eigen vectors of the initial eigen values
%NOT USED FOR NOW
%div1=sqrt(1+(b./(lambda1-d)).^2);
%div2=sqrt(1+(b./(lambda2-d)).^2);
%vector1=[1./div1 (b./(lambda1-d))./div1];
%vector2=[1./div2 (b./(lambda2-d))./div2];

%compute the metric
%R=[vector1(:,1) vector2(:,1) vector1(:,2) vector2(:,2)];
%det=vector1(:,1).*vector2(:,2)-vector1(:,2).*vector1(:,2);
%Rinv=[vector2(:,2)./det   -vector2(:,1)./det   -vector1(:,2)./det   vector1(:,1)./det];
metric=20*1./sqrt(lambda1.^2+lambda2.^2);

%ok, metric can't go under 1km resolution.
metric(find(metric<threshold))=threshold;

%Remesh with this new metric
disp(sprintf('      initial number of element: %i', md.numberofelements))
md2=meshrefine(md,full(metric));
disp(sprintf('      new number of elements:    %i', md2.numberofelements))
