function ismodelselfconsistent(md),
%ISMODELSELFCONSISTENT - check that model forms a closed form solvable problem.
%
%   Usage:
%      ismodelselfconsistent(md),

%tolerance we use in litmus checks for the consistency of the model
tolerance=10^-9;
%check usage {{{1
if nargin~=1,
	help ismodelselfconsistent
	error('ismodelselfconsistent error message: wrong usage');
end
%}}}

% Common checks
%VERBOSE{{{1
if ~strcmp('verbose',class(md.verbose))
	disp('WARNING: md.verbose must now be an object of class ''verbose''');
	disp('         To update your model use the following command:');
	disp(' ');
	disp('         md.verbose=verbose;');
	disp(' ');
	error(['field verbose should be of class ''verbose'' ']);
end
%}}}
%COUNTER {{{1
if md.counter<3,
	error(['model ' md.name ' is not correctly configured. You forgot one step in the following sequence (mesh, geography, parameterize,setelementstype)!']);
end
%}}}
%NAME{{{1
if isempty(md.name),
	error(['model is not correctly configured: missing name!']);
end
%}}}
%ELEMENTS{{{1
fields={'elements'};
if (md.dim==2),
	checksize(md,fields,[md.numberofelements 3]);
else
	checksize(md,fields,[md.numberofelements 6]);
end
if any(~ismember(1:md.numberofgrids,sort(unique(md.elements(:)))));
	error('orphan nodes have been found. Check the mesh');
end
%}}}

%DG {{{1
if md.prognostic_DG==1;
	%CHECK THE COMPATIBILITY OF THE EDGES
	fields={'edges'};
	checksize(md,fields,[NaN 4]);
	fields={'edges(:,1:3)'};
	checknan(md,fields);
	checkgreaterstrict(md,fields,0);
end
%}}}
%PRESSURELOAD{{{1
if (md.dim==2),
	fields={'pressureload'};
	checksize(md,fields,[NaN 4]);
elseif md.dim==3,
	fields={'pressureload'};
	checksize(md,fields,[NaN 6]);
else
	error('dim should be either 2 3');
end
checkvalues(md,{'pressureload(:,end)'},[WaterEnum() AirEnum() IceEnum()]);
%}}}
%NO NAN {{{1
fields={'numberofelements','numberofgrids','x','y','z','drag_coefficient','drag_type','drag_p','drag_q',...
	'rho_ice','rho_water','rheology_B','elementoniceshelf','surface','thickness','bed','g','lowmem','sparsity','nsteps','maxiter',...
	'tolx','eps_res','max_nonlinear_iterations','rheology_n','gridonbed','gridonsurface','elementonbed','elementonsurface','deltaH','DeltaH','timeacc','timedec','elementconnectivity'};
checknan(md,fields);
%}}}}
%FIELDS >= 0 {{{1
fields={'numberofelements','numberofgrids','elements','drag_coefficient','drag_type','drag_p','drag_q',...
	'rho_ice','rho_water','rheology_B','elementoniceshelf','thickness','g','eps_res','max_nonlinear_iterations','eps_rel','eps_abs','nsteps','maxiter','tolx',...
	'sparsity','lowmem','rheology_n','gridonbed','gridonsurface','elementonbed','elementonsurface','deltaH','DeltaH','timeacc','timedec'};
checkgreater(md,fields,0);
%}}}
%FIELDS > 0 {{{1
fields={'numberofelements','numberofgrids','elements','drag_type','drag_p',...
	'rho_ice','rho_water','rheology_B','thickness','g','max_nonlinear_iterations','eps_res','eps_rel','eps_abs','maxiter','tolx',...
	'sparsity','deltaH','DeltaH','timeacc','timedec'};
checkgreaterstrict(md,fields,0);
%}}}
%SIZE NUMBEROFELEMENTS {{{1
fields={'drag_p','drag_q','elementoniceshelf','rheology_n','elementonbed'};
checksize(md,fields,[md.numberofelements 1]);
%}}}
%SIZE NUMBEROFGRIDS {{{1
fields={'x','y','z','rheology_B','drag_coefficient','melting_rate','accumulation_rate','surface','thickness','bed','gridonbed','gridonsurface'};
checksize(md,fields,[md.numberofgrids 1]);
%}}}
%OTHER SIZES {{{1
fields={'spcvelocity','diagnostic_ref'};
checksize(md,fields,[md.numberofgrids 6]);
%}}}
%THICKNESS = SURFACE - BED {{{1
if any((md.thickness-md.surface+md.bed)>tolerance),
	error(['model not consistent: model ' md.name ' violates the equality thickness=surface-bed!']);
end
%GROUNDING LINE MIGRATION {{{1
checkvalues(md,{'gl_migration'},[NoneEnum() AgressiveMigrationEnum() SoftMigrationEnum()]);
if (md.gl_migration~=NoneEnum),
	if (md.dim==3 | strcmpi(md.cluster.name,'none')),
		error(['model ' md.name ' requesting grounding line migration, but grounding line module only implemented for 2d models and parallel runs!']);
	end
	if isnan(md.bathymetry),
		error(['model not consistent: model ' md.name ' requesting grounding line migration, but bathymetry is absent!']);
	end
end

%}}}
%RIFTS{{{1
if md.numrifts,
	if ~(md.dim==2),
		error(['model not consistent: models with rifts are only supported in 2d for now!']);
	end
	if ~isstruct(md.rifts),
		error(['model not consistent: md.rifts should be a structure!']);
	end
	if ~isempty(find(md.segmentmarkers>=2)),
		%We have segments with rift markers, but no rift structure!
		error(['model not consistent: model ' md.name ' should be processed for rifts (run meshprocessrifts)!']);
	end
	%Check that rifts are filled with proper material
	checkvalues(md,{'rifts.fill'},[WaterEnum() AirEnum() IceEnum() MelangeEnum()]);
else
	if ~isnans(md.rifts),
		error(['model not consistent: md.rifts shoud be NaN since md.numrifts is 0!']);
	end
end
%}}}
%FLAGS (0 or 1){{{1
if ~ismember(md.artificial_diffusivity,[0 1 2]),
	error('model not consistent: artificial_diffusivity should be a scalar (1 or 0)');
end
if ~ismember(md.prognostic_DG,[0 1]),
	error('model not consistent: prognostic_DG should be a scalar (1 or 0)');
end
if ~ismember(md.lowmem,[0 1]),
	error(['model not consistent: model ' md.name ' lowmem field should be 0 or 1']);
end
if ~ismember(md.time_adapt,[0 1]),
	error(['model not consistent: model ' md.name ' time_adapt field should be 0 or 1']);
end
if ~ismember(md.hydrostatic_adjustment,[AbsoluteEnum IncrementalEnum]),
	error(['model not consistent: model ' md.name ' hydrostatic_adjustment field should be AbsoluteEnum or IncrementalEnum']);
end
if ~ismember(md.rheology_law,[NoneEnum PatersonEnum ArrheniusEnum]),
	error(['model not consistent: model ' md.name ' rheology_law field should be NoneEnum, PatersonEnum or ArrheniusEnum']);
end
%}}}
%PARAMETEROUTPUT {{{1
if md.numoutput~=length(md.parameteroutput),
	error('model not consistent: numoutput should be the same size as parameteroutput');
end
%}}}
%CONNECTIVITY {{{1
if (md.dim==2),
	if md.connectivity<9, 
		error('model not consistent: connectivity should be at least 9 for 2d models');
	end
end
if md.dim==3,
	if md.connectivity<24, 
		error('model not consistent: connectivity should be at least 24 for 3d models');
	end
end
%}}}
%PARALLEL{{{1
IsConsistent(md.cluster);
%}}}
%CONTROL{{{1
if md.control_analysis,

	%CONTROL TYPE
	num_controls=numel(md.control_type);
	checkvalues(md,{'control_type'},[DhDtEnum DragCoefficientEnum RheologyBbarEnum VxEnum VyEnum]);

	%LENGTH CONTROL FIELDS
	fields={'maxiter','cm_responses','cm_jump'};
	checksize(md,fields,[md.nsteps 1]);
	fields={'optscal'};
	checksize(md,fields,[md.nsteps num_controls]);
	fields={'cm_min','cm_max'};
	checksize(md,fields,[md.numberofgrids num_controls]);

	%RESPONSES
	checkvalues(md,{'cm_responses'},[SurfaceAbsVelMisfitEnum SurfaceRelVelMisfitEnum SurfaceLogVelMisfitEnum SurfaceLogVxVyMisfitEnum SurfaceAverageVelMisfitEnum ThicknessAbsMisfitEnum]);

	%WEIGHTS
	fields={'weights'};
	checksize(md,fields,[md.numberofgrids 1]);
	checkgreater(md,fields,0);

	%OBSERVED VELOCITIES
	if md.solution_type==BalancethicknessSolutionEnum
		fields={'thickness_obs'};
		checksize(md,fields,[md.numberofgrids 1]);
		checknan(md,fields);
	else
		fields={'vx_obs','vy_obs'};
		checksize(md,fields,[md.numberofgrids 1]);
		checknan(md,fields);
	end

	%DIRICHLET IF THICKNESS <= 0
	if any(md.thickness<=0),
		pos=find(md.thickness<=0);
		if any(find(md.spcthickness(pos,1)==0)),
			error(['model not consistent: model ' md.name ' has some grids with 0 thickness']);
		end
	end

	%parameters
	fields={'cm_noisedmp'};
	checknan(md,fields);
end
%}}}
%QMU {{{1
if md.qmu_analysis,
	if md.qmu_params.evaluation_concurrency~=1,
		error(['model not consistent: concurrency should be set to 1 when running dakota in library mode']);
	end
	if ~isempty(md.part),
		if numel(md.part)~=md.numberofgrids,
			error(['model not consistent: user supplied partition for qmu analysis should have size md.numberofgrids x 1 ']);
		end
		if find(md.part)>=md.numberofgrids,
			error(['model not consistent: user supplied partition should be indexed from 0 (c-convention)']);
		end
		if min(md.part)~=0,
			error(['model not consistent: partition vector not indexed from 0 on']);
		end
		if max(md.part)>=md.numberofgrids,
			error(['model not consistent: partition vector cannot have maximum index larger than number of grids']);
		end
		if ~isempty(find(md.part<0)),
			error(['model not consistent: partition vector cannot have values less than 0']);
		end
		if ~isempty(find(md.part>=md.npart)),
			error(['model not consistent: partition vector cannot have values more than md.npart-1']);
		end
		if max(md.part)>=md.npart,
			error(['model not consistent: for qmu analysis, partitioning vector cannot go over npart, number of partition areas']);
		end
	end
	if ~md.qmu_relax,
		if md.eps_rel>1.1*10^-5,
			error(['model not consistent: for qmu analysis, eps_rel should be least than 10^-5, 10^-15 being a better value']);
		end
	end
end

if strcmpi(md.solution_type,'qmu'),
	if ~strcmpi(md.cluster.name,'none'),
		if md.waitonlock==0,
			error(['model is not correctly configured: waitonlock should be activated when running qmu in parallel mode!']);
		end
	end
end
%}}}

%Solution specific check
%TRANSIENT {{{1
if (md.solution_type==Transient2DSolutionEnum | md.solution_type==Transient3DSolutionEnum),

	if md.dt<=0,
		error('model not consistent: field dt must be positive for a transient run')
	end

	if(md.cfl_coefficient>1 | md.cfl_coefficient<0),
		error(['model not consistent: model ' md.name ' cfl_coefficient field should between  0 and 1']);
	end
end
%}}}
%STEADYSTATE{{{1
if md.solution_type==SteadystateSolutionEnum,

	%NDT
	if md.dt~=0,
		error(['model not consistent: for a steadystate computation, dt must be zero.']);
	end

	%eps: 
	if isnan(md.eps_rel),
		error(['model not consistent: for a steadystate computation, eps_rel (relative convergence criterion) must be defined!']);
	end
end
%}}}
%GROUNDINGLINEMIGRATION2D{{{1
if md.solution_type==GroundingLineMigration2DSolutionEnum,
	if strcmpi(md.cluster.name,'none'),
		error(['model not consistent: ' md.solution_type ' is only implemented in parallel mode !'])
	end

	if md.dt<=0,
		error('model not consistent: field dt must be positive for a transient run')
	end

	%recursive call to ismodelselfconsistent
	if (md.dim~=2),
		error(['model not consistent: for a ' md.solution_type ' computation, the grounding line module is only implemented in 2d !'])
	end

	if(md.cfl_coefficient>1 | md.cfl_coefficient<0),
		error(['model not consistent: model ' md.name ' cfl_coefficient field should between  0 and 1']);
	end
end
%}}}
%FLAIM {{{1
if (md.solution_type == FlaimSolutionEnum),
	if ~exist(md.fm_tracks,'file')
		error(['model not consistent: fm_tracks file ''' md.fm_tracks ''' must exist.']);
	end
	%   probably going to need some checks on fm_flightreqs here
	if (numel(md.fm_criterion) ~= md.numberofgrids) && (numel(md.fm_criterion) ~= md.numberofelements)
		error(['model not consistent: fm_criterion vector must have number of nodes (' int2str(md.numberofgrids) ') or elements (' int2str(md.numberofelements) ') values, not ' int2str(numel(md.fm_criterion)) ' values.']);
	end
end
%}}}

%Now check all analyses called for a given solution
%ANALYSESCHECKS {{{1
[analyses,numanalyses,solutioncore]=SolutionConfiguration(md.solution_type);
for i=1:numanalyses,
	checkforanalysis(md,analyses(i));
end
%}}}
end

%analysis checks
%checkforanalysis {{{1
function checkforanalysis(md,analysis_enum)

	switch(analysis_enum),
		case DiagnosticHorizAnalysisEnum,
			% {{{2
			%SINGULAR
			if ~any(sum(md.spcvelocity(:,1:2),2)==2),
				error(['model not consistent: model ' md.name ' is not well posed (singular). You need at least one grid with fixed velocity!'])
			end

			%DIRICHLET IF THICKNESS <= 0
			if any(md.thickness<=0),
				pos=find(md.thickness<=0);
				if any(find(md.spcthickness(pos,1)==0)),
					error(['model not consistent: model ' md.name ' has some grids with 0 thickness']);
				end
			end

			%ROTATED SPC
			%CHECK THAT EACH LINES CONTAINS ONLY NAN VALUES OR NO NAN VALUES
			if any(sum(isnan(md.diagnostic_ref),2)~=0 & sum(isnan(md.diagnostic_ref),2)~=6),
				error(['model not consistent: model ' md.name ' has problem with rotated spc. Each line of diagnostic_ref should contain either only NaN values or no NaN values']);
			end
			%CHECK THAT THE TWO VECTORS PROVIDED ARE ORTHOGONAL
			if any(sum(isnan(md.diagnostic_ref),2)==0),
				pos=find(sum(isnan(md.diagnostic_ref),2)==0);
				if any(dot(md.diagnostic_ref(pos,1:3),md.diagnostic_ref(pos,4:6))),
					dot(md.diagnostic_ref(pos,1:3),md.diagnostic_ref(pos,4:6))
					error(['model not consistent: model ' md.name ' has problem with rotated spc. Vectors in diagnostic_ref (colums 1 to 3 and 4 to 6) must be orthogonal']);
				end

			end
			%CHECK THAT ROTATION IS IN THE (X,Y) PLANE FOR 2D MODELS
			if any(md.gridonmacayeal),
				pos=find(sum(isnan(md.diagnostic_ref),2)==0  & md.gridonmacayeal);
				if any(md.diagnostic_ref(pos,3:5)~=0);
					error(['model not consistent: model ' md.name ' has problem with rotated spc. The rotation should be in the (x,y) plane for 2D diagnostic models (gridonmacayeal)']);
				end
			end

			%INITIAL VELOCITY
			if ~isnan(md.vx) & ~isnan(md.vy),
				fields={'vx','vy'};
				checknan(md,fields);
				checksize(md,fields,[md.numberofgrids 1]);
			end

			%ELEMENTSTYPE
			%Check the size of elements_type
			fields={'elements_type'};
			checksize(md,fields,[md.numberofelements 1]);
			%Check the values of elements_type
			checkvalues(md,{'elements_type'},[MacAyealApproximationEnum() HutterApproximationEnum() PattynApproximationEnum()...
				MacAyealPattynApproximationEnum() MacAyealStokesApproximationEnum() PattynStokesApproximationEnum() StokesApproximationEnum() NoneApproximationEnum()]);
			if (md.dim==2),
				checkvalues(md,{'elements_type'},[MacAyealApproximationEnum() HutterApproximationEnum()]);
			end
			if (md.ismacayealpattyn==0 && md.ishutter==0 && md.isstokes==0),
				error(['model not consistent: no elements type set for this model. at least one of ismacayealpattyn, ishutter and isstokes need to be =1']);
			end
			
			%VERTICESTYPE
			%Check the size of verticess_type
			fields={'vertices_type'};
			checksize(md,fields,[md.numberofgrids 1]);
			%Check the values of vertices_type
			checkvalues(md,{'vertices_type'},[MacAyealApproximationEnum() HutterApproximationEnum() PattynApproximationEnum()...
				MacAyealPattynApproximationEnum() StokesApproximationEnum() MacAyealStokesApproximationEnum() PattynStokesApproximationEnum() NoneApproximationEnum()]);
			if (md.dim==2),
				checkvalues(md,{'vertices_type'},[MacAyealApproximationEnum() HutterApproximationEnum()]);
			end
			if (md.ismacayealpattyn==0 && md.ishutter==0 && md.isstokes==0),
				error(['model not consistent: no elements type set for this model. at least one of ismacayealpattyn, ishutter and isstokes need to be =1']);
			end
			%}}}
		case DiagnosticVertAnalysisEnum,
			% {{{2
			if md.dim==3,
				% No checks for now
			end
			%}}}
		case DiagnosticHutterAnalysisEnum,
			% {{{2
			%HUTTER ON ICESHELF WARNING
			if any(md.elements_type==HutterApproximationEnum & md.elementoniceshelf),
				disp(sprintf('\n !!! Warning: Hutter''s model is not consistent on ice shelves !!!\n'));
			end
			%}}}
		case BedSlopeAnalysisEnum,
			% {{{2
			% No checks for now
			%}}}
		case SurfaceSlopeAnalysisEnum,
			% {{{2
			% No checks for now
			%}}}
		case PrognosticAnalysisEnum,
			% {{{2
			%INITIAL VELOCITIES
			fields={'vx','vy'};
			checksize(md,fields,[md.numberofgrids 1]);
			checknan(md,fields);

			%CHECK THAT WE ARE NOT FULLY CONSTRAINED
			if (md.dim==2),
				if isnan(find(~md.spcthickness(:,1))),
					error(['model not consistent: model ' md.name ' is totally constrained for prognostic, no need to solve!']);
				end
			end
			%}}}
		case ThermalAnalysisEnum,
			% {{{2
			%EXTRUSION
			if (md.dim==2),
				error(['model not consistent: for a ' md.solution_type ' computation, the model must be 3d, extrude it first!'])
			end

			%CHECK THAT WE ARE NOT FULLY CONSTRAINED
			if isnan(find(~md.spctemperature(:,1))),
				error(['model not consistent: model ' md.name ' is totally constrained for temperature, no need to solve!']);
			end

			%VELOCITIES AND PRESSURE
			fields={'vx','vy','vz','pressure','geothermalflux'};
			checksize(md,fields,[md.numberofgrids 1]);
			checknan(md,fields);

			%THERMAL TRANSIENT
			if md.dt~=0,

				%DT and NDT
				fields={'dt','ndt'};
				checkgreaterstrict(md,fields,0);

				%INITIAL TEMPERATURE, MELTING AND ACCUMULATION
				fields={'temperature','accumulation_rate','melting_rate'};
				checksize(md,fields,[md.numberofgrids 1]);
				checknan(md,fields);

				%INITIAL TEMPERATURE
				fields={'temperature','spctemperature(:,2)','observed_temperature'};
				checkgreater(md,fields,0)
			end
			%}}}
		case MeltingAnalysisEnum,
			% {{{2
			% No checks for now
			%}}}
		case BalancethicknessAnalysisEnum,
			% {{{2
			%VELOCITIES MELTING AND ACCUMULATION
			fields={'vx','vy','accumulation_rate','melting_rate','dhdt'};
			checksize(md,fields,[md.numberofgrids 1]);
			checknan(md,fields);

			%SPC				 
			%	if ~md.prognostic_DG,
			%		if any(md.spcthickness(find(md.gridonboundary))~=1),		 
			%			error(['model not consistent: model ' md.name ' should have all the nodes on boundary constrained in field spcthickness']);			 
			%		end 
			%	end

			%Triangle with zero velocity
			if any(sum(abs(md.vx(md.elements)),2)==0 & sum(abs(md.vy(md.elements)),2)==0)
				error('model not consistent: at least one triangle has all its vertices with a zero velocity');
			end
			%}}}
		case BalancevelocitiesAnalysisEnum,
			% {{{2
			%VELOCITIES MELTING AND ACCUMULATION
			fields={'vx','vy','accumulation_rate','melting_rate'};
			checksize(md,fields,[md.numberofgrids 1]);
			checknan(md,fields);

			%SPC
			if any(md.spcvelocity(find(md.gridonboundary),[1:2])~=1),
				error(['model not consistent: model ' md.name ' should have all the nodes on boundary constrained in field spcvelocity']);
			end
			%}}}
		case FlaimAnalysisEnum,
			% {{{2
			% No checks for now
			%}}}
		otherwise
			disp(['WARNING: no check implemented for analysis ' EnumToString(analysis_enum) '!']);
			error('stop')
	end

end
%}}}

%checks additional functions
%checklength {{{1
function checklength(md,fields,fieldlength)
	%CHECKSIZE - check length of a field
	for i=1:length(fields),
		eval(['field=md.' fields{i} ';']);
		if length(field)~=fieldlength,
			error(['model not consistent: field ' fields{i} ' length should be ' num2str(fieldlength)]);
		end
	end
end
%}}}
%checksize {{{1
function checksize(md,fields,fieldsize)
	%CHECKSIZE - check size of a field
	for i=1:length(fields),
		eval(['field=md.' fields{i} ';']);
		if isnan(fieldsize(1)),
			if (size(field,2)~=fieldsize(2)),
				error(['model not consistent: field ' fields{i} ' should have ' num2str(fieldsize(2)) ' columns']);
			end
		elseif isnan(fieldsize(2)),
			if (size(field,1)~=fieldsize(1)),
				error(['model not consistent: field ' fields{i} ' should have ' num2str(fieldsize(1)) ' rows']);
			end
		else
			if ((size(field)~=fieldsize(1)) |  (size(field,2)~=fieldsize(2)))
				%WARNING: new version
				if strcmp(fields{i},'cm_min'),
					disp('!!! ');
					disp('!!! WARNING: cm_min must now be of size [md.numberofgrids x 1]. Update your parameter file as follows:');
					disp('!!! ');
					disp('!!! md.cm_min=md.cm_min*ones(md.numberofgrids,1);');
					disp('!!! ');
				end
				%WARNING: new version
				if strcmp(fields{i},'cm_max'),
					disp('!!! ');
					disp('!!! WARNING: cm_max must now be of size [md.numberofgrids x 1]. Update your parameter file as follows:');
					disp('!!! ');
					disp('!!! md.cm_max=md.cm_max*ones(md.numberofgrids,1);');
					disp('!!! ');
				end
				error(['model not consistent: field ' fields{i} ' size should be ' num2str(fieldsize(1)) ' x ' num2str(fieldsize(2))]);
			end
		end
	end
end
%}}}
%checknan {{{1
function checknan(md,fields)
	%CHECKNAN - check nan values of a field
	for i=1:length(fields),
		eval(['field=reshape(md.' fields{i} ',[prod(size(md.' fields{i} ')) 1]);']);
		if any(isnan(field)),
			error(['model not consistent: NaN values found in field ' fields{i}]);
		end
	end
end
%}}}
%checkreal{{{1
function checkreal(md,fields)
	%CHECKREAL - check real values of a field
	for i=1:length(fields),
		eval(['field=reshape(md.' fields{i} ',[prod(size(md.' fields{i} ')) 1]);']);
		if any(~isreal(field)),
			error(['model not consistent: complex values found in field ' fields{i}]);
		end
	end
end
%}}}
%checkgreaterstrict{{{1
function checkgreaterstrict(md,fields,lowerbound)
	%CHECKGREATERSTRICT - check values of a field
	for i=1:length(fields),
		eval(['field=reshape(md.' fields{i} ',[prod(size(md.' fields{i} ')) 1]);']);
		if any(field<=lowerbound),
			error(['model not consistent: field ' fields{i} ' should have values stricly above ' num2str(lowerbound)]);
		end
	end
end
%}}}
%checkgreater{{{1
function checkgreater(md,fields,lowerbound)
	%CHECKGREATER - check values of a field
	for i=1:length(fields),
		eval(['field=reshape(md.' fields{i} ',[prod(size(md.' fields{i} ')) 1]);']);
		if any(field<lowerbound),
			error(['model not consistent: field ' fields{i} ' should have values above ' num2str(lowerbound)]);
		end
	end
end
%}}}
%checklessstrict{{{1
function checklessstrict(md,fields,upperbound)
	%CHECKLESSSTRICT - check values of a field
	for i=1:length(fields),
		eval(['field=reshape(md.' fields{i} ',[prod(size(md.' fields{i} ')) 1]);']);
		if any(field>=upperbound),
			error(['model not consistent: field ' fields{i} ' should have values stricly below ' num2str(upperbound)]);
		end
	end
end
%}}}
%checkless{{{1
function checkless(md,fields,upperbound)
	%CHECKLESS - check values of a field
	for i=1:length(fields),
		eval(['field=reshape(md.' fields{i} ',[prod(size(md.' fields{i} ')) 1]);']);
		if any(field>upperbound),
			error(['model not consistent: field ' fields{i} ' should have values below ' num2str(upperbound)]);
		end
	end
end
%}}}
%checkvalues {{{1
function checkvalues(md,fields,values)
	%CHECKVALUE - check that a field has specified values
	for i=1:length(fields),
		eval(['field=reshape(md.' fields{i} ',[prod(size(md.' fields{i} ')) 1]);']);
		if any(~ismember(field,values)),
			error(['model not consistent: field ' fields{i} ' should have values in ' num2str(values)]);
		end
	end
end
%}}}
