/*!\file:  transient.cpp
 * \brief: transient solution
 */ 

#include "../issm.h"
#include "./parallel.h"

#undef __FUNCT__ 
#define __FUNCT__ "transient"

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif


int main(int argc,char* *argv){
	
	/*I/O: */
	FILE* fid=NULL;
	char* inputfilename=NULL;
	char* outputfilename=NULL;
	char* lockname=NULL;
	char* qmuinname=NULL;
	char* qmuoutname=NULL;
	char* qmuerrname=NULL;
	int   numberofnodes;
	int   qmu_analysis=0;

	/*Fem models : */
	FemModel femmodels[8];
	int dim=-1;

	/*Results: */
	DataSet* results=NULL;
	
	ParameterInputs* inputs=NULL;
	int waitonlock=0;
	
	/*inputs: */
	double* u_g=NULL;
	double* m_g=NULL;
	double* a_g=NULL;
	double  dt;
	Param*  param=NULL;

	MODULEBOOT();

	#if !defined(_PARALLEL_) || (defined(_PARALLEL_) && !defined(_HAVE_PETSC_))
	throw ErrorException(__FUNCT__," parallel executable was compiled without support of parallel libraries!");
	#endif

	PetscInitialize(&argc,&argv,(char *)0,"");  

	/*Size and rank: */
	MPI_Comm_rank(MPI_COMM_WORLD,&my_rank);  
	MPI_Comm_size(MPI_COMM_WORLD,&num_procs); 

	_printf_("recover , input file name and output file name:\n");
	inputfilename=argv[2];
	outputfilename=argv[3];
	lockname=argv[4];
	qmuinname=argv[5];
	qmuoutname=argv[6];
	qmuerrname=argv[7];

	/*Open handle to data on disk: */
	fid=pfopen(inputfilename,"rb");

	_printf_("read and create finite element model:\n");
	_printf_("\n   reading diagnostic horiz model data:\n");
	CreateFemModel(&femmodels[0],fid,"diagnostic","horiz");
	_printf_("\n   reading diagnostic vert model data:\n");
	CreateFemModel(&femmodels[1],fid,"diagnostic","vert");
	_printf_("\n   reading diagnostic stokes model data:\n");
	CreateFemModel(&femmodels[2],fid,"diagnostic","stokes");
	_printf_("\n   reading diagnostic hutter model data:\n");
	CreateFemModel(&femmodels[3],fid,"diagnostic","hutter");
	_printf_("\n   reading surface and bed slope computation model data:\n");
	CreateFemModel(&femmodels[4],fid,"slope_compute","");
	_printf_("\n   reading prognositc model data:\n");
	CreateFemModel(&femmodels[5],fid,"prognostic","");
	
	/*Do we run in 3d?, in which case we need thermal and melting also:*/
	femmodels[0].parameters->FindParam((void*)&dim,"dim");
	if(dim==3){
		_printf_("read and create thermal finite element model:\n");
		CreateFemModel(&femmodels[6],fid,"thermal",NULL);
		_printf_("read and create melting finite element model:\n");
		CreateFemModel(&femmodels[7],fid,"melting",NULL);
	}

	_printf_("initialize inputs:\n");
	femmodels[5].parameters->FindParam((void*)&numberofnodes,"numberofnodes");
	
	femmodels[5].parameters->FindParam((void*)&u_g,"u_g");
	inputs=new ParameterInputs; inputs->Add("velocity",u_g,2,numberofnodes);

	femmodels[5].parameters->FindParam((void*)&m_g,"m_g");
	inputs=new ParameterInputs; inputs->Add("melting",m_g,1,numberofnodes);

	femmodels[5].parameters->FindParam((void*)&a_g,"a_g");
	inputs=new ParameterInputs; inputs->Add("accumulation",a_g,1,numberofnodes);

	femmodels[5].parameters->FindParam((void*)&dt,"dt");
	inputs=new ParameterInputs; inputs->Add("dt",dt);
	
	_printf_("initialize results:\n");
	results=new DataSet(ResultsEnum());

	/*are we running the solution sequence, or a qmu wrapper around it? : */
	femmodels[5].parameters->FindParam((void*)&qmu_analysis,"qmu_analysis");
	if(!qmu_analysis){

		/*run diagnostic analysis: */
		_printf_("call computational core:\n");
		transient_core(results,femmodels,inputs);

	}
	else{
		/*run qmu analysis: */
		_printf_("calling qmu analysis on transient core:\n");

		#ifdef _HAVE_DAKOTA_ 
		qmu(qmuinname,qmuoutname,qmuerrname,&femmodels[0],inputs,TransientAnalysisEnum(),NoneAnalysisEnum());
	 	#else
		throw ErrorException(__FUNCT__," Dakota not present, cannot do qmu!");
		#endif
	}

	_printf_("process results:\n");
	ProcessResults(&results,&femmodels[0],DiagnosticAnalysisEnum());
	
	_printf_("write results to disk:\n");
	OutputResults(results,outputfilename);

	_printf_("write lock file:\n");
	femmodels[0].parameters->FindParam((void*)&waitonlock,"waitonlock");
	if (waitonlock){
		WriteLockFile(lockname);
	}
		
	_printf_("closing MPI and Petsc\n");
	PetscFinalize(); 
	

	/*end module: */
	MODULEEND();
	
	return 0; //unix success return;
}
