/*
 * \file Inputs.c
 * \brief: implementation of the Inputs class, derived from DataSet class
 */

/*Headers: {{{1*/
#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include <vector>
#include <functional>
#include <algorithm>
#include <iostream>

#include "./DataSet.h"
#include "../shared/shared.h"
#include "../include/include.h"
#include "../EnumDefinitions/EnumDefinitions.h"

using namespace std;
/*}}}*/

/*Object constructors and destructor*/
/*FUNCTION Inputs::Inputs(){{{1*/
Inputs::Inputs(){
	return;
}
/*}}}*/
/*FUNCTION Inputs::Inputs(int in_enum){{{1*/
Inputs::Inputs(int in_enum): DataSet(in_enum) {
	//do nothing;
	return;
}
/*}}}*/
/*FUNCTION Inputs::~Inputs(){{{1*/
Inputs::~Inputs(){
	return;
}
/*}}}*/

/*Object management*/
/*FUNCTION Inputs::GetParameterValue(bool* pvalue,int enum-type){{{1*/
void Inputs::GetParameterValue(bool* pvalue,int enum_type){

	vector<Object*>::iterator object;
	Input* input=NULL;
	bool   found=false;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type){
			found=true;
			break;
		}
	}

	if (!found){
		/*we could not find an input with the correct enum type. No defaults values were provided, 
		 * error out: */
		_error_("could not find input with enum type %i (%s)",enum_type,EnumToStringx(enum_type));
	}

	/*Ok, we have an input if we made it here, request the input to return the value: */
	input->GetParameterValue(pvalue);

}
/*}}}*/
/*FUNCTION Inputs::GetParameterValue(int* pvalue,int enum-type){{{1*/
void Inputs::GetParameterValue(int* pvalue,int enum_type){

	vector<Object*>::iterator object;
	Input* input=NULL;
	bool   found=false;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type){
			found=true;
			break;
		}
	}

	if (!found){
		/*we could not find an input with the correct enum type. No defaults values were provided, 
		 * error out: */
		_error_("could not find input with enum type %i (%s)",enum_type,EnumToStringx(enum_type));
	}

	/*Ok, we have an input if we made it here, request the input to return the value: */
	input->GetParameterValue(pvalue);

}
/*}}}*/
/*FUNCTION Inputs::GetParameterValue(double* pvalue,int enum-type){{{1*/
void Inputs::GetParameterValue(double* pvalue,int enum_type){

	vector<Object*>::iterator object;
	Input* input=NULL;
	bool   found=false;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type){
			found=true;
			break;
		}
	}

	if (!found){
		/*we could not find an input with the correct enum type. No defaults values were provided, 
		 * error out: */
		_error_("could not find input with enum type %i (%s)",enum_type,EnumToStringx(enum_type));
	}

	/*Ok, we have an input if we made it here, request the input to return the value: */
	input->GetParameterValue(pvalue);

}
/*}}}*/
/*FUNCTION Inputs::GetParameterAverage{{{1*/
void Inputs::GetParameterAverage(double* pvalue,int enum_type){

	vector<Object*>::iterator object;
	Input* input=NULL;
	bool   found=false;

	/*Go through inputs and check whether any input with the same name is already in: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 
		if (input->EnumType()==enum_type){
			found=true;
			break;
		}
	}

	if (!found){
		/*we could not find an input with the correct enum type. No defaults values were provided, 
		 * error out: */
		_error_("could not find input with enum type %i (%s)",enum_type,EnumToStringx(enum_type));
	}

	/*Ok, we have an input if we made it here, request the input to return the value: */
	input->GetParameterAverage(pvalue);

}
/*}}}*/
/*FUNCTION Inputs::AddInput{{{1*/
int  Inputs::AddInput(Input* in_input){

	/*First, go through dataset of inputs and check whether any input 
	 * with the same name is already in. If so, erase the corresponding 
	 * object before adding this new one: */
	vector<Object*>::iterator object;
	Input* input=NULL;

	/*In debugging mode, check that the input is not a NULL pointer*/
	_assert_(in_input);

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 

		if (input->EnumType()==in_input->EnumType()){
			this->DeleteObject(input);
			break;
		}
	}
	this->AddObject(in_input);

	return 1;
}
/*}}}*/
/*FUNCTION Inputs::ChangeEnum{{{1*/
void  Inputs::ChangeEnum(int oldenumtype,int newenumtype){

	/*Go through dataset of inputs and look for input with 
	 * same enum as input enum, once found, just change its name */
	vector<Object*>::iterator object;
	Input* input=NULL;

	/*Delete existing input of newenumtype if it exists*/
	for ( object=objects.begin() ; object < objects.end(); object++ ){
		input=(Input*)(*object); 

		if (input->EnumType()==newenumtype){
			this->DeleteObject(input);
			break;
		}
	}

	/*Change enum_type of input of oldenumtype*/
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 

		if (input->EnumType()==oldenumtype){
			input->ChangeEnum(newenumtype);
			break;
		}
	}
}
/*}}}*/
/*FUNCTION Inputs::ConstrainMin{{{1*/
void  Inputs::ConstrainMin(int constrain_enum, double minimum){
	   
	Input* constrain_input=NULL; 
	/*Find x and y inputs: */
	constrain_input=(Input*)this->GetInput(constrain_enum);

	/*some checks: */
	if(!constrain_input) _error_(" input %s could not be found!",EnumToStringx(constrain_enum));

	/*Apply ContrainMin: */
	constrain_input->ConstrainMin(minimum);
}
/*}}}*/
/*FUNCTION Inputs::InfinityNorm{{{1*/
double Inputs::InfinityNorm(int enumtype){

	/*Output*/
	double norm;

	/*Get input*/
	Input* input=(Input*)this->GetInput(enumtype);

	/*Apply ContrainMin: */
	if (input){
		norm=input->InfinityNorm();
	}
	else{
		norm=0;
	}

	/*Return output*/
	return norm;
}
/*}}}*/
/*FUNCTION Inputs::Max{{{1*/
double Inputs::Max(int enumtype){

	/*Output*/
	double max;

	/*Get input*/
	Input* input=(Input*)this->GetInput(enumtype);

	/*Apply ContrainMin: */
	if (input){
		max=input->Max();
	}
	else{
		_error_("Input %s not found",EnumToStringx(enumtype));
	}

	/*Return output*/
	return max;
}
/*}}}*/
/*FUNCTION Inputs::MaxAbs{{{1*/
double Inputs::MaxAbs(int enumtype){

	/*Output*/
	double max;

	/*Get input*/
	Input* input=(Input*)this->GetInput(enumtype);

	/*Apply ContrainMin: */
	if (input){
		max=input->MaxAbs();
	}
	else{
		_error_("Input %s not found",EnumToStringx(enumtype));
	}

	/*Return output*/
	return max;
}
/*}}}*/
/*FUNCTION Inputs::Min{{{1*/
double Inputs::Min(int enumtype){

	/*Output*/
	double min;

	/*Get input*/
	Input* input=(Input*)this->GetInput(enumtype);

	/*Apply ContrainMin: */
	if (input){
		min=input->Min();
	}
	else{
		_error_("Input %s not found",EnumToStringx(enumtype));
	}

	/*Return output*/
	return min;
}
/*}}}*/
/*FUNCTION Inputs::MinAbs{{{1*/
double Inputs::MinAbs(int enumtype){

	/*Output*/
	double min;

	/*Get input*/
	Input* input=(Input*)this->GetInput(enumtype);

	/*Apply ContrainMin: */
	if (input){
		min=input->MinAbs();
	}
	else{
		_error_("Input %s not found",EnumToStringx(enumtype));
	}

	/*Return output*/
	return min;
}
/*}}}*/
/*FUNCTION Inputs::GetInput{{{1*/
Input* Inputs::GetInput(int enum_name){

	vector<Object*>::iterator object;
	Input* input=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 

		if (input->EnumType()==enum_name){
			return input;
		}
	}
	return NULL;
}
/*}}}*/
/*FUNCTION Inputs::DeleteInput{{{1*/
int  Inputs::DeleteInput(int enum_type){

	vector<Object*>::iterator object;
	Input* input=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		input=(Input*)(*object); 

		if (input->EnumType()==enum_type){
			this->DeleteObject(input);
			break;
		}
	}

	return 1;

}
/*}}}*/
/*FUNCTION Inputs::DuplicateInput{{{1*/
void  Inputs::DuplicateInput(int original_enum,int new_enum){

	Input* original=NULL;
	Input* copy=NULL;

	/*Make a copy of the original input: */
	original=(Input*)this->GetInput(original_enum);
	if(!original)_error_("could not find input with enum: %s",EnumToStringx(original_enum)); 
	copy=(Input*)original->copy();

	/*Change copy enum to reinitialized_enum: */
	copy->ChangeEnum(new_enum);

	/*Add copy into inputs, it will wipe off the one already there: */
	this->AddInput((Input*)copy);
}
/*}}}*/
/*FUNCTION Inputs::SpawnTriaInputs{{{1*/
Inputs* Inputs::SpawnTriaInputs(int* indices){

	/*Intermediary*/
	vector<Object*>::iterator object;
	Input* inputin=NULL;
	Input* inputout=NULL;

	/*Output*/
	Inputs* newinputs=new Inputs();

	/*Go through inputs and call Spawn function*/
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Create new input*/
		inputin=(Input*)(*object); 
		inputout=inputin->SpawnTriaInput(indices);

		/*Add input to new inputs*/
		newinputs->AddObject(inputout);
	}

	/*Assign output pointer*/
	return newinputs;
}
/*}}}*/
/*FUNCTION Inputs::AXPY{{{1*/
void  Inputs::AXPY(int YEnum, double scalar, int XEnum){
	   
	Input* xinput=NULL;
	Input* yinput=NULL;

	/*Find x and y inputs: */
	xinput=(Input*)this->GetInput(XEnum);
	yinput=(Input*)this->GetInput(YEnum);

	/*some checks: */
	if(!xinput) _error_(" input %s could not be found!",EnumToStringx(XEnum));
	if(!yinput) _error_(" input %s could not be found!",EnumToStringx(YEnum));

	/*Apply AXPY: */
	yinput->AXPY(xinput,scalar);
}
/*}}}*/
