/*! \file IoModel.cpp
 * \brief  file containing the methods that will help in processing the input data coming 
 * into ISSM, from Matlab, or through a binary file opened for reading.
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include <string.h>
#include "stdio.h"
#include "./objects.h"
#include "../shared/shared.h"
#include "../io/io.h"
#include "../include/include.h"

/*FUNCTION IoModel::IoModel(){{{1*/
IoModel::IoModel(){
	this->IoModelInit();
}
/*}}}*/
/*FUNCTION IoModel::~IoModel(){{{1*/
IoModel::~IoModel(){
	
	int i;

	/*!Two cases here: 
	 * - serial mode: matlab's memory manager will take care of delete iomodel when returning from Imp. Do nothing here, so as not to confuse 
	 *                the memory manager.
     * - in parallel, anything the io layer does (FetchData) did needs to be erased explicitely in the iomodel.
	 */

	#ifdef _PARALLEL_
	xfree((void**)&this->x);
	xfree((void**)&this->y);
	xfree((void**)&this->z);
	xfree((void**)&this->elements);
	xfree((void**)&this->elementconnectivity);
	xfree((void**)&this->elements_type);
	xfree((void**)&this->vertices_type);
	xfree((void**)&this->gridonhutter);
	xfree((void**)&this->gridonmacayeal);
	if (this->dim==3){
		xfree((void**)&this->elements2d);
		xfree((void**)&this->upperelements);
		xfree((void**)&this->lowerelements);
		xfree((void**)&this->gridonpattyn);
	}
	xfree((void**)&this->elementonbed);
	xfree((void**)&this->elementonsurface);
	xfree((void**)&this->gridonbed);
	xfree((void**)&this->gridonsurface);
	xfree((void**)&this->gridonstokes);
	xfree((void**)&this->borderstokes);
	xfree((void**)&this->thickness_obs);
	xfree((void**)&this->thickness_coeff);
	xfree((void**)&this->thickness);
	xfree((void**)&this->surface);
	xfree((void**)&this->bed);
	xfree((void**)&this->bathymetry);
	xfree((void**)&this->vx_obs);
	xfree((void**)&this->vy_obs);
	xfree((void**)&this->vz_obs);
	xfree((void**)&this->vx);
	xfree((void**)&this->vy);
	xfree((void**)&this->vz);
	xfree((void**)&this->pressure);
	xfree((void**)&this->temperature);
	xfree((void**)&this->drag_coefficient);
	xfree((void**)&this->drag_p);
	xfree((void**)&this->drag_q);
	xfree((void**)&this->elementoniceshelf);
	xfree((void**)&this->elementonwater);
	xfree((void**)&this->gridonicesheet);
	xfree((void**)&this->gridoniceshelf);
	xfree((void**)&this->gridonwater);
	xfree((void**)&this->pressureload);
	xfree((void**)&this->spcvelocity);
	xfree((void**)&this->spcthickness);
	xfree((void**)&this->spcwatercolumn);
	xfree((void**)&this->spctemperature);
	xfree((void**)&this->edges);
	xfree((void**)&this->geothermalflux);
	xfree((void**)&this->melting_rate);
	xfree((void**)&this->watercolumn);
	xfree((void**)&this->melting_rate_correction);
	xfree((void**)&this->accumulation_rate);
	xfree((void**)&this->dhdt);
	xfree((void**)&this->rheology_B);
	xfree((void**)&this->rheology_n);
	xfree((void**)&this->control_type);
	xfree((void**)&this->cm_responses);
	xfree((void**)&this->weights);
	xfree((void**)&this->cm_jump);
	xfree((void**)&this->cm_min);
	xfree((void**)&this->cm_max);
	xfree((void**)&this->optscal);
	xfree((void**)&this->maxiter);


	/*!Delete structure fields: */
	xfree((void**)&this->inputfilename);
	xfree((void**)&this->outputfilename);
	xfree((void**)&this->repository);
	xfree((void**)&this->name);
	xfree((void**)&this->riftinfo);
	xfree((void**)&this->penalties);
	
	/*exterior data: */
	xfree((void**)&this->my_elements);
	xfree((void**)&this->my_vertices);
	xfree((void**)&this->my_nodes);
	xfree((void**)&this->singlenodetoelementconnectivity);
	xfree((void**)&this->numbernodetoelementconnectivity);
	#endif
}
/*}}}*/
/*FUNCTION IoModel::IoModel(ConstDataHandle iomodel_handle){{{1*/
IoModel::IoModel(ConstDataHandle iomodel_handle){
	
	int i,j;
		
	/*First, initialize the structure: */
	this->IoModelInit();
	
	/*Get all the data that consists of scalars, integers and strings: */

	IoModelFetchData(&this->name,iomodel_handle,"name"); 
	IoModelFetchData(&this->inputfilename,iomodel_handle,"inputfilename"); 
	IoModelFetchData(&this->outputfilename,iomodel_handle,"outputfilename"); 
	IoModelFetchData(&this->qmu_analysis,iomodel_handle,"qmu_analysis"); 
	IoModelFetchData(&this->control_analysis,iomodel_handle,"control_analysis"); 
	IoModelFetchData(&this->dim,iomodel_handle,"dim");
	/*!Get numberofelements and numberofvertices: */
	IoModelFetchData(&this->numberofvertices,iomodel_handle,"numberofgrids");
	IoModelFetchData(&this->numberofelements,iomodel_handle,"numberofelements");
	/*!In case we are running 3d, we are going to need the collapsed and non-collapsed 2d meshes, from which the 3d mesh was extruded: */
	if (this->dim==3){
	
		/*!Deal with 2d mesh: */
		IoModelFetchData(&this->numberofelements2d,iomodel_handle,"numberofelements2d");
		IoModelFetchData(&this->numberofvertices2d,iomodel_handle,"numberofgrids2d");
		IoModelFetchData(&this->numlayers,iomodel_handle,"numlayers");
	}


	/*elements type: */
	IoModelFetchData(&this->ishutter,iomodel_handle,"ishutter");
	IoModelFetchData(&this->ismacayealpattyn,iomodel_handle,"ismacayealpattyn");
	IoModelFetchData(&this->isstokes,iomodel_handle,"isstokes");

	/*!Get drag_type, drag and p,q: */
	IoModelFetchData(&this->drag_type,iomodel_handle,"drag_type");

	/*!Get materials: */
	IoModelFetchData(&this->rho_water,iomodel_handle,"rho_water");
	IoModelFetchData(&this->rho_ice,iomodel_handle,"rho_ice");
	IoModelFetchData(&this->g,iomodel_handle,"g");

	/*Get control parameters: */
	IoModelFetchData(&this->num_control_type,iomodel_handle,"num_control_type"); 

	/*!Get solution parameters: */
	IoModelFetchData(&this->yts,iomodel_handle,"yts");
	IoModelFetchData(&this->meanvel,iomodel_handle,"meanvel");
	IoModelFetchData(&this->epsvel,iomodel_handle,"epsvel");
	IoModelFetchData(&this->verbose,iomodel_handle,"verbose_binary");
	IoModelFetchData(&this->output_frequency,iomodel_handle,"output_frequency");
	IoModelFetchData(&this->prognostic_DG,iomodel_handle,"prognostic_DG");
	IoModelFetchData(&this->nsteps,iomodel_handle,"nsteps");
	IoModelFetchData(&this->eps_cm,iomodel_handle,"eps_cm");
	IoModelFetchData(&this->tolx,iomodel_handle,"tolx");
	IoModelFetchData(&this->cm_noisedmp,iomodel_handle,"cm_noisedmp");
	IoModelFetchData(&this->cm_gradient,iomodel_handle,"cm_gradient");
	IoModelFetchData(&this->eps_res,iomodel_handle,"eps_res");
	IoModelFetchData(&this->eps_rel,iomodel_handle,"eps_rel");
	IoModelFetchData(&this->eps_abs,iomodel_handle,"eps_abs");
	IoModelFetchData(&this->max_nonlinear_iterations,iomodel_handle,"max_nonlinear_iterations");
	IoModelFetchData(&this->dt,iomodel_handle,"dt");
	IoModelFetchData(&this->ndt,iomodel_handle,"ndt");
	IoModelFetchData(&this->time_adapt,iomodel_handle,"time_adapt");
	IoModelFetchData(&this->cfl_coefficient,iomodel_handle,"cfl_coefficient");
	IoModelFetchData(&this->hydrostatic_adjustment,iomodel_handle,"hydrostatic_adjustment");
	IoModelFetchData(&this->penalty_offset,iomodel_handle,"penalty_offset");
	IoModelFetchData(&this->penalty_melting,iomodel_handle,"penalty_melting");
	IoModelFetchData(&this->penalty_lock,iomodel_handle,"penalty_lock");
	IoModelFetchData(&this->sparsity,iomodel_handle,"sparsity");
	IoModelFetchData(&this->connectivity,iomodel_handle,"connectivity");
	IoModelFetchData(&this->lowmem,iomodel_handle,"lowmem");
	IoModelFetchData(&this->viscosity_overshoot,iomodel_handle,"viscosity_overshoot");
	IoModelFetchData(&this->artdiff,iomodel_handle,"artificial_diffusivity");
	IoModelFetchData(&this->prognostic_DG,iomodel_handle,"prognostic_DG");
	IoModelFetchData(&this->stokesreconditioning,iomodel_handle,"stokesreconditioning");
	IoModelFetchData(&this->shelf_dampening,iomodel_handle,"shelf_dampening");
	IoModelFetchData(&this->waitonlock,iomodel_handle,"waitonlock");
	IoModelFetchData(&this->kff,iomodel_handle,"kff");
	IoModelFetchData(&this->gl_migration,iomodel_handle,"gl_migration");

	/*!Get thermal parameters: */
	IoModelFetchData(&this->beta,iomodel_handle,"beta");
	IoModelFetchData(&this->hydro_gamma,iomodel_handle,"hydro_gamma");
	IoModelFetchData(&this->hydro_kn,iomodel_handle,"hydro_kn");
	IoModelFetchData(&this->meltingpoint,iomodel_handle,"meltingpoint");
	IoModelFetchData(&this->latentheat,iomodel_handle,"latentheat");
	IoModelFetchData(&this->heatcapacity,iomodel_handle,"heatcapacity");
	IoModelFetchData(&this->thermalconductivity,iomodel_handle,"thermalconductivity");
	IoModelFetchData(&this->min_thermal_constraints,iomodel_handle,"min_thermal_constraints");
	IoModelFetchData(&this->min_mechanical_constraints,iomodel_handle,"min_mechanical_constraints");
	IoModelFetchData(&this->stabilize_constraints,iomodel_handle,"stabilize_constraints");
	IoModelFetchData(&this->mixed_layer_capacity,iomodel_handle,"mixed_layer_capacity");
	IoModelFetchData(&this->thermal_exchange_velocity,iomodel_handle,"thermal_exchange_velocity");
	IoModelFetchData(&this->melting_rate_correction_apply,iomodel_handle,"melting_rate_correction_apply");
	IoModelFetchData(&this->gl_melting_rate,iomodel_handle,"gl_melting_rate");
	IoModelFetchData(&this->rheology_law,iomodel_handle,"rheology_law");
	
	/*qmu: */
	if(this->qmu_analysis){
		IoModelFetchData(&this->numberofvariables,iomodel_handle,"numberofvariables");
		IoModelFetchData(&this->numvariabledescriptors,iomodel_handle,"numvariabledescriptors");
		IoModelFetchData(&this->numberofresponses,iomodel_handle,"numberofresponses");
		IoModelFetchData(&this->numresponsedescriptors,iomodel_handle,"numresponsedescriptors");
		IoModelFetchData(&this->qmu_npart,iomodel_handle,"npart");
		IoModelFetchData(&this->qmu_mass_flux_num_profiles,iomodel_handle,"qmu_mass_flux_num_profiles");
		IoModelFetchData(&this->qmu_save_femmodel,iomodel_handle,"qmu_save_femmodel"); 
	}

	/*i/o: */
	IoModelFetchData(&this->io_gather,iomodel_handle,"io_gather");
	
	/*parameter output : */
	IoModelFetchData(&this->numoutput,iomodel_handle,"numoutput");
}
/*}}}*/
/*FUNCTION IoModel::IoModelInit{{{1*/
void IoModel::IoModelInit(void){
	
	/*!initialize all pointers to 0: */
	this->name=NULL;
	this->inputfilename=NULL;
	this->outputfilename=NULL;
	this->repository=NULL;
	this->qmu_analysis=0;
	this->control_analysis=0;
	this->numberofvariables=0;
	this->numvariabledescriptors=0;
	this->numberofresponses=0;
	this->numresponsedescriptors=0;
	this->qmu_npart=0; 
	this->numberofelements=0;
	this->numberofvertices=0;
	this->x=NULL; 
	this->y=NULL;
	this->z=NULL;
	this->elements=NULL;
	this->elementconnectivity=NULL;
	this->elements_type=NULL;
	this->vertices_type=NULL;
	this->numberofvertices2d=0;
	this->elements2d=NULL;
	this->numlayers=0;
	this->upperelements=NULL;
	this->lowerelements=NULL;
	this->gridonhutter=NULL;
	this->gridonmacayeal=NULL;
	this->gridonpattyn=NULL;
	this->io_gather=1;
	
	this->vx_obs=NULL;
	this->vy_obs=NULL;
	this->vz_obs=NULL;
	this->vx=NULL;
	this->vy=NULL;
	this->vz=NULL;
	this->pressure=NULL;
	this->temperature=NULL;
	this->gl_melting_rate=0;
	this->melting_rate=NULL;
	this->watercolumn=NULL;
	this->melting_rate_correction=NULL;
	this->melting_rate_correction_apply=0;
	this->geothermalflux=NULL;
	this->elementonbed=NULL;
	this->elementonsurface=NULL;
	this->gridonbed=NULL;
	this->gridonsurface=NULL;
	this->gridonstokes=NULL;
	this->borderstokes=NULL;
	this->thickness_obs=NULL;
	this->thickness_coeff=NULL;
	this->thickness=NULL;
	this->surface=NULL;
	this->bed=NULL;
	this->bathymetry=NULL;
	this->elementoniceshelf=NULL;
	this->elementonwater=NULL;
	this->gridonicesheet=NULL;
	this->gridoniceshelf=NULL;
	this->gridonwater=NULL;

	this->drag_type=0;
	this->drag_coefficient=NULL;
	this->drag_p=NULL;
	this->drag_q=NULL;
	
	
	this->numberofpressureloads=0;
	this->pressureload=NULL;
	this-> spcvelocity=NULL;
	this-> spctemperature=NULL;
	this-> spcthickness=NULL;
	this-> spcwatercolumn=NULL;
	this->numberofedges=0;
	this->edges=NULL;
	
	/*!materials: */
	this->rho_water=0;
	this->rho_ice=0;
	this->g=0;
	this->rheology_n=NULL;
	this->rheology_B=NULL;
	this->rheology_law=0;

	/*!solution parameters: */
	this->control_type=NULL;
	this->cm_responses=NULL;
	this->weights=NULL;
	this->cm_jump=NULL;
	this->meanvel=0;
	this->epsvel=0;
	this->nsteps=0;
	this->eps_cm=0;
	this->tolx=0;
	this->maxiter=NULL;
	this->cm_noisedmp=0;
	this->cm_min=NULL;
	this->cm_max=NULL;
	this->cm_gradient=0;
	this->verbose=0;
	this->output_frequency=0;
	this->eps_res=0;
	this->eps_rel=0;
	this->eps_abs=0;
	this->max_nonlinear_iterations=0;
	this->dt=0;
	this->ndt=0;
	this->time_adapt=0;
	this->cfl_coefficient=0;
	this->hydrostatic_adjustment=0;
	this->gl_migration=0;
	this->penalty_offset=0;
	this->penalty_melting=0;
	this->penalty_lock=0;
	this->sparsity=0;
	this->connectivity=0;
	this->lowmem=0;
	this->kff=0;
	this->optscal=NULL;
	this->yts=0;
	this->viscosity_overshoot=0;
	this->artdiff=0;
	this->prognostic_DG=0;
	this->stokesreconditioning=0;
	this->shelf_dampening=0;
	this->waitonlock=0;

	/*!thermal parameters: */
	this->beta=0;
	this->meltingpoint=0;
	this->latentheat=0;
	this->heatcapacity=0;
	this->thermalconductivity=0;
	this->min_thermal_constraints=0;
	this->min_mechanical_constraints=0;
	this->stabilize_constraints=0;
	this->mixed_layer_capacity=0;
	this->thermal_exchange_velocity=0;

	
	this->numrifts=0;
	this->riftinfo=NULL;

	/*!penalties: */
	this->numpenalties=0;
	this->penalties=NULL;

	/*!basal: */
	this->accumulation_rate=NULL;
	this->dhdt=NULL;
	
	/*parameter output: */
	this->numoutput=0;

	/*elements type: */
	this->ishutter=0;
	this->ismacayealpattyn=0;
	this->isstokes=0;

	/*exterior data: */
	this->my_elements=NULL;
	this->my_vertices=NULL;
	this->my_nodes=NULL;
	this->singlenodetoelementconnectivity=NULL;
	this->numbernodetoelementconnectivity=NULL;
	this->nodecounter=0;
	this->loadcounter=0;
	this->constraintcounter=0;
}
/*}}}*/
/*FUNCTION IoModel::Echo{{{1*/
void IoModel::Echo(int which_part,int rank) {

	//which_part  determines what gets echoed, otherwise, we'll get too much output.
	//1-> penalties

	int i,j;

	if(which_part==1 && my_rank==rank && this->dim==3){
		printf("IoModel penalties: \n");
		printf("   number of penalties: %i\n",this->numpenalties);
		printf("   grids: \n");

		for(i=0;i<this->numpenalties;i++){
			for(j=0;j<this->numlayers;j++){
				printf("%i ",(int)*(this->penalties+this->numlayers*i+j));
			}
			printf("\n");
		}
	}
}
/*}}}*/
