function md=qmu(md,varargin)
%QMU - apply Quantification of Margins and Uncertainties techniques 
%      to a solution sequence (like diagnostic.m, progonstic.m, etc ...), 
%      using the Dakota software from Sandia.
%   Usage:
%      md=qmu(md,varargin)
%      where varargin is a lit of paired arguments. 
%      arguments can be: 'analysis_type': 'diagnostic','thermal','prognostic','transient'
%      arguments can be: 'sub_analysis_type': 'transient','steady','' (default if empty = 'steady')
%      arguments can be: 'package': 'macayeal','ice','cielo' (default if not specified = 'cielo')
%
%
%   Examples:
%      md=qmu(md,'analysis_type','diagnostic','package','cielo');
%      md=qmu(md,'analysis_type','control','package','ice');
%      md=qmu(md,'analysis_type','thermal','sub_analysis_type','transient','package','ice');
%      md=qmu(md,'analysis_type','thermal','sub_analysis_type','steady','package','cielo');
%      md=qmu(md,'analysis_type','thermal','package','cielo');
%
%   On top of these solution arguments (found also in solve.m), user can specify QMU 
%   specific arguments: 
%
%       qmudir:  any directory where to run the qmu analysis
%       qmufile: input file for Dakota
%       ivar: selection number for variables input (if several are specified in variables)
%       iresp: same thing for response functions
%       imethod: same thing for methods
%       iparams: same thing for params
%       overwrite: overwrite qmudir
%       outfiles: (John?)
%       rstfile: backup file name
%       rundakota: (John?)
%       runmpi: (John?)

%some checks on list of arguments
global ISSM_DIR;

%recover options
options=recover_qmu_options(md,varargin{:});

%add default options
options=process_qmu_options(options);

%recover some fields
md.analysis_type=options.analysis_type;
md.sub_analysis_type=options.sub_analysis_type;
package=options.package;

%Use package to set solution namespace
usenamespace(package);

if ~ismodelselfconsistent(md,package),
	error(' '); %previous error messages should explain what is going on.
end

displaystring(md.debug,'\n%s\n','launching solution sequence');


%first create temporary directory in which we will work
if exist(options.qmudir,'dir')
    if ~isfield(options,'overwrite')
        options.overwrite=input(['Overwrite existing ''' options.qmudir ''' directory? Y/N [N]: '], 's');
    end
    if strncmpi(options.overwrite,'y',1)
        system(['rm -rf ' options.qmudir]);
%    else
%        error('Existing ''%s'' directory not overwritten.',qmudir);
    end
end
mkdir(options.qmudir)
cd(options.qmudir)

%some data needs to be copied in qmudir when not running in library mode
if ~md.dakotaplugin,
	system('cp $ISSM_DIR/startup.m .');

	%save our model in qmu so that it can be repeatedly used by Dakota.
	save('Qmu.model','md')
end

%when running in library mode, the in file needs to be called md.name.qmu.in
if md.dakotaplugin,
	options.qmufile=[md.name ];
end

%create m and in files for dakota
if (~isfield(md.qmu_params(options.iparams),'direct') || ...
    ~md.qmu_params(options.iparams).direct) && ...
   (~isfield(md.qmu_params(options.iparams),'analysis_driver') || ...
    isempty(md.qmu_params(options.iparams).analysis_driver))
    md.qmu_params(options.iparams).analysis_driver=[ISSM_DIR '/src/m/solutions/dakota/cielo_ice_script.sh'];
end

%create m and in files for dakota
dakota_in_data(md.qmu_method(options.imethod),md.variables(options.ivar),md.responses(options.iresp),md.qmu_params(options.iparams),options.qmufile,package,md);


%in library mode, we only need the dakota in file
if md.dakotaplugin,
	system(['rm -rf ' md.name '.m']);
end

%Dakota can be run in library mode in parallel, or using the 
%standard Dakota (scripted or direct) interface. Those two 
%modes of running are very different, and call for different 
%solution sequences. We branch here. 

if md.dakotaplugin,

	%Check we are running in parallel.
	if strcmpi(md.cluster,'none'),
		error('qmu error message: cannot run in library mode serially! set dakotaplugin to 0');
	end

	%Get cluster.rc location
	cluster_rc_location=which('cluster.rc');

	%let solution sequences know that we are running in qmu mode
	md.qmu_analysis=1;

	%Figure out parameters for this particular cluster
	[codepath,executionpath]=ClusterParameters(md.cluster,cluster_rc_location);

	%Marshall model data into a binary file.
	marshall(md);

	%add qmu fields to binary file
	qmumarshall(md,md.variables(options.ivar),md.responses(options.iresp));

	%Now, we need to build the queuing script, used by the cluster to launch the job.
	BuildQueueingScript(md,executionpath,codepath);

	%Now, launch the queueing script
	LaunchQueueJob(md,executionpath);

	%Do we return, or just wait for results?
	if md.waitonlock,
		%we wait for the done file
		waitonlock([executionpath '/' md.name '.lock']);
		%load results
		md=loadresultsfromcluster(md);
	else
		return;
	end
	
else
%  check for existence of results.out files to use
	if exist('results.out.1','file') || exist('results.out.zip','file')
		if ~isfield(options,'outfiles')
			options.outfiles=input(['Use existing ''results.out'' files? Y/N [N]: '], 's');
		end
		if ~strncmpi(options.outfiles,'y',1)
			system('rm -f results.out.[1-9]*');
		else
			if exist('results.out.zip','file') && ~exist('results.out.1','file')
				display('Inflating ''results.out.zip'' file.');
				system('unzip -q results.out.zip');
			end
		end
	end

	%  check for existence of dakota.rst file to use
	rstflag='';
	if (~isfield(options,'outfiles') || ~strncmpi(options.outfiles,'y',1)) && ...
		exist('dakota.rst','file')
		if ~isfield(options,'rstfile')
			options.rstfiles=input(['Use existing ''dakota.rst'' file? Y/N [N]: '], 's');
		end
		if strncmpi(options.rstfiles,'y',1)
			system('rm -f results.out.[1-9]*');
			system('dakota_restart_util print dakota.rst | grep completed');
			rstflag=' -read_restart dakota.rst';
		end
	end

	%call dakota
	if ~isfield(options,'rundakota')
		options.rundakota=input(['Run Dakota analysis ''' qmufile '''? Y/N [N]: '], 's');
	end
	if ~strncmpi(options.rundakota,'y',1)
		cd ..
		return
	end

	if ~options.runmpi
		system(['dakota -i ' qmufile '.in -o ' qmufile '.out -e ' qmufile '.err' rstflag]);
	else
	%  use 'mpd --ncpus=8 &' to initialize mpi and 'mpdringtest' to verify.
	%  exporting MPIRUN_NPROCS sets mpi in dakota.
	%    system('mpd --ncpus=8 &');
	%    system('mpdringtest');
		system(['export MPIRUN_NPROCS=8;mpirun -np 4 dakota -i ' qmufile '.in -o ' qmufile '.out -e ' qmufile '.err' rstflag]);
	end

end



%Now load the results from dakota output files.
qmuoutput(md,options.qmufile,options.qmudir);

%Get out of qmu directory
cd ..

