/*!\file: transient_2d_core.cpp
 * \brief: core of the transient_2d solution 
 */ 

#include "../toolkits/toolkits.h"
#include "../objects/objects.h"
#include "../shared/shared.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "./solutions.h"
#include "../modules/modules.h"
#include <float.h>

void transient2d_core(FemModel* femmodel){

	int i;

	/*parameters: */
	double finaltime;
	double dt                 ,yts;
	int    dim                  = -1;
	int    solution_type;
	bool   control_analysis;
	bool   time_adapt=false;
	int    output_frequency;
	int    gl_migration;

	/*intermediary: */
	int    step;
	double time;

	/* recover parameters: */
	femmodel->parameters->FindParam(&dim,DimEnum);
	femmodel->parameters->FindParam(&finaltime,NdtEnum);
	femmodel->parameters->FindParam(&dt,DtEnum);
	femmodel->parameters->FindParam(&yts,YtsEnum);
	femmodel->parameters->FindParam(&control_analysis,ControlAnalysisEnum);
	femmodel->parameters->FindParam(&solution_type,SolutionTypeEnum);
	femmodel->parameters->FindParam(&output_frequency,OutputFrequencyEnum);
	femmodel->parameters->FindParam(&time_adapt,TimeAdaptEnum);
	femmodel->parameters->FindParam(&gl_migration,GroundingLineMigrationEnum);

	/*initialize: */
	step=0;
	time=0;

	while(time < finaltime - (yts*DBL_EPSILON)){ //make sure we run up to finaltime.
	
		/*Increment*/
		if(time_adapt){
			TimeAdaptx(&dt,femmodel->elements, femmodel->nodes,femmodel->vertices,femmodel->loads, femmodel->materials, femmodel->parameters); 
			if(time+dt>finaltime) dt=finaltime-time;
			femmodel->parameters->SetParam(dt,DtEnum);
		}
		time+=dt;
		step+=1;

		_printf_(VerboseSolution(),"%s%g%s%i%s%g%s%g\n","time [yr]: ",time/yts,"    iteration number: ",step,"/",floor((finaltime-time)/dt)," dt [yr]: ",dt/yts);

		_printf_(VerboseSolution(),"%s\n","   computing new velocity");
		diagnostic_core(femmodel);

		_printf_(VerboseSolution(),"%s\n","   computing new thickness");
		prognostic_core(femmodel);

		if (gl_migration!=NoneEnum){
			_printf_(VerboseSolution(),"%s\n","   computing new grounding line position");
			GroundingLineMigrationx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters);
		}

		if(solution_type==Transient2DSolutionEnum && !control_analysis && (step%output_frequency==0 || time==finaltime)){
			_printf_(VerboseSolution(),"%s\n","   saving results\n");
			InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,VxEnum,step,time); 
			InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,VyEnum,step,time);
			InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,VelEnum,step,time);
			if(dim==3) InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,VzEnum,step,time);
			InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,PressureEnum,step,time);
			InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,ThicknessEnum,step,time);
			InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,SurfaceEnum,step,time);
			InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,BedEnum,step,time);
			if(gl_migration!=NoneEnum)InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,ElementOnIceShelfEnum,step,time);

			/*unload results*/
			_printf_(VerboseSolution(),"%s","   saving temporary results");
			OutputResultsx(femmodel->elements, femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,&femmodel->results,step,time);
		}
	}

}
