/*\file FetchData.cpp:
 * \brief: general I/O interface to fetch data.
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "./io.h"
#include "../shared/shared.h"
#include "../include/include.h"

#ifdef _SERIAL_
#include <mex.h>
/*FUNCTION FetchData(DataSet** pdataset,const mxArray* dataref){{{1*/
void FetchData(DataSet** pdataset,const mxArray* dataref){

	/*output*/
	DataSet* outdataset=NULL;
	char*    outdataset_buffer=NULL;
	int      outdataset_size;

	/*First, check that our reference is a double, otherwise, error out: */
	if (mxIsClass(dataref,"double")){
			
		/*We need to copy the data pointed by dataref, so that our dataset is not actually a pointer!:*/
		if (!mxIsEmpty(dataref)){
			outdataset_buffer=(char*)mxGetPr(dataref);
			outdataset_size=mxGetM(dataref)*mxGetN(dataref);
			if(outdataset_size)outdataset=DataSetDemarshall(outdataset_buffer);
		}
	}
	else{
		if (mxIsEmpty(dataref)){
			/*Nothing to pick up. Just initialize pointer to NULL, and warn the server we are not uploading anything: */
			outdataset_size=0;
			outdataset=NULL;
		}
		else{
			/*This is an error: we don't have the correct input!: */
			_error_("Input parameter of class %s not supported yet",mxGetClassName(dataref));
		}
	}

	/*Assign output pointers:*/
	*pdataset=outdataset;
}
/*}}}*/
/*FUNCTION FetchData(double** pmatrix,int* pM,int *pN,const mxArray* dataref){{{1*/
void FetchData(double** pmatrix,int* pM,int *pN,const mxArray* dataref){

	double*  outmatrix=NULL;
	int      outmatrix_rows,outmatrix_cols;

	if(mxIsEmpty(dataref) ){
		/*Nothing to pick up. Just initialize matrix pointer to NULL: */
		outmatrix_rows=0;
		outmatrix_cols=0;
		outmatrix=NULL;
	}
	else if (mxIsClass(dataref,"double") ){

		/*Check dataref is not pointing to NaN: */
		if ( mxIsNaN(*(mxGetPr(dataref))) && (mxGetM(dataref)==1) && (mxGetN(dataref)==1) ){
			outmatrix_rows=0;
			outmatrix_cols=0;
			outmatrix=NULL;
		}
		else{

			/*Convert matlab matrix to double* matrix: */
			MatlabMatrixToDoubleMatrix(&outmatrix,&outmatrix_rows,&outmatrix_cols,dataref);
		}
	}
	else{
		/*This is an error: we don't have the correct input!: */
		_error_("Input parameter of class %s not supported yet",mxGetClassName(dataref));
	}
			
	/*Assign output pointers:*/
	*pmatrix=outmatrix;
	if (pM)*pM=outmatrix_rows;
	if (pN)*pN=outmatrix_cols;

}
/*}}}*/
/*FUNCTION FetchData(int** pmatrix,int* pM,int *pN,const mxArray* dataref){{{1*/
void FetchData(int** pmatrix,int* pM,int *pN,const mxArray* dataref){

	int     i,outmatrix_rows,outmatrix_cols;
	double *doublematrix=NULL;
	int    *outmatrix=NULL;

	if(mxIsEmpty(dataref) ){
		/*Nothing to pick up. Just initialize matrix pointer to NULL: */
		outmatrix_rows=0;
		outmatrix_cols=0;
		outmatrix=NULL;
	}
	else if (mxIsClass(dataref,"double") ){

		/*Check dataref is not pointing to NaN: */
		if ( mxIsNaN(*(mxGetPr(dataref))) && (mxGetM(dataref)==1) && (mxGetN(dataref)==1) ){
			outmatrix_rows=0;
			outmatrix_cols=0;
			outmatrix=NULL;
		}
		else{

			/*Convert matlab matrix to double* matrix: */
			MatlabMatrixToDoubleMatrix(&doublematrix,&outmatrix_rows,&outmatrix_cols,dataref);

			/*Convert double matrix into integer matrix: */
			outmatrix=(int*)xmalloc(outmatrix_rows*outmatrix_cols*sizeof(int));
			for(i=0;i<outmatrix_rows*outmatrix_cols;i++)outmatrix[i]=(int)doublematrix[i];
		}
	}
	else{
		/*This is an error: we don't have the correct input!: */
		_error_("Input parameter of class %s not supported yet",mxGetClassName(dataref));
	}

	/*Assign output pointers:*/
	*pmatrix=outmatrix;
	if (pM)*pM=outmatrix_rows;
	if (pN)*pN=outmatrix_cols;
}
/*}}}*/
/*FUNCTION FetchData(bool** pmatrix,int* pM,int *pN,const mxArray* dataref){{{1*/
void FetchData(bool** pmatrix,int* pM,int *pN,const mxArray* dataref){

	int     i,outmatrix_rows,outmatrix_cols;
	double *doublematrix=NULL;
	bool   *outmatrix=NULL;

	if(mxIsEmpty(dataref) ){
		/*Nothing to pick up. Just initialize matrix pointer to NULL: */
		outmatrix_rows=0;
		outmatrix_cols=0;
		outmatrix=NULL;
	}
	else if (mxIsClass(dataref,"double") ){

		/*Check dataref is not pointing to NaN: */
		if ( mxIsNaN(*(mxGetPr(dataref))) && (mxGetM(dataref)==1) && (mxGetN(dataref)==1) ){
			outmatrix_rows=0;
			outmatrix_cols=0;
			outmatrix=NULL;
		}
		else{

			/*Convert matlab matrix to double* matrix: */
			MatlabMatrixToDoubleMatrix(&doublematrix,&outmatrix_rows,&outmatrix_cols,dataref);

			/*Convert double matrix into integer matrix: */
			outmatrix=(bool*)xmalloc(outmatrix_rows*outmatrix_cols*sizeof(bool));
			for(i=0;i<outmatrix_rows;i++)outmatrix[i]=(bool)doublematrix[i];
		}
	}
	else{
		/*This is an error: we don't have the correct input!: */
		_error_("Input parameter of class %s not supported yet",mxGetClassName(dataref));
	}

	/*Assign output pointers:*/
	*pmatrix=outmatrix;
	if (pM)*pM=outmatrix_rows;
	if (pN)*pN=outmatrix_cols;
}
/*}}}*/
/*FUNCTION FetchData(Mat* pmatrix,const mxArray* dataref){{{1*/
void FetchData(Mat* pmatrix,const mxArray* dataref){
	
	Mat outmatrix=NULL;
	int dummy=0;

	if (mxIsClass(dataref,"double") ){

		/*Check dataref is not pointing to NaN: */
		if ( mxIsNaN(*(mxGetPr(dataref))) && (mxGetM(dataref)==1) && (mxGetN(dataref)==1) ){
			outmatrix=NULL;
		}
		else{

			/*Convert matlab matrix to petsc matrix: */
			MatlabMatrixToPetscMatrix(&outmatrix,&dummy,&dummy,dataref);
		}
	}
	else{
		/*This is an error: we don't have the correct input!: */
		_error_("Input parameter of class %s not supported yet",mxGetClassName(dataref));
	}

	/*Assign output pointers:*/
	*pmatrix=outmatrix;
}
/*}}}*/
/*FUNCTION FetchData(double** pvector,int* pM,const mxArray* dataref){{{1*/
void FetchData(double** pvector,int* pM,const mxArray* dataref){

	double* outvector=NULL;
	int outvector_rows;

	if(mxIsEmpty(dataref)){
		/*Nothing to pick up. Just initialize matrix pointer to NULL: */
		outvector_rows=0;
		outvector=NULL;
	}
	else if (mxIsClass(dataref,"double") ){

		/*Convert matlab vector to double*  vector: */
		MatlabVectorToDoubleVector(&outvector,&outvector_rows,dataref);

	}
	else{
		/*This is an error: we don't have the correct input!: */
		_error_("Input parameter of class %s not supported yet",mxGetClassName(dataref));
	}

	/*Assign output pointers:*/
	*pvector=outvector;
	if (pM)*pM=outvector_rows;
}
/*}}}*/
/*FUNCTION FetchData(int** pvector,int* pM,const mxArray* dataref){{{1*/
void FetchData(int** pvector,int* pM,const mxArray* dataref){

	int    i;
	double *doublevector   = NULL;
	int    *outvector      = NULL;
	int     outvector_rows;

	if(mxIsEmpty(dataref)){
		/*Nothing to pick up. Just initialize matrix pointer to NULL: */
		outvector_rows=0;
		outvector=NULL;
	}
	else if (mxIsClass(dataref,"double") ){

		/*Convert matlab vector to double*  vector: */
		MatlabVectorToDoubleVector(&doublevector,&outvector_rows,dataref);

		/*Convert double vector into integer vector: */
		outvector=(int*)xmalloc(outvector_rows*sizeof(int));
		for(i=0;i<outvector_rows;i++)outvector[i]=(int)doublevector[i];
	}
	else{
		/*This is an error: we don't have the correct input!: */
		_error_("Input parameter of class %s not supported yet",mxGetClassName(dataref));
	}

	/*Assign output pointers:*/
	*pvector=outvector;
	if (pM)*pM=outvector_rows;
}
/*}}}*/
/*FUNCTION FetchData(bool** pvector,int* pM,const mxArray* dataref){{{1*/
void FetchData(bool** pvector,int* pM,const mxArray* dataref){

	int    i;
	double *doublevector   = NULL;
	bool   *outvector      = NULL;
	int     outvector_rows;

	if(mxIsEmpty(dataref)){
		/*Nothing to pick up. Just initialize matrix pointer to NULL: */
		outvector_rows=0;
		outvector=NULL;
	}
	else if (mxIsClass(dataref,"double") ){

		/*Convert matlab vector to double*  vector: */
		MatlabVectorToDoubleVector(&doublevector,&outvector_rows,dataref);

		/*Convert double vector into integer vector: */
		outvector=(bool*)xmalloc(outvector_rows*sizeof(bool));
		for(i=0;i<outvector_rows;i++)outvector[i]=(bool)doublevector[i];
	}
	else{
		/*This is an error: we don't have the correct input!: */
		_error_("Input parameter of class %s not supported yet",mxGetClassName(dataref));
	}

	/*Assign output pointers:*/
	*pvector=outvector;
	if (pM)*pM=outvector_rows;
}
/*}}}*/
/*FUNCTION FetchData(float** pvector,int* pM,const mxArray* dataref){{{1*/
void FetchData(float** pvector,int* pM,const mxArray* dataref){

	int    i;
	double *doublevector   = NULL;
	float  *outvector      = NULL;
	int     outvector_rows;

	if(mxIsEmpty(dataref)){
		/*Nothing to pick up. Just initialize matrix pointer to NULL: */
		outvector_rows=0;
		outvector=NULL;
	}
	else if (mxIsClass(dataref,"double") ){

		/*Convert matlab vector to double*  vector: */
		MatlabVectorToDoubleVector(&doublevector,&outvector_rows,dataref);

		/*Convert double vector into float vector: */
		outvector=(float*)xmalloc(outvector_rows*sizeof(float));
		for(i=0;i<outvector_rows;i++)outvector[i]=(float)doublevector[i];
	}
	else{
		/*This is an error: we don't have the correct input!: */
		_error_("Input parameter of class %s not supported yet",mxGetClassName(dataref));
	}

	/*Assign output pointers:*/
	*pvector=outvector;
	if (pM)*pM=outvector_rows;
}
/*}}}*/
/*FUNCTION FetchData(Vec* pvector,const mxArray* dataref){{{1*/
void FetchData(Vec* pvector,const mxArray* dataref){

	Vec vector=NULL;
	int dummy;

	if(mxIsEmpty(dataref)){
		/*Nothing to pick up. Just initialize matrix pointer to NULL: */
		vector=NULL;
	}
	else if (mxIsClass(dataref,"double") ){

		/*Convert matlab vector to petsc vector: */
		MatlabVectorToPetscVector(&vector,&dummy,dataref);
	}
	else{
		/*This is an error: we don't have the correct input!: */
		_error_("Input parameter of class %s not supported yet",mxGetClassName(dataref));
	}

	/*Assign output pointers:*/
	*pvector=vector;
}
/*}}}*/
/*FUNCTION FetchData(char** pstring,const mxArray* dataref){{{1*/
void FetchData(char** pstring,const mxArray* dataref){

	char* outstring=NULL;


	/*Ok, the string should be coming directly from the matlab workspace: */
	if (!mxIsClass(dataref,"char")){
		_error_("input data_type is not a string!");
	}
	else{
		/*Recover the string:*/
		int stringlen;
		
		stringlen = mxGetM(dataref)*mxGetN(dataref)+1;
		outstring = (char*)xmalloc(sizeof(mxChar)*stringlen);
		mxGetString(dataref,outstring,stringlen);
	}

	/*Assign output pointers:*/
	*pstring=outstring;
}
/*}}}*/
/*FUNCTION FetchData(double* pscalar,const mxArray* dataref){{{1*/
void FetchData(double* pscalar,const mxArray* dataref){

	double scalar;

	if (!mxIsClass(dataref,"double")){
		_error_("input data_type is not a double!");
	}
	else{
		/*Recover the double: */
		scalar=mxGetScalar(dataref);
	}

	/*Assign output pointers:*/
	*pscalar=scalar;
}
/*}}}*/
/*FUNCTION FetchData(int* pinteger,const mxArray* dataref){{{1*/
void FetchData(int* pinteger,const mxArray* dataref){

	int integer;

	if (!mxIsClass(dataref,"double")){
		_error_("input data_type is not a scalar!");
	}
	else{
		/*Recover the double: */
		integer=(int)mxGetScalar(dataref);
	}

	/*Assign output pointers:*/
	*pinteger=integer;
}
/*}}}*/
/*FUNCTION FetchData(bool* pboolean,const mxArray* dataref){{{1*/
void FetchData(bool* pboolean,const mxArray* dataref){

	bool* mxbool_ptr=NULL;

	if (mxIsClass(dataref,"logical")){
		if(mxGetM(dataref)!=1) _error_("input data is not of size 1x1");
		if(mxGetN(dataref)!=1) _error_("input data is not of size 1x1");
		mxbool_ptr=mxGetLogicals(dataref);
	}
	else{
		_error_("input data_type is not a bool!");
	}

	*pboolean=*mxbool_ptr;
}
/*}}}*/
#endif

#if defined(_PARALLEL_) && defined(_HAVE_PETSC_)
/*FUNCTION FetchData(double** pmatrix, int* pM,int* pN,FILE* fid){{{1*/
void FetchData(double** pmatrix, int* pM,int* pN,FILE* fid){

	extern int my_rank;
	extern int num_procs;

	/*output: */
	int M,N;
	double* matrix=NULL;
	
	/*We have to read a matrix from disk. First read the dimensions of the matrix, then the whole matrix: */
	/*numberofelements: */
	if(my_rank==0){  
		if(fread(&M,sizeof(int),1,fid)!=1) _error_("could not read number of rows for matrix ");
	}

	MPI_Bcast(&M,1,MPI_INT,0,MPI_COMM_WORLD); 

	if(my_rank==0){  
		if(fread(&N,sizeof(int),1,fid)!=1) _error_("could not read number of columns for matrix ");
	}
	MPI_Bcast(&N,1,MPI_INT,0,MPI_COMM_WORLD); 

	/*Now allocate matrix: */
	if(M*N){
		matrix=(double*)xmalloc(M*N*sizeof(double));

		/*Read matrix on node 0, then broadcast: */
		if(my_rank==0){  
			if(fread(matrix,M*N*sizeof(double),1,fid)!=1) _error_("could not read matrix ");
		}
		
		MPI_Bcast(matrix,M*N,MPI_DOUBLE,0,MPI_COMM_WORLD); 
	}


	/*Assign output pointers: */
	*pmatrix=matrix;
	if (pM)*pM=M;
	if (pN)*pN=N;

}
/*}}}*/
/*FUNCTION FetchData(char** pstring,FILE* fid){{{1*/
void FetchData(char** pstring,FILE* fid){

	extern int my_rank;
	extern int num_procs;

	/*output: */
	char* string=NULL;
	int   string_size;

	/*We have to read a string from disk. First read the dimensions of the string, then the string: */
	if(my_rank==0){  
		if(fread(&string_size,sizeof(int),1,fid)!=1) _error_(" could not read length of string ");
	}

	MPI_Bcast(&string_size,1,MPI_INT,0,MPI_COMM_WORLD); 

	/*Now allocate string: */
	if(string_size){
		string=(char*)xmalloc((string_size+1)*sizeof(char));
		string[string_size]='\0';

		/*Read string on node 0, then broadcast: */
		if(my_rank==0){  
			if(fread(string,string_size*sizeof(char),1,fid)!=1)_error_("  could not read string ");
		}
		MPI_Bcast(string,string_size,MPI_CHAR,0,MPI_COMM_WORLD); 
	}
	else{
		string=(char*)xmalloc(sizeof(char));
		string[0]='\0';
	}


	/*Assign output pointers: */
	*pstring=string;
}
/*}}}*/
/*FUNCTION FetchData(double* pscalar,FILE* fid){{{1*/
void FetchData(double* pscalar,FILE* fid){

	extern int my_rank;
	extern int num_procs;

	/*output: */
	double   scalar;

	/*We have to read a scalar from disk. First read the dimensions of the scalar, then the scalar: */
	if(my_rank==0){
		if(fread(&scalar,sizeof(double),1,fid)!=1)_error_(" could not read scalar ");
	}
	MPI_Bcast(&scalar,1,MPI_DOUBLE,0,MPI_COMM_WORLD); 

	/*Assign output pointers: */
	*pscalar=scalar;
		 
}
/*}}}*/
/*FUNCTION FetchData(int* pinteger,FILE* fid){{{1*/
void FetchData(int* pinteger,FILE* fid){

	extern int my_rank;
	extern int num_procs;

	/*output: */
	int   integer;

	/*We have to read a integer from disk. First read the dimensions of the integer, then the integer: */
	if(my_rank==0){  
		if(fread(&integer,sizeof(int),1,fid)!=1) _error_(" could not read integer ");
	}

	MPI_Bcast(&integer,1,MPI_INT,0,MPI_COMM_WORLD); 

	/*Assign output pointers: */
	*pinteger=integer;

}
/*}}}*/
#endif
