/*\file KMLMeshWrite.c
 *\brief: KML mesh writer module.
 */
#include "./KMLMeshWrite.h"

void mexFunction( int nlhs, mxArray* plhs[], int nrhs, const mxArray* prhs[]){

	int i,j,nnodes=0,verbose=1;

	/*input: */
	char*   name=NULL;

	char*   notes=NULL;
	const mxArray* notesi;
	mwIndex        nindex;

	int*    elem=NULL;
	int     melem=0,nelem=0;

	int*    nodecon=NULL;
	int     mncon=0,nncon=0;

	double* lat=NULL;
	int     mlat=0,nlat=0,llat=0;

	double* lng=NULL;
	int     mlng=0,nlng=0,llng=0;

	int     nparts=0;

	int*    part=NULL;
	int     mprt=0,nprt=0,lprt=0;

	double* data=NULL;
	int     mdata=0,ndata=0;

	double* cmap=NULL;
	int     mcmap=0,ncmap=0;

	char*   filnam=NULL;

	FILE*   fid=NULL;

	Options* options=NULL;

	/* output: */
	int     ierror=0;

	/*Boot module: */
	MODULEBOOT();

	/*checks on arguments on the matlab side: */
	if (nlhs > NLHS) {
		KMLMeshWriteUsage();
		_error_("KMLMeshWrite usage error");
	}
	if (nrhs < NRHS) {
		KMLMeshWriteUsage();
		_error_("KMLMeshWrite usage error");
	}

	/*Input datasets: */
	if (verbose) printf("Fetching inputs:\n");
	FetchData(&name,NAME);
	if (verbose) printf("  name   =\"%s\"\n",name);

/*  notes is typically a cell array of character strings  */
	if (mxIsCell(NOTES)) {
		for (nindex=0; nindex<mxGetNumberOfElements(NOTES); nindex++) {
			notesi=mxGetCell(NOTES,nindex);
			if (notesi && mxIsChar(notesi) && mxGetNumberOfElements(notesi)) {
				if (!notes) {
					notes=(char *) xmalloc((mxGetNumberOfElements(notesi)+1)*sizeof(char));
					mxGetString(notesi,notes,mxGetNumberOfElements(notesi)+1);
				}
				else {
/*  note that strlen does not include trailing null  */
					notes=(char *) xrealloc(notes,(strlen(notes)+1+mxGetNumberOfElements(notesi)+1)*sizeof(char));
					strcat(notes,"\n");
					mxGetString(notesi,&notes[strlen(notes)],mxGetNumberOfElements(notesi)+1);
				}
			}
		}
	}
	else
		FetchData(&notes,NOTES);
	if (verbose) printf("  notes  =\"%s\"\n",notes);

	FetchData(&elem,&melem,&nelem,ELEMHANDLE);
	if (verbose) printf("  elem   =size [%d x %d]\n",melem,nelem);
	FetchData(&nodecon,&mncon,&nncon,NODECONHANDLE);
	if (verbose) printf("  nodecon=size [%d x %d]\n",mncon,nncon);
	FetchData(&lat,&mlat,&nlat,LATHANDLE);
	llat=mlat*nlat;
	if (verbose) printf("  lat    =length [%d]\n",llat);
	FetchData(&lng,&mlng,&nlng,LNGHANDLE);
	llng=mlng*nlng;
	if (verbose) printf("  lng    =length [%d]\n",llng);
	FetchData(&part,&mprt,&nprt,PARTHANDLE);
	lprt=mprt*nprt;
	if (verbose) printf("  part   =length [%d]\n",lprt);
	FetchData(&data,&mdata,&ndata,DATAHANDLE);
	if (verbose) printf("  data   =size [%d x %d]\n",mdata,ndata);
	FetchData(&cmap,&mcmap,&ncmap,CMAPHANDLE);
	if (verbose) printf("  cmap   =size [%d x %d]\n",mcmap,ncmap);
	FetchData(&filnam,FILENAME);
	if (verbose) printf("  filnam =\"%s\"\n",filnam);

	if (verbose) printf("Parsing options:\n");
	options=new Options(NRHS,nrhs,prhs);

	/*some checks*/
	if (verbose) printf("Checking inputs:\n");

	for (i=0; i<melem*nelem; i++)
		if (elem[i] > nnodes)
			nnodes=elem[i];
	if (verbose) printf("  nnodes =%d\n",nnodes);
	if (part)
		for (i=0; i<lprt; i++)
			if (part[i]+1 > nparts)
				nparts=part[i]+1;
	if (verbose) printf("  nparts =%d\n",nparts);

	if      (nodecon && (mncon != nnodes))
		_error_("Nodal connectivity table, if supplied, must be supplied for all nodes.");
	else if (!nodecon)
		mncon=nnodes;
	if ((llat != nnodes) || (llng != nnodes) || (llat != llng))
		_error_("Latitude and longitude vectors must be supplied for all nodes.");
	if (part && (lprt != nnodes))
		_error_("Partitioning vector, if supplied, must be supplied for all nodes.");
	if (data && !((mdata == nnodes) || (mdata == melem)))
		_error_("Data matrix, if supplied, must be supplied for all nodes or all elements.");
	if (cmap && (ncmap != 3))
		_error_("Colormap matrix, if supplied, must have three columns for rgb.");

	if (!strlen(filnam))
		strcpy(filnam,"stdout");

	if (verbose) printf("Opening file \"%s\".\n",filnam);
	fid=fopen(filnam,"w");

	/* Run core computations: */
	if (verbose) printf("Calling core:\n");
	KMLMeshWritex(&ierror,
				  name,
				  notes,
				  elem,melem,nelem,
				  nodecon,mncon,nncon,
				  lat,lng,
				  part,
				  data,mdata,ndata,
				  cmap,mcmap,ncmap,
				  fid);

	if (verbose) printf("Closing file \"%s\".\n",filnam);
	fclose(fid);

	/*Write data: */
	WriteData(ERRORFLAG,ierror);

	/*Clean-up*/
	delete options;
	if (mxIsCell(NOTES) && notes) xfree((void**)&notes);

	/*end module: */
	MODULEEND();
}

void KMLMeshWriteUsage(void)
{
	_printf_(true,"INTERFROMMESHTOMESH2D - interpolation from a 2d triangular mesh onto a list of point\n");
	_printf_(true,"\n");
	_printf_(true,"   This function is a multi-threaded mex file that interpolates a field\n");
	_printf_(true,"   defined on a triangular mesh onto a list of point\n");
	_printf_(true,"\n");
	_printf_(true,"   Usage:\n");
	_printf_(true,"         data_interp=InterpFromMeshToMesh2d(index,x,y,data,x_interp,y_interp);\n");
	_printf_(true,"      or data_interp=InterpFromMeshToMesh2d(index,x,y,data,x_interp,y_interp,default_value,contourname);\n");
	_printf_(true,"\n");
	_printf_(true,"      index: index of the mesh where data is defined\n");
	_printf_(true,"      x,y: coordinates of the nodes where data is defined\n");
	_printf_(true,"      data: matrix holding the data to be interpolated onto the mesh. (one column per field)\n");
	_printf_(true,"      x_interp,y_interp: coordinates of the points onto which we interpolate.\n");
	_printf_(true,"      if default_value and contourname not specified: linear interpolation will happen on all x_interp,y_interp.\n");
	_printf_(true,"      if (default_value,contourname) specified: linear interpolation will happen on all x_interp,y_interp inside the contour, default value will be adopted on the rest of the mesh.\n");
	_printf_(true,"      note that default_value is either a scalar, or a vector of size  length(x_interp)\n");
	_printf_(true,"      data_interp: vector of mesh interpolated data.\n");
	_printf_(true,"\n");
	_printf_(true,"   Example:\n");
	_printf_(true,"      load('temperature.mat');\n");
	_printf_(true,"      md.temperature=InterpFromMeshToMesh2d(index,x,y,temperature,md.x,md.y);\n");
	_printf_(true,"      md.temperature=InterpFromMeshToMesh2d(index,x,y,temperature,md.x,md.y,253,'Contour.exp');\n");
	_printf_(true,"\n");
}

