function md=thermal(md)
%THERMAL - thermal matlab driven solution
%
%   Usage:
%      md=thermal(md)

%start timing
t1=clock;

%define global variables
iceglobal

%First, build elements,grids,loads, etc ... for thermal and melting 
analysis_type=[md.analysis_type md.sub_analysis_type];
m_t=CreateFemModel(md,analysis_type);
if strcmpi(analysis_type,'thermalsteady'),
	m_m=CreateFemModel(md,'meltingsteady');
else
	m_m=CreateFemModel(md,'meltingtransient');
end
%initialize velocity,pressure and material properties
velocity=zeros(m_t.gridset.gsize,1);
velocity(1:6:m_t.gridset.gsize)=md.vx(m_t.part)/md.yts;
velocity(2:6:m_t.gridset.gsize)=md.vy(m_t.part)/md.yts;
velocity(3:6:m_t.gridset.gsize)=md.vz(m_t.part)/md.yts;

pressure=zeros(m_t.gridset.gsize,1);
pressure(1:6:m_t.gridset.gsize)=md.pressure;

if strcmpi(analysis_type,'thermalsteady'),

	%Call core thermal computation
	disp('   computing temperature...');
	gridset=m_t.gridset;
	inputs=struct('pressure',pressure,'velocity',velocity,'dt',md.dt);
	[t_g m_t.loads_t melting_offset]=thermal_core(m_t,analysis_type,inputs);
	
	%Call core melting computation
	disp('   computing melting...');
	gridset=m_m.gridset;
	inputs=struct('pressure',pressure,'temperature',t_g,'melting_offset',melting_offset);
	melting_g=melting_core(m_m,analysis_type,inputs);
	
	%Wrap up
	indx=1:6:gridset.gsize;
	indx=indx(m_t.tpart);
	if isnan(md.results), md.results=struct(); end
	md.results.thermal.step=1;
	md.results.thermal.time=0;
	md.results.thermal.temperature=t_g(indx);
	md.results.thermal.melting=melting_g(indx)*md.yts; %from m/s to m/a
else

	nsteps=md.ndt/md.dt;

	%initialize temperature and melting
	temperature=zeros(gridset.gsize,1);
	melting=zeros(gridset.gsize,1);
	temperature(1:6:gridset.gsize)=md.temperature;
	melting(1:6:gridset.gsize)=md.melting/md.yts; %in m/s
	soln.t_g=temperature;
	soln.melting_g=melting;
	soln.time=0;

	for n=1:nsteps, 

		soln(n+1).time=(n)*md.dt;
		disp(sprintf('\n%s%i/%i\n','time step: ',n,md.ndt/md.dt));

		%Call core thermal computation
		disp('   computing temperature...');
		gridset=m_t.gridset;
		inputs=struct('pressure',pressure,'temperature',soln(n).t_g,'velocity',velocity,'dt',md.dt);
		[soln(n+1).t_g m_t.loads_t  melting_offset]=thermal_core(m_t,analysis_type,inputs);

		%Call core melting computation
		disp('   computing melting...');
		gridset=m_m.gridset;
		inputs=struct('pressure',pressure,'temperature',soln(n).t_g,'melting_offset',melting_offset,'dt',md.dt);
		soln(n+1).melting_g=melting_core(m_m,analysis_type,inputs);
		
	end
	
	%Wrap up
	indx=1:6:m_t.gridset.gsize;
	indx=indx(m_t.tpart);
 	solution=struct('time',{},'temperature',{},'melting',{});

	for n=1:nsteps+1,
		solution(n).step=n;
		solution(n).temperature=soln(n).t_g(indx);
		solution(n).melting=soln(n).melting_g(indx)*md.yts; %in m/a
		solution(n).time=soln(n).time/md.yts; %in year
	end
	if isnan(md.results), md.results=struct(); end
	md.results.thermal=solution;
end

%stop timing
t2=clock;
disp(sprintf('\n%s\n',['solution converged in ' num2str(etime(t2,t1)) ' seconds']));
