/*!\file:  SpawnCore.cpp
 * \brief: run core solution, according to analysis_type and sub_analysis_type
 */ 

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#undef __FUNCT__ 
#define __FUNCT__ "SpawnCore"

#include "../objects/objects.h"
#include "../io/io.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../shared/shared.h"
#include "./parallel.h"
#include "../include/macros.h"

void SpawnCore(double* responses, double* variables, char** variables_descriptors,int numvariables, FemModel* femmodels,ParameterInputs* inputs,int analysis_type,int sub_analysis_type){

	int i;
	
	/*output from core solutions: */
	DataSet* results=NULL;

	char** responses_descriptors=NULL;
	int    numresponses;
	Param* param=NULL;
	char*  string=NULL;
	int    string_length;
	double* qmu_part=NULL;
	int     qmu_npart;

	extern int my_rank;
	
	
	/*synchronize all cpus, as CPU 0 is probably late (it is starting the entire dakota strategy!) : */
	MPI_Barrier(MPI_COMM_WORLD);

	/*First off, recover the response descriptors for the response functions: */
	param=(Param*)femmodels[0].parameters->FindParamObject("responsedescriptors");
	if(!param)throw ErrorException(__FUNCT__," could not find response descriptors!");

	numresponses=param->GetM();
	param->GetParameterValue(&responses_descriptors);

	/*Recover partitioning for dakota: */
	femmodels[0].parameters->FindParam((void*)&qmu_npart,"qmu_npart");
	femmodels[0].parameters->FindParam((void*)&qmu_part,"qmu_part");
	#ifdef _DEBUG_
	for(i=0;i<numresponses;i++){
		PetscSynchronizedPrintf(MPI_COMM_WORLD,"response descriptor %i: %s\n",i,responses_descriptors[i]);
		PetscSynchronizedFlush(MPI_COMM_WORLD);
	}
	#endif

	/*broadcast variables: only cpu 0 has correct values*/
	MPI_Bcast(&numvariables,1,MPI_INT,0,MPI_COMM_WORLD); 
	if(my_rank!=0)variables=(double*)xmalloc(numvariables*sizeof(double));
	MPI_Bcast(variables,numvariables,MPI_DOUBLE,0,MPI_COMM_WORLD); 

	#ifdef _DEBUG_
	for(i=0;i<numvariables;i++){
		PetscSynchronizedPrintf(MPI_COMM_WORLD,"variable %i: %g\n",i,variables[i]);
		PetscSynchronizedFlush(MPI_COMM_WORLD);
	}
	#endif

	/*broadcast variables_descriptors: */
	if(my_rank!=0){
		variables_descriptors=(char**)xmalloc(numvariables*sizeof(char*));
	}
	for(i=0;i<numvariables;i++){
		if(my_rank==0){
			string=variables_descriptors[i];
			string_length=(strlen(string)+1)*sizeof(char);
		}
		MPI_Bcast(&string_length,1,MPI_INT,0,MPI_COMM_WORLD); 
		if(my_rank!=0)string=(char*)xmalloc(string_length);
		MPI_Bcast(string,string_length,MPI_CHAR,0,MPI_COMM_WORLD); 
		if(my_rank!=0)variables_descriptors[i]=string;
	}

	#ifdef _DEBUG_
	for(i=0;i<numvariables;i++){
		PetscSynchronizedPrintf(MPI_COMM_WORLD,"variable descriptor %i: %s\n",i,variables_descriptors[i]);
		PetscSynchronizedFlush(MPI_COMM_WORLD);
	}
	#endif


	_printf_("initialize results:\n");
	results=new DataSet(ResultsEnum());

	/*Modify core inputs to reflect the dakota variables inputs: */
	inputs->UpdateFromDakota(variables,variables_descriptors,numvariables,femmodels[0].parameters,qmu_part,qmu_npart); //femmodel #0 is the one holding the parameters for Dakota.
	

	/*Run the analysis core solution sequence, with the updated inputs: */
	if(analysis_type==DiagnosticAnalysisEnum()){
			
		_printf_("Starting diagnostic core\n");
		diagnostic_core(results,femmodels,inputs);

	}
	else if(analysis_type==ThermalAnalysisEnum()){
		
		_printf_("Starting thermal core\n");
		thermal_core(results,femmodels,inputs);

	}
	else if(analysis_type==PrognosticAnalysisEnum()){

		_printf_("Starting prognostic core\n");
		prognostic_core(results,femmodels,inputs);

	}
	else if(analysis_type==TransientAnalysisEnum()){

		_printf_("Starting transient core\n");
		throw ErrorException(__FUNCT__,"not supported yet!");

	}
	else throw ErrorException(__FUNCT__,exprintf("%s%i%s%i%s"," analysis_type ",analysis_type," and sub_analysis_type ",sub_analysis_type," not supported yet!"));
	
		

	/*Now process the outputs, before computing the dakota responses: */
	_printf_("process results:\n");
	ProcessResults(&results,femmodels,analysis_type); 
	

	/*compute responses on cpu 0: dummy for now! */
	_printf_("compute dakota responses:\n");
	DakotaResponses(responses,responses_descriptors,numresponses,femmodels,results,analysis_type,sub_analysis_type);

	/*Free ressources:*/
	delete results;

	//variables only on cpu != 0
	if(my_rank!=0){
		xfree((void**)&variables);
		for(i=0;i<numvariables;i++){
			string=variables_descriptors[i];
			xfree((void**)&string);
		}
		xfree((void**)&variables_descriptors);
	}
	//responses descriptors
	for(i=0;i<numresponses;i++){
		string=responses_descriptors[i];
		xfree((void**)&string);
	}
	//rest of dynamic allocations.
	xfree((void**)&responses_descriptors);
	xfree((void**)&qmu_part);
}

