/*!\file:  InterpFromGridToMeshxt.cpp
 * \brief  thread core for InterpFromGridToMeshxt code
 */ 

#include "./InterpFromGridToMeshx.h"
#include "../../shared/shared.h"

void* InterpFromGridToMeshxt(void* vpthread_handle){

	/*gate variables :*/
	InterpFromGridToMeshxThreadStruct* gate=NULL;
	pthread_handle* handle=NULL;
	int     my_thread;
	int     num_threads;
	
	Vec     data_mesh=NULL;
	double* x_mesh=NULL;
	double* y_mesh=NULL;
	int     x_rows,y_rows;
	double* x=NULL;
	double* y=NULL;
	int     nods;
	double  default_value;
	double* data=NULL;
	int     M,N;


	/*intermediary: */
	int     i0;
	int     i1;
	int     i,m,n;
	double  x_grid;
	double  y_grid;
	double  area;
	double G1,G2,G3,G4,data_value;
	double area_1,area_2,area_3;
	double xi,eta;
	bool   triangle=true;


	/*recover handle and gate: */
	handle=(pthread_handle*)vpthread_handle;
	gate=(InterpFromGridToMeshxThreadStruct*)handle->gate;
	my_thread=handle->id;
	num_threads=handle->num;
	
	/*recover parameters :*/
	x_mesh=gate->x_mesh;
	y_mesh=gate->y_mesh;
	x_rows=gate->x_rows;
	y_rows=gate->y_rows;
	x=gate->x;
	y=gate->y;
	nods=gate->nods;
	data_mesh=gate->data_mesh;
	data=gate->data;
	default_value=gate->default_value;
	M=gate->M;
	N=gate->N;

	/*partition loop across threads: */
	PartitionRange(&i0,&i1,nods,num_threads,my_thread);

	for ( i=i0; i<i1; i++) {

		x_grid=*(x_mesh+i);
		y_grid=*(y_mesh+i);

		/*Find indices m and n into y and x, for which  y(m)<=y_grids<=y(m+1) and x(n)<=x_grid<=x(n+1)*/
		if(findindices(&n,&m,x,x_rows, y,y_rows, x_grid,y_grid)){
			
			if(triangle){
				/*Linear (triangle) interpolation: {{{1*/
				/*Get area*/
				area=(x[n+1]-x[n])*(y[m+1]-y[m]);

				/*is it the upper right triangle?*/
				/*2'     3'
				 *+-----+
				 *1\    |
				 *| \   |
				 *|  \  |
				 *|   \ |
				 *|    \|
				 *2----3+1' */
				if ((x_grid-x[n])/(x[n+1]-x[n])<(y_grid-y[m])/(y[m+1]-y[m])){

					/*Then find values of data at each summit*/
					G1=*(data+m*N+n);
					G2=*(data+(m+1)*N+n+1);
					G3=*(data+(m+1)*N+n);

					/*Get first area coordinate*/
					area_1=((y[m+1]-y_grid)*(x[n+1]-x[n]))/area;
					/*Get second area coordinate*/
					area_2=((x_grid-x[n])*(y[m+1]-y[m]))/area;
					/*Get third area coordinate = 1-area1-area2*/
					area_3=1-area_1-area_2;

					/*interpolate*/
					data_value=area_1*G1+area_2*G2+area_3*G3;
				}
				else {

					/*Then find values of data at each summit*/
					G1=*(data+(m+1)*N+n+1);
					G2=*(data+m*N+n);
					G3=*(data+m*N+n+1);

					/*Get first area coordinate*/
					area_1=((y_grid-y[m])*(x[n+1]-x[n]))/area;
					/*Get second area coordinate*/
					area_2=((x[n+1]-x_grid)*(y[m+1]-y[m]))/area;
					/*Get third area coordinate = 1-area1-area2*/
					area_3=1-area_1-area_2;

					/*interpolate*/
					data_value=area_1*G1+area_2*G2+area_3*G3;
				}
				/*}}}*/
			}
			else{
				/*Bilinear  interpolation: (http://en.wikipedia.org/wiki/Bilinear_interpolation) {{{1*/

				/*    Q12    R2        Q22
				 * y2 x------x---------x
				 *    |      |         |
				 *    |      |         |
				 *    |      +P        |
				 *    |      |         |
				 *    |Q11   R1        Q21
				 * y1 x------x---------x
				 *    x1               x2
				 *
				 */

				/*Coordinates*/
				double x1=x[n];
				double x2=x[n+1];
				double y1=y[m];
				double y2=y[m+1];

				/*Values on each summit*/
				double Q11=*(data+m    *N+n);
				double Q21=*(data+m    *N+n+1);
				double Q12=*(data+(m+1)*N+n);
				double Q22=*(data+(m+1)*N+n+1);

				/*Interpolate*/
				data_value=
				  +Q11*(x2-x_grid)*(y2-y_grid)/((x2-x1)*(y2-y1))
				  +Q21*(x_grid-x1)*(y2-y_grid)/((x2-x1)*(y2-y1))
				  +Q12*(x2-x_grid)*(y_grid-y1)/((x2-x1)*(y2-y1))
				  +Q22*(x_grid-x1)*(y_grid-y1)/((x2-x1)*(y2-y1));
				/*}}}*/
			}

			/*Treat NANs*/
			if (isnan(data_value)){
				data_value=default_value;
			}
		}
		else{
			data_value=default_value;
		}
		VecSetValue(data_mesh,i,data_value,INSERT_VALUES);
	}

}


int findindices(int* pm,int* pn,double* x,int x_rows, double* y,int y_rows, double xgrid,double ygrid){

	int foundx=0;
	int foundy=0;
	int i;
	int m=-1;
	int n=-1;

	for (i=0;i<x_rows-1;i++){
		if ( (*(x+i)<=xgrid) && (xgrid<*(x+i+1)) ){
			m=i;
			foundx= 1;
			break;
		}
	}
	if(*(x+x_rows-1)==xgrid){
		m=x_rows-2;
		foundx=1;
	}
	
	for (i=0;i<y_rows-1;i++){
		if ( (*(y+i)<=ygrid) && (ygrid<*(y+i+1)) ){
			n=i;
			foundy= 1;
			break;
		}
	}
	if(*(y+y_rows-1)==ygrid){
		m=y_rows-2;
		foundy=1;
	}

	/*Assign output pointers:*/
	*pm=m;
	*pn=n;
	return foundx*foundy;
}
