/*!\file SystemMatricesx
 * \brief: create system matrices (stiffness matrix, loads vector)
 */

#include "./SystemMatricesx.h"
#include "../../shared/shared.h"
#include "../../include/include.h"
#include "../../toolkits/toolkits.h"
#include "../../EnumDefinitions/EnumDefinitions.h"

void SystemMatricesx(Mat* pKgg, Mat* pKff, Mat* pKfs, Vec* ppg, Vec* ppf, Vec* pdg, Vec* pdf, double* pkmax,Elements* elements,Nodes* nodes, Vertices* vertices,Loads* loads,Materials* materials, Parameters* parameters,bool kflag,bool pflag,bool penalty_kflag,bool penalty_pflag){
	
	/*intermediary: */
	int      i,j;
	int      gsize,fsize,ssize;
	int      connectivity, numberofdofspernode;
	int      analysis_type, configuration_type;
	Element *element = NULL;
	Load    *load    = NULL;
	bool    buildkff=false;
	
	/*output: */
	Mat    Kgg  = NULL;
	Mat    Kff  = NULL;
	Mat    Kfs  = NULL;
	Vec    pg   = NULL;
	Vec    pf   = NULL;
	Vec    dg=NULL;
	Vec    df=NULL;
	double kmax = 0;

	/*Display message*/
	_printf_(VerboseModule(),"   Generating matrices\n");

	/*retrive parameters: */
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);
	parameters->FindParam(&configuration_type,ConfigurationTypeEnum);
	parameters->FindParam(&connectivity,ConnectivityEnum);
	parameters->FindParam(&buildkff,KffEnum);

	/*Get size of matrix: */
	gsize=nodes->NumberOfDofs(configuration_type,GsetEnum);
	fsize=nodes->NumberOfDofs(configuration_type,FsetEnum);
	ssize=nodes->NumberOfDofs(configuration_type,SsetEnum);

	numberofdofspernode=nodes->MaxNumDofs(configuration_type,GsetEnum);

	/*Checks in debugging mode {{{1*/
	if(penalty_kflag)_assert_(kflag);
	if(penalty_pflag)_assert_(pflag);
	/*}}}*/

	/*Compute penalty free mstiffness matrix and load vector*/
	if(kflag){

		if(!buildkff){
			Kgg=NewMat(gsize,gsize,connectivity,numberofdofspernode);
			dg=NewVec(gsize);
		}
		else{
			Kff=NewMat(fsize,fsize,connectivity,numberofdofspernode);
			Kfs=NewMat(fsize,ssize,connectivity,numberofdofspernode);
			df=NewVec(fsize);
		}

		/*Fill stiffness matrix from elements: */
		for (i=0;i<elements->Size();i++){
			element=(Element*)elements->GetObjectByOffset(i);
			element->CreateKMatrix(Kgg,Kff,Kfs,dg,df);
		}

		/*Fill stiffness matrix from loads if loads have the current configuration_type: */
		for (i=0;i<loads->Size();i++){
			load=(Load*)loads->GetObjectByOffset(i);
			if (load->InAnalysis(configuration_type)) load->CreateKMatrix(Kgg,Kff,Kfs);
		}

		/*Assemble matrix and doftypes and compress matrix to save memory: */
		if(!buildkff){
			MatAssemblyBegin(Kgg,MAT_FINAL_ASSEMBLY);
			MatAssemblyEnd(Kgg,MAT_FINAL_ASSEMBLY);
			#if _PETSC_VERSION_ == 2 
			MatCompress(Kgg);
			#endif
			VecAssemblyBegin(dg);
			VecAssemblyEnd(dg);
		}
		else{
			MatAssemblyBegin(Kff,MAT_FINAL_ASSEMBLY);
			MatAssemblyEnd(Kff,MAT_FINAL_ASSEMBLY);
			#if _PETSC_VERSION_ == 2 
			MatCompress(Kff);
			#endif

			MatAssemblyBegin(Kfs,MAT_FINAL_ASSEMBLY);
			MatAssemblyEnd(Kfs,MAT_FINAL_ASSEMBLY);
			#if _PETSC_VERSION_ == 2 
			MatCompress(Kfs);
			#endif
			VecAssemblyBegin(df);
			VecAssemblyEnd(df);
		}
		
	}
	
	if(pflag){

		if(!buildkff)pg=NewVec(gsize);
		else         pf=NewVec(fsize);

		/*Fill right hand side vector, from elements: */
		for (i=0;i<elements->Size();i++){
			element=(Element*)elements->GetObjectByOffset(i);
			element->CreatePVector(pg,pf);
		}

		/*Fill right hand side from loads if loads have the current configuration_type: */
		for (i=0;i<loads->Size();i++){
			load=(Load*)loads->GetObjectByOffset(i);
			if (load->InAnalysis(configuration_type)) load->CreatePVector(pg,pf);
		}

		if(!buildkff){
			VecAssemblyBegin(pg);
			VecAssemblyEnd(pg);
		}
		else{
			VecAssemblyBegin(pf);
			VecAssemblyEnd(pf);
		}
	}

	/*Now, figure out maximum value of K_gg, so that we can penalize it correctly: */
	if(!buildkff)MatNorm(Kgg,NORM_INFINITY,&kmax);
	else MatNorm(Kff,NORM_INFINITY,&kmax);

	/*Now, deal with penalties*/
	if(penalty_kflag){

		/*Fill stiffness matrix from loads: */
		for (i=0;i<loads->Size();i++){
			load=(Load*)loads->GetObjectByOffset(i);
			if (load->InAnalysis(configuration_type)) load->PenaltyCreateKMatrix(Kgg,Kff,Kfs,kmax);
		}

		/*Assemble matrix and compress matrix to save memory: */
		if(!buildkff){
			MatAssemblyBegin(Kgg,MAT_FINAL_ASSEMBLY);
			MatAssemblyEnd(Kgg,MAT_FINAL_ASSEMBLY);
			#if _PETSC_VERSION_ == 2 
			MatCompress(Kgg);
			#endif
		}
		else{
			MatAssemblyBegin(Kff,MAT_FINAL_ASSEMBLY);
			MatAssemblyEnd(Kff,MAT_FINAL_ASSEMBLY);
			#if _PETSC_VERSION_ == 2 
			MatCompress(Kff);
			#endif

			MatAssemblyBegin(Kfs,MAT_FINAL_ASSEMBLY);
			MatAssemblyEnd(Kfs,MAT_FINAL_ASSEMBLY);
			#if _PETSC_VERSION_ == 2 
			MatCompress(Kfs);
			#endif
		}
	}

	
	if(penalty_pflag){

		/*Fill right hand side vector, from loads: */
		for (i=0;i<loads->Size();i++){
			load=(Load*)loads->GetObjectByOffset(i);
			if (load->InAnalysis(configuration_type)) load->PenaltyCreatePVector(pg,pf,kmax);
		}

		if(!buildkff){
			VecAssemblyBegin(pg);
			VecAssemblyEnd(pg);
		}
		else{
			VecAssemblyBegin(pf);
			VecAssemblyEnd(pf);
		}
	}

	/*Assign output pointers: */
	if(pKgg) *pKgg=Kgg;
	if(pKff) *pKff=Kff;
	if(pKfs) *pKfs=Kfs;
	if(ppg)  *ppg=pg;
	if(ppf)  *ppf=pf;
	if(pdg)  *pdg=dg;
	if(pdf)  *pdf=df;
	if(pkmax) *pkmax=kmax;
}
