/*!\file: CreateParametersQmu.cpp
 * \brief general driver for creating parameters dataset
 */ 

#include "../../../Container/Container.h"
#include "../../../toolkits/toolkits.h"
#include "../../../EnumDefinitions/EnumDefinitions.h"
#include "../../../objects/objects.h"
#include "../../../shared/shared.h"
#include "../../../io/io.h"
#include "../../../include/include.h"
#include "../../MeshPartitionx/MeshPartitionx.h"
#include "../ModelProcessorx.h"

void CreateParametersQmu(Parameters** pparameters,IoModel* iomodel,ConstDataHandle iomodel_handle,int solution_type,int analysis_type){

	/*variable declarations: {{{1*/
	int i,j,k;
	
	Parameters* parameters = NULL;
	int      second_count;
	
	int*     part=NULL;
	double*  dpart=NULL;

	char**  responsedescriptors=NULL;
	char**  variabledescriptors=NULL;
	char*   descriptor=NULL;
	double* dakota_parameter=NULL;

	//qmu files
	char* qmuinname=NULL;
	char* qmuerrname=NULL;
	char* qmuoutname=NULL;
	
	//descriptors:
	char tag[50];
	
	/*parameters for mass flux: */
	bool     qmu_mass_flux_present=false;
	double** array=NULL;
	double*  matrix=NULL;
	double*  temp_matrix=NULL;
	int*     mdims_array=NULL;
	int*     ndims_array=NULL;
	int      M;
	int      temp_m,temp_n;
	int      m,n;
	int      count;

	#ifdef _SERIAL_
		mxArray* pfield=NULL;
		mxArray* pfield2=NULL;
	#endif
	/*}}}*/
	
	/*recover parameters : */
	parameters=*pparameters;

	parameters->AddObject(new   BoolParam(QmuAnalysisEnum,iomodel->qmu_analysis));
	if(iomodel->qmu_analysis){

		/*name of qmu input, error and output files:{{{1*/
		qmuinname=(char*)xmalloc((strlen(iomodel->name)+strlen(".qmu.in")+1)*sizeof(char));
		sprintf(qmuinname,"%s%s",iomodel->name,".qmu.in");
		parameters->AddObject(new   StringParam(QmuInNameEnum,qmuinname));

		qmuoutname=(char*)xmalloc((strlen(iomodel->name)+strlen(".qmu.out")+1)*sizeof(char));
		sprintf(qmuoutname,"%s%s",iomodel->name,".qmu.out");
		parameters->AddObject(new   StringParam(QmuOutNameEnum,qmuoutname));

		qmuerrname=(char*)xmalloc((strlen(iomodel->name)+strlen(".qmu.err")+1)*sizeof(char));
		sprintf(qmuerrname,"%s%s",iomodel->name,".qmu.err");
		parameters->AddObject(new   StringParam(QmuErrNameEnum,qmuerrname));
		/*}}}*/

		/*Deal with variable descriptors: {{{1*/
		variabledescriptors=(char**)xmalloc(iomodel->numvariabledescriptors*sizeof(char*));

		/*Fetch descriptors: logic varies if we are running parallel or serial. In parallel, qmumarshall 
		 * took care of marshalling all the variable descriptors, so it's easy. In serial mode, 
		 * the variables are in md.variables(md.ivar), as a strucuture: */

		#ifdef _SERIAL_
		pfield=mxGetAssignedField(iomodel_handle,0,"variabledescriptors");
		for(i=0;i<iomodel->numvariabledescriptors;i++){
			pfield2=mxGetCell(pfield,i);
			FetchData(&descriptor,pfield2);
			variabledescriptors[i]=descriptor;
		}
		#else
		for(i=0;i<iomodel->numvariabledescriptors;i++){
			sprintf(tag,"%s%i","variabledescriptor",i+1);
			IoModelFetchData(&descriptor,iomodel_handle,tag);
			variabledescriptors[i]=descriptor;
		}
		#endif

		/*Ok, we have all the variable descriptors. Build a parameter with it: */
		parameters->AddObject(new StringArrayParam(VariableDescriptorsEnum,variabledescriptors,iomodel->numvariabledescriptors));

		/*}}}*/

		/*Deal with response descriptors: {{{1*/
		responsedescriptors=(char**)xmalloc(iomodel->numresponsedescriptors*sizeof(char*));

		/*Fetch descriptors: */
		#ifdef _SERIAL_
		pfield=mxGetAssignedField(iomodel_handle,0,"responsedescriptors");
		for(i=0;i<iomodel->numresponsedescriptors;i++){
			pfield2=mxGetCell(pfield,i);
			FetchData(&descriptor,pfield2);
			responsedescriptors[i]=descriptor;
		}
		#else
		for(i=0;i<iomodel->numresponsedescriptors;i++){
			sprintf(tag,"%s%i","responsedescriptor",i+1);
			IoModelFetchData(&descriptor,iomodel_handle,tag);
			responsedescriptors[i]=descriptor;
		}
		#endif

		/*Ok, we have all the response descriptors. Build a parameter with it: */
		parameters->AddObject(new StringArrayParam(ResponseDescriptorsEnum,responsedescriptors,iomodel->numresponsedescriptors));
		parameters->AddObject(new    IntParam(QmuNumberOfResponsesEnum,iomodel->numberofresponses));
		/*}}}*/

		/*Deal with partitioning: {{{1*/
		/*partition vertices in iomodel->qmu_npart parts, unless a partition is already present: */
		parameters->AddObject(new    IntParam(QmuNPartEnum,iomodel->qmu_npart));
		
		IoModelFetchData(&dpart,NULL,NULL,iomodel_handle,"part");

		if(!dpart){

			/*Partition elements and vertices and nodes: */
			ElementsAndVerticesPartitioning(&iomodel->my_elements,&iomodel->my_vertices,iomodel,iomodel_handle);

			dpart=(double*)xmalloc(iomodel->numberofvertices*sizeof(double));
			for(i=0;i<iomodel->numberofvertices;i++)dpart[i]=iomodel->my_vertices[i];
		}
		parameters->AddObject(new DoubleVecParam(QmuPartEnum,dpart,iomodel->numberofvertices));
		/*}}}*/

		/*Deal with data needed because of qmu variables: {{{1*/
		
		for(i=0;i<iomodel->numvariabledescriptors;i++){

			if (strncmp(variabledescriptors[i],"scaled_",7)==0){
				/*Ok, we are dealing with a variable that is distributed over grids. Recover the name of the variable (ex: scaled_Thickness): */
				sscanf(variabledescriptors[i],"scaled_%s",tag);
				
				/*Recover data: */
				IoModelFetchData(&dakota_parameter,NULL,NULL,iomodel_handle,EnumToModelField(StringToEnum(tag)));

				/*Convert units: */
				UnitConversion(dakota_parameter,iomodel->numberofvertices,ExtToIuEnum,StringToEnum(tag),parameters);

				/*Add to parameters: */
				parameters->AddObject(new DoubleVecParam(StringToEnum(tag),dakota_parameter,iomodel->numberofvertices));
				
				/*Free ressources:*/
				xfree((void**)&dakota_parameter);
			}
		}
		/*}}}*/

		/*Deal with data needed to compute qmu responses: {{{1*/
		for(i=0;i<iomodel->numresponsedescriptors;i++){
			
			if(strncmp(responsedescriptors[i],"indexed_MassFlux",16)==0){
				qmu_mass_flux_present=true;
			}
		}
		
		
		if(qmu_mass_flux_present){
			
			/*We need the segments to be able to compute the mass flux. We have as many groups of segments as we have MassFlux 
			 *responses. Let's build a DoubleMatArrayParam object with the array of segments: */
			
			if(iomodel->qmu_mass_flux_num_profiles==0)_error_(" qmu_mass_flux_num_profiles is 0, when MassFlux computations were requested!");

			array=(double**)xmalloc(iomodel->qmu_mass_flux_num_profiles*sizeof(double*));
			mdims_array=(int*)xmalloc(iomodel->qmu_mass_flux_num_profiles*sizeof(int));
			ndims_array=(int*)xmalloc(iomodel->qmu_mass_flux_num_profiles*sizeof(int));
			M=iomodel->qmu_mass_flux_num_profiles;

			#ifdef _SERIAL_
			pfield=mxGetAssignedField(iomodel_handle,0,"qmu_mass_flux_segments");
			for(i=0;i<M;i++){
				pfield2=mxGetCell(pfield,i);
				FetchData(&matrix,mdims_array+i,ndims_array+i,pfield2);
				array[i]=matrix;
			}
			#else
			for(i=0;i<M;i++){
				sprintf(tag,"%s%i","qmu_mass_flux_segments",i+1); //names were created using Matlab indexing convention
				IoModelFetchData(&temp_matrix,&temp_m,&temp_n,iomodel_handle,tag);

				/*This temp_matrix represents all the segments, for all elements. On this cpu, we only have a subset of these 
				 * elements. Extract matrix, out of temp_matrix, corresponding to only our elements: */
				m=0;
				for(j=0;j<temp_m;j++){
					if (  iomodel->my_elements[(int)(*(temp_matrix+5*j+4))-1])m++;
				}
				if(m){
					matrix=(double*)xcalloc(5*m,sizeof(double));
					count=0;
					for(j=0;j<temp_m;j++){
						if (iomodel->my_elements[(int)*(temp_matrix+5*j+4)-1]){
							for(k=0;k<5;k++)*(matrix+5*count+k)=*(temp_matrix+5*j+k);
							count++;
						}
					}
				}
				else{
					matrix=NULL;
				}

				/*Assign: */
				array[i]=matrix;
				mdims_array[i]=m;
				ndims_array[i]=5;

				/*Free data: */
				xfree((void**)&temp_matrix);
			}
			#endif

			/*Ok, we have an array of segments, different on every cpu. Create a DoubleMatArrayParam object with it: */
			parameters->AddObject(new DoubleMatArrayParam(QmuMassFluxSegmentsEnum,array,M,mdims_array,ndims_array));

			/*Free data: */
			for(i=0;i<M;i++){
				double* matrix=array[i];
				xfree((void**)&matrix);
			}
			xfree((void**)&mdims_array); 
			xfree((void**)&ndims_array);
			xfree((void**)&array);
		}
		/*}}}*/

		parameters->AddObject(new   BoolParam(QmuSaveFemmodelEnum,iomodel->qmu_save_femmodel));

		/*Free data: {{{1*/
		for(i=0;i<iomodel->numresponsedescriptors;i++){
			descriptor=responsedescriptors[i];
			xfree((void**)&descriptor);
		}
		xfree((void**)&responsedescriptors);

		for(i=0;i<iomodel->numvariabledescriptors;i++){
			descriptor=variabledescriptors[i];
			xfree((void**)&descriptor);
		}
		xfree((void**)&variabledescriptors);

		xfree((void**)&iomodel->elements);
		xfree((void**)&iomodel->elements2d);
		xfree((void**)&part);
		xfree((void**)&dpart);
		xfree((void**)&qmuinname);
		xfree((void**)&qmuerrname);
		xfree((void**)&qmuoutname);
		/*}}}*/
	} //if(iomodel->qmu_analysis)

	/*Assign output pointer: */
	*pparameters=parameters;
}
