%ORGANIZER class definition
%
%   Usage:
%      org = organizer(varargin)
%
%   Examples:
%      org = organizer;                             %build an empty organizer object
%      org = organizer('../Models/');               %build an empty organizer object with a given repository
%      org = organizer('../Models/','models.AGU.'); %build an empty organizer object with a given repository and a prefix
%      org = organizer('../Models/','models.AGU.','model.init'); %repository, prefix, and initial trunk prefix

classdef organizer
    properties (SetAccess=private) 
		 % {{{1
		 num         =0;
	 end
    properties (SetAccess=public) 
		 repository  ='./';
		 prefix      ='model.';
		 trunkprefix ='';
		 steps       =[];
		 %}}}
	 end
	 methods
		 function org=organizer(varargin) % {{{1
			 if nargin>3,
				 help organizer
				 error('organizer constructor error message: bad usage');
			 end
			 if nargin>2,
				 trunkprefix=varargin{3};
				 if ~ischar(trunkprefix),                                 error('trunkprefix is not a string'); end
				 if ~strcmp(regexprep(trunkprefix,'\s+',''),trunkprefix), error('trunkprefix should not have any white space'); end
				 org.trunkprefix=trunkprefix;
			 end
			 if nargin>1,
				 prefix=varargin{2};
				 if ~ischar(prefix),                            error('prefix is not a string'); end
				 if ~strcmp(regexprep(prefix,'\s+',''),prefix), error('prefix should not have any white space'); end
				 org.prefix=prefix;
			 end
			 if nargin>0,
				 repository=varargin{1};
				 if ~ischar(repository),        error('repository is not a string'); end
				 if exist(repository,'dir')~=7, error(['Directory ' repository ' not found']), end
				 org.repository=repository;
			 end
		 end
		 %}}}
		 function disp(org) % {{{1
			 disp(sprintf('   Repository: ''%s''',org.repository));
			 disp(sprintf('   Prefix:     ''%s''',org.prefix));
			 if isempty(org.steps)
				 disp('   no step');
			 else
				 for i=1:length(org.steps),
					 disp(sprintf('   step #%2i: ''%s''',org.steps(i).id,org.steps(i).string));
				 end
			 end
		 end
		 %}}}
		 function md=loadmodel(org,string),% {{{1

			 %Get model path
			 if ~ischar(string), error('argument provided is not a string'); end
			 path=[org.repository '/' org.prefix string];

			 %figure out if the model is there, otherwise, we have to use the default path supplied by user.
			 if exist(path,'file'),
				 md=loadmodel(path);
				 return;
			 end

			 %If we are here, the model has not been found. Try trunk prefix if provided
			 if ~isempty(org.trunkprefix),
				 path2=[org.repository '/' org.trunkprefix string];
				 if ~exist(path2,'file'),
					 error(['Could neither find ' path ', nor ' path2]);
				 else
					 disp(['--> Branching ' org.prefix ' from trunk ' org.trunkprefix]);
					 md=loadmodel(path2);
					 return;
				 end
			 else
				 error(['Could not find ' path ]);
			 end
		 end%}}}
		 function bool=perform(org,steps,string) % {{{1
			 
			 bool=false;

			 %Some checks
			 if ~ischar(string),                            error('Step provided should be a string'); end
			 if ~strcmp(regexprep(string,'\s+',''),string), error('Step provided should not have any white space'); end

			 %Add step
			 org.steps(end+1).id=length(org.steps)+1;
			 org.steps(end).string=string;
			 org.num=org.num+1;

			 %if steps = 0, print all steps in org 
			 if any(steps==0),
				 if org.num==1,
					 disp(sprintf('   prefix: %s',org.prefix(1:end-6)));
				 end
				 disp(sprintf('   step #%2i : %s',org.steps(org.num).id,org.steps(org.num).string));
			 end

			 %Ok, now if num is a member of steps, return true
			 if ismember(org.num,steps),
				 disp(sprintf('\n   step #%i : %s\n',org.steps(org.num).id,org.steps(org.num).string));
				 bool=true;
			 end

			 %assign org back to calling workspace
			 assignin('caller',inputname(1),org);

		 end%}}}
		 function savemodel(org,md) % {{{1

			 %check that md is a model
			 if ~isa(md,'model'),       error('savemodel error message: third argument is not a model'); end
			 if (org.num>length(org.steps)), error(['organizer error message: element with id ' num2str(org.num) ' not found']); end

			 %before saving model, try and find a runme.m file, and save it.
			 A=dbstack;
			 if length(A)>=2,
				 runmefilename=A(2).file;
				 md.runmefile=char(textread(runmefilename,'%s','delimiter','\n'));
			 end

			 %save model
			 name=[org.repository '/' org.prefix org.steps(org.num).string ];
			 save(name,'md','-v7.3');
			 disp(['model saved as: ' name]);
		 end%}}}
	end
end
