/*!\file:  InterpFromMesh2dx.cpp
 * \brief  "c" core code for interpolating values from a structured grid.
 */ 

#include "./InterpFromMesh2dx.h"
#include "../../shared/shared.h"
#include "../../include/include.h"
#include "../../toolkits/toolkits.h"
#include "../../objects/objects.h"
#include "../modules.h"

int InterpFromMesh2dx( Vec* pdata_prime,double* index_data, double* x_data, double* y_data, int nods_data,int nels_data, double* data, int data_length, double* x_prime, double* y_prime, int nods_prime,
		double* default_values,int num_default_values,Contour** contours,int numcontours){

	
	/*Output*/
	Vec data_prime=NULL;

	/*Intermediary*/
	int i,j;
	int interpolation_type;
	bool debug;
	double area;
	double area_1,area_2,area_3;
	double data_value;
	double xmin,xmax;
	double ymin,ymax;

	/*contours: */
	Vec    vec_incontour=NULL;
	double*    incontour=NULL;

	/*some checks*/
	if (nels_data<1 || nods_data<3 || nods_prime==0){
		_error_("nothing to be done according to the mesh given in input");
	}

	/*Set debug to 1 if there are lots of elements*/
	debug=(bool)((double)nels_data*(double)nods_prime >= pow((double)10,(double)9));

	/*figure out what kind of interpolation is needed*/
	if (data_length==nods_data){
		interpolation_type=1;
	}
	else if (data_length==nels_data){
		interpolation_type=2;
	}
	else{
		_error_("length of vector data not supported yet. It should be of length (number of nodes) or (number of elements)!");
	}

	if((numcontours) && (interpolation_type==2)){
		_error_(" element interpolation_type with contours not supported yet!");
	}

	/*Get prime mesh extrema coordinates*/
	xmin=x_prime[0]; xmax=x_prime[0];ymin=y_prime[0]; ymax=y_prime[0];
	for (i=1;i<nods_prime;i++){
		if (x_prime[i]<xmin) xmin=x_prime[i];
		if (x_prime[i]>xmax) xmax=x_prime[i];
		if (y_prime[i]<ymin) ymin=y_prime[i];
		if (y_prime[i]>ymax) ymax=y_prime[i];
	}

	/*Initialize output*/
	data_prime=NewVec(nods_prime);
	if(num_default_values){
		if(num_default_values==1)for (i=0;i<nods_prime;i++) VecSetValue(data_prime,i,default_values[0],INSERT_VALUES);
		else for (i=0;i<nods_prime;i++) VecSetValue(data_prime,i,default_values[i],INSERT_VALUES);
	}

	/*Build indices of contour: */
	if(num_default_values){
		ContourToNodesx( &vec_incontour,x_prime,y_prime,nods_prime,contours,numcontours,1);
		VecToMPISerial(&incontour,vec_incontour);
	}
	else{
		 incontour=(double*)xmalloc(nods_prime*sizeof(double));
		 for (i=0;i<nods_prime;i++) incontour[i]=1;
	}

	/*Loop over the elements*/
	if (debug) printf("      interpolation progress:   %5.2lf %%",0.0);
	for (i=0;i<nels_data;i++){

		/*display current iteration*/
		if (debug && fmod((double)i,(double)100)==0) printf("\b\b\b\b\b\b\b%5.2lf %%",(double)i/nels_data*100);

		/*if there is no point inside the domain, go to next iteration*/
		if ( (x_data[(int)index_data[3*i+0]-1]<xmin) && (x_data[(int)index_data[3*i+1]-1]<xmin) && (x_data[(int)index_data[3*i+2]-1]<xmin)) continue;
		if ( (x_data[(int)index_data[3*i+0]-1]>xmax) && (x_data[(int)index_data[3*i+1]-1]>xmax) && (x_data[(int)index_data[3*i+2]-1]>xmax)) continue;
		if ( (y_data[(int)index_data[3*i+0]-1]<ymin) && (y_data[(int)index_data[3*i+1]-1]<ymin) && (y_data[(int)index_data[3*i+2]-1]<ymin)) continue;
		if ( (y_data[(int)index_data[3*i+0]-1]>ymax) && (y_data[(int)index_data[3*i+1]-1]>ymax) && (y_data[(int)index_data[3*i+2]-1]>ymax)) continue;

		/*get area of the current element (Jacobian = 2 * area)*/
		//area =x2 * y3 - y2*x3 + x1 * y2 - y1 * x2 + x3 * y1 - y3 * x1;
		area=x_data[(int)index_data[3*i+1]-1]*y_data[(int)index_data[3*i+2]-1]-y_data[(int)index_data[3*i+1]-1]*x_data[(int)index_data[3*i+2]-1]
		  +  x_data[(int)index_data[3*i+0]-1]*y_data[(int)index_data[3*i+1]-1]-y_data[(int)index_data[3*i+0]-1]*x_data[(int)index_data[3*i+1]-1]
		  +  x_data[(int)index_data[3*i+2]-1]*y_data[(int)index_data[3*i+0]-1]-y_data[(int)index_data[3*i+2]-1]*x_data[(int)index_data[3*i+0]-1];

		/*loop over the prime nodes*/
		for (j=0;j<nods_prime;j++){

			if(incontour[j]){

				/*Get first area coordinate = det(x-x3  x2-x3 ; y-y3   y2-y3)/area*/
				area_1=((x_prime[j]-x_data[(int)index_data[3*i+2]-1])*(y_data[(int)index_data[3*i+1]-1]-y_data[(int)index_data[3*i+2]-1]) 
						-  (y_prime[j]-y_data[(int)index_data[3*i+2]-1])*(x_data[(int)index_data[3*i+1]-1]-x_data[(int)index_data[3*i+2]-1]))/area;
				/*Get second area coordinate =det(x1-x3  x-x3 ; y1-y3   y-y3)/area*/
				area_2=((x_data[(int)index_data[3*i+0]-1]-x_data[(int)index_data[3*i+2]-1])*(y_prime[j]-y_data[(int)index_data[3*i+2]-1]) 
						- (y_data[(int)index_data[3*i+0]-1]-y_data[(int)index_data[3*i+2]-1])*(x_prime[j]-x_data[(int)index_data[3*i+2]-1]))/area;
				/*Get third area coordinate = 1-area1-area2*/
				area_3=1-area_1-area_2;

				/*is the current point in the current element?*/
				if (area_1>=0 && area_2>=0 && area_3>=0){

					/*Yes ! compute the value on the point*/
					if (interpolation_type==1){
						/*nodal interpolation*/
						data_value=area_1*data[(int)index_data[3*i+0]-1]+area_2*data[(int)index_data[3*i+1]-1]+area_3*data[(int)index_data[3*i+2]-1];
					}
					else{
						/*element interpolation*/
						data_value=data[i];
					}
					if (isnan(data_value)){
						if(num_default_values==1) data_value=default_values[0];
						else data_value=default_values[j];
					}

					/*insert value and go to the next point*/
					VecSetValue(data_prime,j,data_value,INSERT_VALUES);
				}
			}
		}
	}
	 if (debug) printf("\b\b\b\b\b\b\b%5.2lf %%\n",100.0);

	/*Assign output pointers:*/
	 xfree((void**)&incontour);
	*pdata_prime=data_prime;
}
