/*!\file: transient_3d_core.cpp
 * \brief: core of the transient_3d solution 
 */ 

#include "../toolkits/toolkits.h"
#include "../objects/objects.h"
#include "../shared/shared.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "./solutions.h"
#include "../modules/modules.h"
#include "../include/include.h"
#include "../solvers/solvers.h"
#include <float.h>

void transient3d_core(FemModel* femmodel){

	/*parameters: */
	double finaltime;
	double dt,yts;
	bool control_analysis;
	bool   time_adapt=false;
	int    solution_type;
	int    output_frequency;
	
	/*intermediary: */
	int    step;
	double time;

	//first recover parameters common to all solutions
	femmodel->parameters->FindParam(&finaltime,NdtEnum);
	femmodel->parameters->FindParam(&dt,DtEnum);
	femmodel->parameters->FindParam(&yts,YtsEnum);
	femmodel->parameters->FindParam(&control_analysis,ControlAnalysisEnum);
	femmodel->parameters->FindParam(&solution_type,SolutionTypeEnum);
	femmodel->parameters->FindParam(&output_frequency,OutputFrequencyEnum);
	femmodel->parameters->FindParam(&time_adapt,TimeAdaptEnum);

	/*initialize: */
	step=0;
	time=0;

	while(time < finaltime - (yts*DBL_EPSILON)){ //make sure we run up to finaltime.

		/*Increment*/
		if(time_adapt){
			TimeAdaptx(&dt,femmodel->elements, femmodel->nodes,femmodel->vertices,femmodel->loads, femmodel->materials, femmodel->parameters); 
			if(time+dt>finaltime) dt=finaltime-time;
			InputUpdateFromConstantx( femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,dt,DtEnum);
		}
		step+=1;
		time+=dt;

		_printf_(VerboseSolution(),"%s%g%s%i%s%g%s%g\n","time [yr]: ",time/yts,"    iteration number: ",step,"/",floor(finaltime/dt)," dt [yr]: ",dt/yts);

		_printf_(VerboseSolution(),"   computing temperatures:\n");
		thermal_core_step(femmodel,step,time);

		_printf_(VerboseSolution(),"%s\n","   computing new velocity");
		diagnostic_core(femmodel);

		_printf_(VerboseSolution(),"%s\n","   computing new thickness");
		prognostic_core(femmodel);
	
		_printf_(VerboseSolution(),"%s\n","   updating vertices positions");
		UpdateVertexPositionsx(femmodel->elements, femmodel->nodes,femmodel->vertices,femmodel->loads, femmodel->materials, femmodel->parameters); 

		if(solution_type==Transient3DSolutionEnum && !control_analysis && (step%output_frequency==0)){
			_printf_(VerboseSolution(),"%s\n","   saving results\n");
			InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,VxEnum,step,time);
			InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,VyEnum,step,time);
			InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,VzEnum,step,time);
			InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,VelEnum,step,time);
			InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,PressureEnum,step,time);
			InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,ThicknessEnum,step,time);
			InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,SurfaceEnum,step,time);
			InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,BedEnum,step,time);
			InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,TemperatureEnum,step,time);
			InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,MeltingRateEnum,step,time);

			/*unload results*/
			_printf_(VerboseSolution(),"%s","   saving temporary results");
			OutputResultsx(femmodel->elements, femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,&femmodel->results,step,time);
		}
	}

}
