%%
%  create kml polygons for the element mesh.
%
%  [kfold]=kml_mesh_elem(md,params)
%
%  where the required input is:
%    md            (model, model class object)
%
%  the optional input is:
%    params        (string/numeric, parameter names and values)
%
%  and the optional input is:
%    data          (numeric, element or nodal results data)
%    alt           (numeric, altitude for polygons, default 10000)
%    cmin          (numeric, minimum of color map)
%    cmax          (numeric, maximum of color map)
%    cmap          (char or numeric, colormap definition)
%
%  and the required output is:
%    kfold         (kml_folder, folder of polygon placemarks)
%
function [kfold]=kml_mesh_elem(varargin)

if ~nargin
    help kml_mesh_elem
    return
end

%%  process input data

iarg=1;
if (nargin >= 1)
    md=varargin{1};
end
if ~exist('md','var') || isempty(md) || ~isa(md,'model')
    error(['Model ''' inputname(iarg) ''' is unrecognized class ''' class(md) '''.']);
end

%  parameters

iarg=iarg+1;
while (iarg <= nargin-1)
    if ischar(varargin{iarg})
        eval([varargin{iarg} '=varargin{iarg+1};']);
        if (numel(varargin{iarg+1}) <= 20)
            disp([varargin{iarg} '=' any2str(varargin{iarg+1}) ';']);
        else
            disp([varargin{iarg} '=' string_size(varargin{iarg+1}) ' ' class(varargin{iarg+1}) ';']);
        end
        if strcmpi(varargin{iarg},'data')
            cdata=inputname(iarg+1);
        end
    else
        error(['''' any2str(varargin{iarg}) ''' is not a parameter name.']);
    end
    iarg=iarg+2;
end

if exist('data','var') && ~isempty(data)
    if     (numel(data)==md.numberofelements)
        edata=data;
    elseif (numel(data)==md.numberofgrids)
        ndata=data;
        display('Averaging nodal data to element data.');
        edata=zeros(1,md.numberofelements);
        for i=1:size(md.elements,1)
            for j=1:size(md.elements,2)
                edata(i)=edata(i)+ndata(md.elements(i,j));
            end
            edata(i)=edata(i)/size(md.elements,2);
        end
    else
        error(['Data has incorrect number of ' num2str(numel(data)) ' values.']);
    end
end

%  colormap command operates on a figure, so create an invisible one
%  (could also directly call colormaps, e.g. jet(64), but risky)

hfig=figure('Visible','off');
if exist('cmap','var')
    colormap(cmap)
end
cmap=colormap;
close(hfig)
    
if exist('edata','var')
    if ~exist('cmin','var')
        cmin=min(min(edata));
    end
    if ~exist('cmax','var')
        cmax=max(max(edata));
    end
end

if ~exist('alt','var')
    alt=10000;
end

%%  write folder for mesh

kfold=kml_folder();
if exist('cdata','var') && ~isempty(cdata)
    kfold.name      =sprintf('Data: %s',cdata);
else
    kfold.name      =sprintf('Mesh');
end
kfold.visibility=1;
kfold.descript  =sprintf('Elements=%d, Grids=%d',...
    md.numberofelements,md.numberofgrids);
kfold.feature   ={repmat(kml_placemark(),1,size(md.elements,1))};

%  write each element as a polygon placemark

disp(['Writing ' num2str(size(md.elements,1)) ' tria elements as KML polygons.']);
for i=1:size(md.elements,1)
    kplace=kml_placemark();
    kplace.name      =sprintf('Element %d',i);
    kplace.visibility=1;
    if exist('edata','var')
        kplace.descript  =sprintf('Element data: %g',edata(i));
        imap = fix((edata(i)-cmin)/(cmax-cmin)*size(cmap,1))+1;
        if     (imap >= 1) && (imap <= size(cmap,1))
            kplace.styleurl  =sprintf('#MatlabColor%d',imap);
        elseif (edata(i) == cmax)
            kplace.styleurl  =sprintf('#MatlabColor%d',size(cmap,1));
        else
            kplace.styleurl  =sprintf('#BlackLineEmptyPoly');
        end
    else
        kplace.styleurl  =sprintf('#BlackLineRandomPoly');
    end

    kpoly=kml_polygon();
    kpoly.extrude   =1;
    kpoly.altmode   ='relativeToGround';

    kring=kml_linearring();
    kring.coords    =zeros(size(md.elements,2)+1,3);

	if ~isempty(md.lat) & ~isempty(md.long),
		for j=1:size(md.elements,2)
			lat=md.lat(md.elements(i,j)); long=md.long(md.elements(i,j));
			kring.coords(j,:)=[long lat alt];
		end
	else
		for j=1:size(md.elements,2)
			[lat,long]=mapxy(md.x(md.elements(i,j)),md.y(md.elements(i,j)),'s');
			kring.coords(j,:)=[long lat alt];
		end
	end
    kring.coords(end,:)=kring.coords(1,:);

    kpoly.outer=kring;
    kplace.geometry=kpoly;
    kfold.feature{1}(i)=kplace;
    clear kring kpoly kplace
end

end

