function md=setelementstype(md,varargin)
%SETELEMENTSTYPE - associate a solution type to each element
%
%   This routine works like plotmodel: it works with an even number of inputs
%   'hutter','macayeal','pattyn','stokes' and 'fill' are the possible options
%   that must be followed by the corresponding exp file or flags list
%   It can either be a domain file (argus type, .exp extension), or an array of element flags. 
%   If user wants every element outside the domain to be 
%   setelementstyped, add '~' to the name of the domain file (ex: '~Pattyn.exp');
%   an empty string '' will be considered as an empty domain
%   a string 'all' will be considered as the entire domain
%   You can specify the type of coupling, 'penalties' or 'tiling', to use with the input 'coupling'
%
%   Usage:
%      md=setelementstype(md,varargin)
%
%   Example:
%      md=setelementstype(md,'pattyn','Pattyn.exp','macayeal',md.elementoniceshelf,'fill','hutter');
%      md=setelementstype(md,'pattyn','Pattyn.exp',fill','hutter','coupling','tiling');

%some checks on list of arguments
if ((nargin<2) | (nargout~=1)),
	error('setelementstype error message');
end

if md.counter<3,
	error('only fully parameterized 2d models can be setelementstyped');
end

%Find_out what kind of coupling to use
options=pairoptions(varargin{:});
coupling_method=getfieldvalue(options,'coupling','tiling');
if (~strcmpi(coupling_method,'tiling') & ~strcmpi(coupling_method,'penalties')),
	error('coupling type can only be: tiling or penalties');
end

[hutterflag macayealflag pattynflag stokesflag filltype]=recover_areas(md,varargin{:});

%Flag the elements that has not been flagged as filltype
if strcmpi(filltype,'hutter'),
	hutterflag(find(~macayealflag & ~pattynflag))=1;
elseif strcmpi(filltype,'macayeal'),
	macayealflag(find(~hutterflag & ~pattynflag))=1;
elseif strcmpi(filltype,'pattyn'),
	pattynflag(find(~hutterflag & ~macayealflag & ~stokesflag))=1;
end

%check that each element has at least one flag
if any(hutterflag+ macayealflag+pattynflag+stokesflag==0),
	error('setelementstype error message: elements type not assigned, must be specified')
end

%check that each element has only one flag
if any(hutterflag+ macayealflag+pattynflag+stokesflag>1),
	disp('setelementstype warning message: some elements have several types, higher order type is used for them')
	hutterflag(find(hutterflag & macayealflag))=0;
	hutterflag(find(hutterflag & pattynflag))=0;
	macayealflag(find(macayealflag & pattynflag))=0;
end

%Check that no pattyn or stokes for 2d mesh
if (md.dim==2),
	if any(stokesflag | pattynflag)
		error('setelementstype error message: stokes and pattyn elements no allowed in 2d mesh, extrude it first')
	end
end

%Stokes can only be used alone for now:
if any(stokesflag) &any(hutterflag),
	error('setelementstype error message: stokes cannot be used with any other model for now, put stokes everywhere')
end

%add in model who is who
md.elements_type=zeros(md.numberofelements,1);

%1: Hutter elements
gridonhutter=zeros(md.numberofgrids,1);
gridonhutter(md.elements(find(hutterflag),:))=1;
md.gridonhutter=gridonhutter;
md.elements_type(find(hutterflag))=HutterApproximationEnum();

%2: MacAyeal elements
gridonmacayeal=zeros(md.numberofgrids,1);
gridonmacayeal(md.elements(find(macayealflag),:))=1;
md.gridonmacayeal=gridonmacayeal;
md.elements_type(find(macayealflag))=MacAyealApproximationEnum();

%3: Pattyn elements
gridonpattyn=zeros(md.numberofgrids,1);
gridonpattyn(md.elements(find(pattynflag),:))=1;
md.gridonpattyn=gridonpattyn;
md.elements_type(find(pattynflag))=PattynApproximationEnum();

%4: Stokes elements
%First modify stokesflag to get rid of elements contrained everywhere (spc + border with pattyn or macayeal)
if any(stokesflag),
	gridonstokes=zeros(md.numberofgrids,1);
	gridonstokes(md.elements(find(stokesflag),:))=1;
	fullspcnodes=double(sum(md.spcvelocity(:,1:3),2)==3 | (gridonpattyn & gridonstokes));         %find all the grids on the boundary of the domain without icefront
	fullspcelems=double(sum(fullspcnodes(md.elements),2)==6);         %find all the grids on the boundary of the domain without icefront
	stokesflag(find(fullspcelems))=0;
end
gridonstokes=zeros(md.numberofgrids,1);
gridonstokes(md.elements(find(stokesflag),:))=1;
md.gridonstokes=gridonstokes;
md.elements_type(find(stokesflag))=StokesApproximationEnum();

%Then complete with NoneApproximation or the other model used if there is no stokes
if any(stokesflag), 
	if any(pattynflag), %fill with pattyn
		pattynflag(~stokesflag)=1;
		gridonpattyn(md.elements(find(pattynflag),:))=1;
		md.gridonpattyn=gridonpattyn;
		md.elements_type(find(~stokesflag))=PattynApproximationEnum();
	else %fill with none 
	%5: None elements (non Stokes)
		md.elements_type(find(~stokesflag))=NoneApproximationEnum();
	end
end

%Now take care of the coupling between MacAyeal and Pattyn
md.penalties=[];
gridonmacayealpattyn=zeros(md.numberofgrids,1);
gridonpattynstokes=zeros(md.numberofgrids,1);
gridonmacayealstokes=zeros(md.numberofgrids,1);
if strcmpi(coupling_method,'penalties'),
	%Create the border grids between Pattyn and MacAyeal and extrude them
	numgrids2d=md.numberofgrids2d;
	numlayers=md.numlayers;
	bordergrids2d=find(gridonpattyn(1:numgrids2d) & gridonmacayeal(1:numgrids2d)); %Grids connected to two different types of elements

	%initialize and fill in penalties structure
	if ~isnan(bordergrids2d),
		penalties=[];
		for	i=1:numlayers-1,
			penalties=[penalties; [bordergrids2d bordergrids2d+md.numberofgrids2d*(i)]];
		end
		md.penalties=penalties;
	end
elseif strcmpi(coupling_method,'tiling'),
	if any(macayealflag) & any(pattynflag), %coupling macayeal pattyn
		%Find grid at the border
		gridonmacayealpattyn(find(gridonmacayeal & gridonpattyn))=1;
		%Macayeal elements in contact with this layer become MacAyealPattyn elements
		matrixelements=ismember(md.elements,find(gridonmacayealpattyn));
		commonelements=sum(matrixelements,2)~=0;
		commonelements(find(pattynflag))=0; %only one layer: the elements previously in macayeal
		macayealflag(find(commonelements))=0; %these elements are now macayealpattynelements
		macayealpattynflag=zeros(md.numberofelements,1);
		macayealpattynflag(find(commonelements))=1;
		gridonmacayeal=zeros(md.numberofgrids,1);
		gridonmacayeal(md.elements(find(macayealflag),:))=1;
		md.gridonmacayeal=gridonmacayeal;

		%Create MacaAyealPattynApproximation where needed
		md.elements_type(find(macayealpattynflag))=MacAyealPattynApproximationEnum();

		%Now recreate gridonmacayeal and gridonmacayealpattyn
		gridonmacayealpattyn(md.elements(find(macayealpattynflag),:))=1;
	elseif any(pattynflag) & any(stokesflag), %coupling pattyn stokes
		%Find grid at the border
		gridonpattynstokes(find(gridonpattyn & gridonstokes))=1;
		%Stokes elements in contact with this layer become PattynStokes elements
		matrixelements=ismember(md.elements,find(gridonpattynstokes));
		commonelements=sum(matrixelements,2)~=0;
		commonelements(find(pattynflag))=0; %only one layer: the elements previously in macayeal
		stokesflag(find(commonelements))=0; %these elements are now macayealpattynelements
		pattynstokesflag=zeros(md.numberofelements,1);
		pattynstokesflag(find(commonelements))=1;
		gridonstokes=zeros(md.numberofgrids,1);
		gridonstokes(md.elements(find(stokesflag),:))=1;
		md.gridonstokes=gridonstokes;

		%Create MacaAyealPattynApproximation where needed
		md.elements_type(find(pattynstokesflag))=PattynStokesApproximationEnum();

		%Now recreate gridonpattynstokes
		gridonpattynstokes=zeros(md.numberofgrids,1);
		gridonpattynstokes(md.elements(find(pattynstokesflag),:))=1;
	elseif any(stokesflag) & any(macayealflag),
		%Find grid at the border
		gridonmacayealstokes(find(gridonmacayeal & gridonstokes))=1;
		%Stokes elements in contact with this layer become MacAyealStokes elements
		matrixelements=ismember(md.elements,find(gridonmacayealstokes));
		commonelements=sum(matrixelements,2)~=0;
		commonelements(find(macayealflag))=0; %only one layer: the elements previously in macayeal
		stokesflag(find(commonelements))=0; %these elements are now macayealmacayealelements
		macayealstokesflag=zeros(md.numberofelements,1);
		macayealstokesflag(find(commonelements))=1;
		gridonstokes=zeros(md.numberofgrids,1);
		gridonstokes(md.elements(find(stokesflag),:))=1;
		md.gridonstokes=gridonstokes;

		%Create MacaAyealMacAyealApproximation where needed
		md.elements_type(find(macayealstokesflag))=MacAyealStokesApproximationEnum();

		%Now recreate gridonmacayealstokes
		gridonmacayealstokes=zeros(md.numberofgrids,1);
		gridonmacayealstokes(md.elements(find(macayealstokesflag),:))=1;
	elseif any(stokesflag) & any(hutterflag),
		error('type of coupling not supported yet');
	end
end

%Create vertices_type
md.vertices_type=zeros(md.numberofgrids,1);
pos=find(gridonhutter);
md.vertices_type(pos)=HutterApproximationEnum();
pos=find(gridonmacayeal);
md.vertices_type(pos)=MacAyealApproximationEnum();
pos=find(gridonpattyn);
md.vertices_type(pos)=PattynApproximationEnum();
pos=find(gridonhutter);
md.vertices_type(pos)=HutterApproximationEnum();
pos=find(gridonpattyn & gridonmacayeal);
md.vertices_type(pos)=PattynApproximationEnum();
pos=find(gridonmacayealpattyn);
md.vertices_type(pos)=MacAyealPattynApproximationEnum();
pos=find(gridonstokes);
md.vertices_type(pos)=StokesApproximationEnum();
if any(stokesflag),
	pos=find(~gridonstokes);
	if(~any(pattynflag)),
		md.vertices_type(pos)=NoneApproximationEnum();
	end
end
pos=find(gridonpattynstokes);
md.vertices_type(pos)=PattynStokesApproximationEnum();
pos=find(gridonmacayealstokes);
md.vertices_type(pos)=MacAyealStokesApproximationEnum();

%figure out solution types
md.ishutter=double(any(md.elements_type==HutterApproximationEnum));
md.ismacayealpattyn=double(any(md.elements_type==MacAyealApproximationEnum | md.elements_type==PattynApproximationEnum));
md.isstokes=double(any(md.elements_type==StokesApproximationEnum));

end
