/*!\file:  ProcessResults.cpp
 * \brief: go through results dataset, and for each result, process it for easier retrieval 
 * by the Matlab side. This usually means splitting the velocities from the g-size nodeset 
 * to the grid set (ug->vx,vy,vz), same for pressure (p_g->pressure), etc ... It also implies 
 * departitioning of the results.
 * This phase is necessary prior to outputting the results on disk.
 */ 

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#undef __FUNCT__ 
#define __FUNCT__ "ProcessResults"

#include "../DataSet/DataSet.h"
#include "../objects/objects.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../shared/shared.h"

void ProcessResults(DataSet** presults,FemModel* fems,int analysis_type){

	int i,n;
	Result* result=NULL;
	Result* newresult=NULL;
	
	/*input: */
	DataSet* results=NULL;

	/*output: */
	DataSet* newresults=NULL;

	/*fem diagnostic models: */
	FemModel* fem_dh=NULL;
	FemModel* fem_dv=NULL;
	FemModel* fem_dhu=NULL;
	FemModel* fem_ds=NULL;
	FemModel* fem_sl=NULL;

	/*fem prognostic models: */
	FemModel* fem_p=NULL;

	int ishutter;
	int ismacayealpattyn;
	int isstokes;

	/*intermediary: */
	Vec     u_g=NULL;
	double* u_g_serial=NULL;
	double* vx=NULL;
	double* vy=NULL;
	double* vz=NULL;
	double* vel=NULL;
	Vec     p_g=NULL;
	double* p_g_serial=NULL;
	double* pressure=NULL;
	double* partition=NULL;
	double  yts;

	Vec     h_g=NULL;
	double* h_g_serial=NULL;
	double* thickness=NULL;

	int numberofnodes;

	/*recover input results: */
	results=*presults;

	/*Initialize new results: */
	newresults=new DataSet(ResultsEnum());
		
	/*Recover femmodels first: */
	if(analysis_type==DiagnosticAnalysisEnum()){

		fem_dh=fems+0;
		fem_dv=fems+1;
		fem_ds=fems+2;
		fem_dhu=fems+3;
		fem_sl=fems+4;
	
		/*some flags needed: */
		fem_dhu->parameters->FindParam((void*)&ishutter,"ishutter");
		fem_ds->parameters->FindParam((void*)&isstokes,"isstokes");
		fem_dh->parameters->FindParam((void*)&ismacayealpattyn,"ismacayealpattyn");
	}

	if(analysis_type==PrognosticAnalysisEnum()){
		fem_p=fems+0;
	}
	

	for(n=0;n<results->Size();n++){
	
		result=(Result*)results->GetObjectByOffset(n);
		
		if(strcmp(result->GetFieldName(),"u_g")==0){
			/*Ok, are we dealing with velocities coming from MacAyeal, Pattyin, Hutter, on 2 dofs? or from 
			 *Stokes on 4 dofs: */
			result->GetField(&u_g);
			VecToMPISerial(&u_g_serial,u_g);

			if(!isstokes){
				/*ok, 2 dofs, on number of nodes: */
				if(ismacayealpattyn){
					fem_dh->parameters->FindParam((void*)&numberofnodes,"numberofnodes");
					VecToMPISerial(&partition,fem_dh->partition);
					fem_dh->parameters->FindParam((void*)&yts,"yts");
				}
				else{
					fem_dhu->parameters->FindParam((void*)&numberofnodes,"numberofnodes");
					VecToMPISerial(&partition,fem_dhu->partition);
					fem_dhu->parameters->FindParam((void*)&yts,"yts");
				}
				vx=(double*)xmalloc(numberofnodes*sizeof(double));
				vy=(double*)xmalloc(numberofnodes*sizeof(double));
				vz=(double*)xmalloc(numberofnodes*sizeof(double)); 
				vel=(double*)xmalloc(numberofnodes*sizeof(double)); 

				for(i=0;i<numberofnodes;i++){
					vx[i]=u_g_serial[2*(int)partition[i]+0]*yts;
					vy[i]=u_g_serial[2*(int)partition[i]+1]*yts;
					vz[i]=0;
					vel[i]=sqrt(pow(vx[i],2)+pow(vy[i],2)+pow(vz[i],2));
				}

			}
			else{
				/* 4 dofs on number of nodes. discard pressure: */
				fem_ds->parameters->FindParam((void*)&numberofnodes,"numberofnodes");
				VecToMPISerial(&partition,fem_ds->partition);
				fem_ds->parameters->FindParam((void*)&yts,"yts");
				vx=(double*)xmalloc(numberofnodes*sizeof(double));
				vy=(double*)xmalloc(numberofnodes*sizeof(double));
				vz=(double*)xmalloc(numberofnodes*sizeof(double));
				vel=(double*)xmalloc(numberofnodes*sizeof(double));
				for(i=0;i<numberofnodes;i++){
					vx[i]=u_g_serial[4*(int)partition[i]+0]*yts;
					vy[i]=u_g_serial[4*(int)partition[i]+1]*yts;
					vz[i]=u_g_serial[4*(int)partition[i]+2]*yts;
					vel[i]=sqrt(pow(vx[i],2)+pow(vy[i],2)+pow(vz[i],2));
				}
			}

			/*Ok, add vx,vy and vz to newresults: */
			newresult=new Result(newresults->Size()+1,result->GetTime(),result->GetStep(),"vx",vx,numberofnodes);
			newresults->AddObject(newresult);

			newresult=new Result(newresults->Size()+1,result->GetTime(),result->GetStep(),"vy",vy,numberofnodes);
			newresults->AddObject(newresult);

			newresult=new Result(newresults->Size()+1,result->GetTime(),result->GetStep(),"vz",vz,numberofnodes);
			newresults->AddObject(newresult);

			newresult=new Result(newresults->Size()+1,result->GetTime(),result->GetStep(),"vel",vel,numberofnodes);
			newresults->AddObject(newresult);

			/*do some cleanup: */
			xfree((void**)&u_g_serial);
			xfree((void**)&partition);
		}
		else if(strcmp(result->GetFieldName(),"p_g")==0){
			/*easy, p_g is of size numberofnodes, on 1 dof, just repartition: */
			result->GetField(&p_g);
			VecToMPISerial(&p_g_serial,p_g);

			if(!isstokes){
				if(ismacayealpattyn){
					fem_dh->parameters->FindParam((void*)&numberofnodes,"numberofnodes");
					VecToMPISerial(&partition,fem_dh->partition);
				}
				else{
					fem_dhu->parameters->FindParam((void*)&numberofnodes,"numberofnodes");
					VecToMPISerial(&partition,fem_dhu->partition);
				}
			}
			else{
				fem_ds->parameters->FindParam((void*)&numberofnodes,"numberofnodes");
				VecToMPISerial(&partition,fem_ds->partition);
			}

			pressure=(double*)xmalloc(numberofnodes*sizeof(double));

			for(i=0;i<numberofnodes;i++){
				pressure[i]=p_g_serial[(int)partition[i]];
			}
			
			/*Ok, add pressure,vy and vz to newresults: */
			newresult=new Result(newresults->Size()+1,result->GetTime(),result->GetStep(),"pressure",pressure,numberofnodes);
			newresults->AddObject(newresult);

			/*do some cleanup: */
			xfree((void**)&p_g_serial);
			xfree((void**)&partition);
		}
		else if(strcmp(result->GetFieldName(),"h_g")==0){
			/*easy, h_g is of size numberofnodes, on 1 dof, just repartition: */
			result->GetField(&h_g);
			VecToMPISerial(&h_g_serial,h_g);
			fem_p->parameters->FindParam((void*)&numberofnodes,"numberofnodes");
			VecToMPISerial(&partition,fem_p->partition);

			thickness=(double*)xmalloc(numberofnodes*sizeof(double));

			for(i=0;i<numberofnodes;i++){
				thickness[i]=h_g_serial[(int)partition[i]];
			}
			
			/*Ok, add pressure to newresults: */
			newresult=new Result(newresults->Size()+1,result->GetTime(),result->GetStep(),"thickness",thickness,numberofnodes);
			newresults->AddObject(newresult);

			/*do some cleanup: */
			xfree((void**)&h_g_serial);
			xfree((void**)&partition);
		}
	}

	/*Delete results: */
	delete results;


	/*Assign output pointers:*/
	*presults=newresults;
}

