function runme(varargin)
% This file can be run to check that the current version of issm is giving 
% coherent results. This test deals with an icesheet without icefront for a 2d model. The geometry 
% is square. Just run this file in Matlab, with a properly setup ISSM code. 
% The results of this test will indicate if there is a difference between current computations 
% and archived results.

% Errors  between archived results and the current version will get flagged if they are not within 
% a certain tolerance. The current tolerance is 10^-12. If you have good reasons to believe this 
% tolerance should be lowered (for example, if you are running single precision compilers?), feel 
% free to tweak the tolerance variable.

%packages and analysis_types to be tested
if nargin==1,
	packages=varargin{1};
else
	packages={'macayeal','ice','cielo_serial'};
end
solutions={'diagnostic','prognostic'};

%Initialize log message for nightly runs.
testname='IceSheetNoIceFrontM2d_16';
tolerance=10^-12;

%go through the solutions requested
for i=1:length(packages),
	package=packages{i};

	for j=1:length(solutions),
		solution=solutions{j};

		%check solution requested
		if ((~strcmpi(package,'ice') & strcmpi(solution,'transient'))...
			| (strcmpi(package,'macayeal') & ~strcmpi(solution,'diagnostic'))),
			disp(sprintf(['\nsolution: ' solution  ', with package: ' package  ', in test: ' testname  ', not supported yet.\n']));
			continue
		end

		%initialize model
		md=model;
		md=mesh(md,'DomainOutline.exp',50000);
		md=geography(md,'','');
		md=parameterize(md,'Square.par');
		md=setelementstype(md,'macayeal','all');
		if md.numberofgrids==388
			load Velocities; md.vx=0.5*vx; md.vy=0.5*vy;
		end

		%compute solution
		[analysis_type sub_analysis_type]=testsgetanalysis(solution);
		[md packagefinal]=testsgetpackage(md,package);
		md=solve(md,'analysis_type',analysis_type,'sub_analysis_type',sub_analysis_type,'package',packagefinal);

		%compute fields to be checked
		fields=testsgetfields(md.type,solution);

		%load archive
		eval(['load Archive' package solution ]);

		for k=1:length(fields),
			field=fields{k};

			%compare to archive
			eval(['Archive=Archive' package solution '_field' num2str(k) ';']);
			eval(['error_diff=abs(norm((Archive(find(Archive>tolerance))-md.' field  '(find(Archive>tolerance)))./Archive(find(Archive>tolerance)),2));']);

			%disp test result
			if (error_diff>tolerance);
				disp(sprintf(['\n\nERROR   (difference=%-7.2g > %g) --> test: %-25s solution: %-16s package: %-14s field: ' field  '.\n\n'],error_diff,tolerance,testname,solution,package));
			else
				disp(sprintf(['\n\nSUCCESS (difference=%-7.2g < %g) --> test: %-25s solution: %-16s package: %-14s field: ' field  '.\n\n'],error_diff,tolerance,testname,solution,package));
			end

		end
	end
end
