function md=preqmu(md,options)
%QMU - apply Quantification of Margins and Uncertainties techniques 
%      to a solution sequence (like diagnostic.m, progonstic.m, etc ...), 
%      using the Dakota software from Sandia.
%
%   options come from the solve.m routine. They can include Dakota options:
%
%       qmudir:  any directory where to run the qmu analysis
%       qmufile: input file for Dakota
%       ivar: selection number for variables input (if several are specified in variables)
%       iresp: same thing for response functions
%       imethod: same thing for methods
%       iparams: same thing for params
%       overwrite: overwrite qmudir before analysis
%       keep: keep qmudir after analysis
%       outfiles: (John?)
%       rstfile: backup file name
%       rundakota: (John?)
%       runmpi: (John?)

global ISSM_TIER;

disp('preprocessing dakota inputs');

%first create temporary directory in which we will work
if strncmpi(options.overwrite,'y',1)
	system(['rm -rf ' options.qmudir '/*']); 
else
	%does the directory exist? if so, then error out
	if exist(options.qmudir)==7,
		error('Existing ''%s'' directory, cannot overwrite. Specify ''overwrite'',''y'' option in solve arguments.',options.qmudir);
	end
end
mkdir(options.qmudir)
cd(options.qmudir)

%when running in library mode, the in file needs to be called md.name.qmu.in
options.qmufile=[md.name ];

%retrieve variables and resposnes for this particular analysis.
variables=md.variables(options.ivar);
responses=md.responses(options.iresp);

%expand variables and responses
variables=expandvariables(md,md.variables);
responses=expandresponses(md,md.responses);

%go through variables and responses, and check they don't have more than md.npart values. Also determine numvariables and numresponses{{{1
numvariables=0;
variable_fieldnames=fieldnames(variables);
for i=1:length(variable_fieldnames),
	field_name=variable_fieldnames{i};
	fieldvariables=variables.(field_name);
	if numel(fieldvariables)>md.npart,
		error('preqmu error message: one of the expanded variables has more values than the number of partitions (setup in md.npart)');
	end
	numvariables=numvariables+numel(variables.(field_name));
end

numresponses=0;
response_fieldnames=fieldnames(responses);
for i=1:length(response_fieldnames),
	field_name=response_fieldnames{i};
	fieldresponses=responses.(field_name);
	if numel(fieldresponses)>md.npart,
		error('preqmu error message: one of the expanded responses has more values than the number of partitions (setup in md.npart)');
	end
	numresponses=numresponses+numel(responses.(field_name));
end
%}}}}

%create in file for dakota
dakota_in_data(md.qmu_method(options.imethod),variables,responses,md.qmu_params(options.iparams),options.qmufile);
system(['rm -rf ' md.name '.m']);

%build a list of variables and responses descriptors. the list is not expanded. {{{1
variabledescriptors={};
variable_fieldnames=fieldnames(md.variables);
for i=1:length(variable_fieldnames),
	field_name=variable_fieldnames{i};
	variabledescriptors{end+1}=md.variables.(field_name).descriptor;
end

responsedescriptors={};
response_fieldnames=fieldnames(md.responses);
for i=1:length(response_fieldnames),
	field_name=response_fieldnames{i};
	responsedescriptors{end+1}=md.responses.(field_name).descriptor;
end
%}}}

%register the fields that will be needed by the Qmu model.
md.numberofvariables=numvariables;
md.numberofresponses=numresponses;
md.variabledescriptors=variabledescriptors;
md.responsedescriptors=responsedescriptors;
md.numvariabledescriptors=numel(md.variabledescriptors);
md.numresponsedescriptors=numel(md.responsedescriptors);

%now, we have to provide all the info necessary for the solutions to compute the responses. For ex, if mass_flux 
%is a response, we need a profile of points.  For a misfit, we need the observed velocity, etc ...
md=process_qmu_response_data(md);
