/*!\file:  thermal.cpp
 * \brief: thermal solution
 */ 

#include "../issm.h"
#include "./parallel.h"

#undef __FUNCT__ 
#define __FUNCT__ "thermal"

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

int main(int argc,char* *argv){

	int i,n;
	
	/*I/O: */
	FILE* fid=NULL;
	char* inputfilename=NULL;
	char* outputfilename=NULL;
	char* lockname=NULL;
	int   numberofnodes;

	/*Fem models : */
	FemModel femmodels[2];

	/*initial velocity and pressure: */
	double* u_g=NULL;
	double* p_g=NULL;

	/*Results: */
	DataSet* results=NULL;
	DataSet* newresults=NULL;
	
	ParameterInputs* inputs=NULL;
	Param*           param=NULL;
	double  dt;

	int    waitonlock=0;
		
	MODULEBOOT();

	#if !defined(_PARALLEL_) || (defined(_PARALLEL_) && !defined(_HAVE_PETSC_))
	throw ErrorException(__FUNCT__," parallel executable was compiled without support of parallel libraries!");
	#endif

	PetscInitialize(&argc,&argv,(char *)0,"");  

	/*Size and rank: */
	MPI_Comm_rank(MPI_COMM_WORLD,&my_rank);  
	MPI_Comm_size(MPI_COMM_WORLD,&num_procs); 

	inputfilename=argv[2];
	outputfilename=argv[3];
	lockname=argv[4];

	/*Open handle to data on disk: */
	fid=pfopen(inputfilename,"rb");

	_printf_("read and create thermal finite element model:\n");
	CreateFemModel(&femmodels[0],fid,"thermal",NULL);
	_printf_("read and create melting finite element model:\n");
	CreateFemModel(&femmodels[1],fid,"melting",NULL);

	_printf_("initialize inputs:\n");
	femmodels[0].parameters->FindParam((void*)&u_g,"u_g");
	femmodels[0].parameters->FindParam((void*)&p_g,"p_g");
	femmodels[0].parameters->FindParam((void*)&numberofnodes,"numberofnodes");
	femmodels[0].parameters->FindParam((void*)&waitonlock,"waitonlock");
	femmodels[0].parameters->FindParam((void*)&dt,"dt");

	inputs=new ParameterInputs;

	inputs->Add("velocity",u_g,3,numberofnodes);
	inputs->Add("pressure",p_g,1,numberofnodes);
	inputs->Add("dt",dt);

	//erase velocities and pressure embedded in parameters
	param=(Param*)femmodels[0].parameters->FindParamObject("u_g");
	femmodels[0].parameters->DeleteObject((Object*)param);
	param=(Param*)femmodels[0].parameters->FindParamObject("p_g");
	femmodels[0].parameters->DeleteObject((Object*)param);
	param=(Param*)femmodels[1].parameters->FindParamObject("u_g");
	femmodels[1].parameters->DeleteObject((Object*)param);
	param=(Param*)femmodels[1].parameters->FindParamObject("p_g");
	femmodels[1].parameters->DeleteObject((Object*)param);
	
	_printf_("call computational core:\n");
	thermal_core(results,femmodels,inputs);

	_printf_("process results:\n");
	ProcessResults(&newresults,results,&femmodels[0],ThermalAnalysisEnum()); delete results;

	_printf_("write results to disk:\n");
	OutputResults(newresults,outputfilename);

	_printf_("write lock file:\n");
	femmodels[0].parameters->FindParam((void*)&waitonlock,"waitonlock");

	if (waitonlock){
		WriteLockFile(lockname);
	}
		
	_printf_("closing MPI and Petsc\n");
	MPI_Barrier(MPI_COMM_WORLD);

	/*Close MPI libraries: */
	PetscFinalize(); 

	/*end module: */
	MODULEEND();
	
	return 0; //unix success return;
}
