%%
%  create kml polygons for the partition elements.
%
%  [kfold]=kml_part_elems(md,params)
%
%  where the required input is:
%    md            (model, model class object)
%
%  the optional input is:
%    params        (string/numeric, parameter names and values)
%
%  and the optional input is:
%    data          (numeric, element or nodal results data)
%    alt           (numeric, altitude for polygons, default 10000)
%    cmin          (numeric, minimum of color map)
%    cmax          (numeric, maximum of color map)
%    cmap          (char or numeric, colormap definition)
%    prtplt        (char, 'off'/'no' for partition segment plot)
%
%  and the required output is:
%    kfold         (kml_folder, folder of polygon placemarks)
%
function [kfold]=kml_part_elems(varargin)

if ~nargin
    help kml_part_elems
    return
end

%%  process input data

iarg=1;
if (nargin >= 1)
    md=varargin{1};
end
if ~exist('md','var') || isempty(md) || ~isa(md,'model')
    error(['Model ''' inputname(iarg) ''' is unrecognized class ''' class(md) '''.']);
end

%  parameters

iarg=iarg+1;
while (iarg <= nargin-1)
    if ischar(varargin{iarg})
        eval([varargin{iarg} '=varargin{iarg+1};']);
        if (numel(varargin{iarg+1}) <= 20)
            disp([varargin{iarg} '=' any2str(varargin{iarg+1}) ';']);
        else
            disp([varargin{iarg} '=' string_size(varargin{iarg+1}) ' ' class(varargin{iarg+1}) ';']);
        end
        if strcmpi(varargin{iarg},'data')
            cdata=inputname(iarg+1);
        end
    else
        error(['''' any2str(varargin{iarg}) ''' is not a parameter name.']);
    end
    iarg=iarg+2;
end

if exist('data','var') && ~isempty(data)
    if     (numel(data)==md.numberofelements)
        edata=data;
    elseif (numel(data)==md.numberofgrids)
        ndata=data;
        display('Averaging nodal data to element data.');
        edata=zeros(1,md.numberofelements);
        for i=1:size(md.elements,1)
            for j=1:size(md.elements,2)
                edata(i)=edata(i)+ndata(md.elements(i,j));
            end
            edata(i)=edata(i)/size(md.elements,2);
        end
    else
        error(['Data has incorrect number of ' num2str(numel(data)) ' values.']);
    end
end

%  colormap command operates on a figure, so create an invisible one
%  (could also directly call colormaps, e.g. jet(64), but risky)

hfig=figure('Visible','off');
if exist('cmap','var')
    colormap(cmap)
end
cmap=colormap;
close(hfig)
    
if exist('edata','var')
    if ~exist('cmin','var')
        cmin=min(min(edata));
    end
    if ~exist('cmax','var')
        cmax=max(max(edata));
    end
end

if ~exist('alt','var')
    alt=10000;
end

%  write folder for partition elements

if (~exist('prtplt','var') || strncmpi(prtplt,'on' ,2) || strncmpi(prtplt,'y',1)) && ...
    md.npart
    kfold=kml_folder();
    kfold.name      ='Partition Elements';
    kfold.visibility=1;
    kfold.descript  =sprintf('Partitions=%d, Nodes=%d\n',...
        md.npart,md.numberofgrids);
    kfold.feature   ={repmat(kml_placemark(),0,0)};

%  write each partition loop as a polygon placemark

    disp(['Writing ' num2str(md.npart) ' partitions as KML polygons.']);
    epart=md.part(md.elements)+1;
    if exist('ndata','var') || exist('edata','var')
        pdata=zeros(1,md.npart);
        pdata(:)=NaN;
    end
    for k=1:md.npart
        [icol,irow]=find(epart'==k);
        irow=unique(irow);
        elem=md.elements(irow,:);
        nodecon=nodeconnectivity(elem,md.numberofgrids);
        [edgeper,elemper,iloop]=edgeperimeter(elem,nodecon);
        iloop(end+1)=size(edgeper,1)+1;
        if exist('ndata','var')
            pdata(k)=ndata(find(md.part+1==k,1));
        elseif exist('edata','var')
            for i=1:size(epart,1)
                if isempty(find(epart(i,:)~=k,1))
                    pdata(k)=edata(i);
                    break
                end
            end
            if isnan(pdata(k))
                warning('Data for Partition %d is not defined.\n',k)
            end
        end
        for i=1:length(iloop)-1
            kplace=kml_placemark();
            if (length(iloop)-1 > 1)
                kplace.name      =sprintf('Partition %d, Loop %d',k,i);
            else
                kplace.name      =sprintf('Partition %d',k);
            end
            kplace.visibility=1;
            if exist('pdata','var')
                kplace.descript  =sprintf('Partition data: %g',pdata(k));
                imap = fix((pdata(k)-cmin)/(cmax-cmin)*size(cmap,1))+1;
                if     (imap >= 1) && (imap <= size(cmap,1))
                    kplace.styleurl  =sprintf('#MatlabColor%d',imap);
                elseif (pdata(k) == cmax)
                    kplace.styleurl  =sprintf('#MatlabColor%d',size(cmap,1));
                else
                    kplace.styleurl  =sprintf('#BlackLineEmptyPoly');
                end
            else
                kplace.styleurl  =sprintf('#BlackLineRandomPoly');
            end

            kpoly=kml_polygon();
            kpoly.extrude   =1;
            kpoly.altmode   ='relativeToGround';

            kring=kml_linearring();
            kring.coords    =zeros(iloop(i+1)-iloop(i)+1,3);

            for j=iloop(i):iloop(i+1)-1
                [lat,long]=mapxy(md.x(edgeper(j,1)),md.y(edgeper(j,1)),'s');
                kring.coords(j-iloop(i)+1,:)=[long lat alt];
            end
            kring.coords(end,:)=kring.coords(1,:);

            kpoly.outer=kring;
            kplace.geometry=kpoly;
            kfold.feature{1}(end+1)=kplace;
            clear kring kpoly kplace
        end
    end
end

end

