%%
%  create kml polygons for the element mesh.
%
%  []=kml_mesh_elem(fid,md,params)
%
%  where the required input is:
%    fid           (numeric, file ID of .kml file)
%    md            (model, model class object)
%
%  the optional input is:
%    params        (string/numeric, parameter names and values)
%
%  and the optional input is:
%    data          (numeric, element or nodal results data)
%    alt           (numeric, altitude for polygons, default 10000)
%    cmin          (numeric, minimum of color map)
%    cmax          (numeric, maximum of color map)
%    cmap          (char or numeric, colormap definition)
%
function []=kml_mesh_elem(varargin)

if ~nargin
    help kml_mesh_elem
    return
end

%%  process input data

iarg=1;
if (nargin >= 1)
    fid=varargin{1};
end
if ~exist('fid','var') || isempty(fid) || (fid < 0)
    error('File ID ''%d'' must be open.',fid);
end

iarg=iarg+1;
if (nargin >= 2)
    md=varargin{2};
end
if ~exist('md','var') || isempty(md) || ~isa(md,'model')
    error(['Model ''' inputname(iarg) ''' is unrecognized class ''' class(md) '''.']);
end

%  parameters

iarg=iarg+1;
while (iarg <= nargin-1)
    if ischar(varargin{iarg})
        eval([varargin{iarg} '=varargin{iarg+1};']);
        if (numel(varargin{iarg+1}) <= 20)
            disp([varargin{iarg} '=' any2str(varargin{iarg+1}) ';']);
        else
            disp([varargin{iarg} '=' string_size(varargin{iarg+1}) ' ' class(varargin{iarg+1}) ';']);
        end
        if strcmpi(varargin{iarg},'data')
            cdata=inputname(iarg+1);
        end
    else
        error(['''' any2str(varargin{iarg}) ''' is not a parameter name.']);
    end
    iarg=iarg+2;
end

if exist('data','var') && ~isempty(data)
    if     (numel(data)==md.numberofelements)
        edata=data;
    elseif (numel(data)==md.numberofgrids)
        ndata=data;
        display('Averaging nodal data to element data.');
        edata=zeros(1,md.numberofelements);
        for i=1:size(md.elements,1)
            for j=1:size(md.elements,2)
                edata(i)=edata(i)+ndata(md.elements(i,j));
            end
            edata(i)=edata(i)/size(md.elements,2);
        end
    else
        error(['Data has incorrect number of ' num2str(numel(data)) ' values.']);
    end
end

%  colormap command operates on a figure, so create an invisible one
%  (could also directly call colormaps, e.g. jet(64), but risky)

hfig=figure('Visible','off');
if exist('cmap','var')
    colormap(cmap)
end
cmap=colormap;
close(hfig)
    
if exist('edata','var')
    if ~exist('cmin','var')
        cmin=min(min(edata));
    end
    if ~exist('cmax','var')
        cmax=max(max(edata));
    end
end

if ~exist('alt','var')
    alt=10000;
end

%%  write folder for mesh

fprintf(fid,'    <Folder>\n');
if exist('cdata','var') && ~isempty(cdata)
    fprintf(fid,'      <name>Data: %s</name>\n',cdata);
else
    fprintf(fid,'      <name>Mesh</name>\n');
end
fprintf(fid,'      <visibility>1</visibility>\n');
fprintf(fid,'      <description>Elements=%d, Grids=%d</description>\n',...
    md.numberofelements,md.numberofgrids);

%  write each element as a polygon

disp(['Writing ' num2str(size(md.elements,1)) ' tria elements as KML polygons.']);
for i=1:size(md.elements,1)
    fprintf(fid,'      <Placemark>\n');
    fprintf(fid,'        <name>Element %d</name>\n',i);
    fprintf(fid,'        <visibility>1</visibility>\n');
    if exist('edata','var')
        fprintf(fid,'        <description>Element data: %g</description>\n',edata(i));
        imap = fix((edata(i)-cmin)/(cmax-cmin)*size(cmap,1))+1;
        if     (imap >= 1) && (imap <= size(cmap,1))
            fprintf(fid,'        <styleUrl>#MatlabColor%d</styleUrl>\n',imap);
        elseif (edata(i) == cmax)
            fprintf(fid,'        <styleUrl>#MatlabColor%d</styleUrl>\n',size(cmap,1));
        else
            fprintf(fid,'        <styleUrl>#BlackLineEmptyPoly</styleUrl>\n');
        end
    else
        fprintf(fid,'        <styleUrl>#BlackLineRandomPoly</styleUrl>\n');
    end
    fprintf(fid,'        <Polygon>\n');
    fprintf(fid,'          <extrude>1</extrude>\n');
    fprintf(fid,'          <altitudeMode>relativeToGround</altitudeMode>\n');
    fprintf(fid,'          <outerBoundaryIs>\n');
    fprintf(fid,'            <LinearRing>\n');
    fprintf(fid,'              <coordinates>\n');
    for j=1:size(md.elements,2)
        [lat(j),long(j)]=mapxy(md.x(md.elements(i,j)),md.y(md.elements(i,j)),'s');
        fprintf(fid,'                %0.16g,%0.16g,%0.16g\n',long(j),lat(j),alt);
    end
    fprintf(fid,'                %0.16g,%0.16g,%0.16g\n',long(1),lat(1),alt);

    fprintf(fid,'              </coordinates>\n');
    fprintf(fid,'            </LinearRing>\n');
    fprintf(fid,'          </outerBoundaryIs>\n');
    fprintf(fid,'        </Polygon>\n');
    fprintf(fid,'      </Placemark>\n');
end
fprintf(fid,'    </Folder>\n');

end

