/*!\file: solver_thermal_nonlinear.cpp
 * \brief: core of the thermal solution 
 */ 

#include "../toolkits/toolkits.h"
#include "../objects/objects.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../modules/modules.h"

void solver_thermal_nonlinear(FemModel* femmodel){

	/*solution : */
	Vec tg=NULL; 
	Vec tf=NULL; 
	Vec tf_old=NULL; 
	double melting_offset;

	/*intermediary: */
	Mat Kgg=NULL;
	Mat Kff=NULL;
	Mat Kfs=NULL;
	Vec pg=NULL;
	Vec pf=NULL;

	bool converged;
	int constraints_converged;
	int num_unstable_constraints;
	int count;
	int min_thermal_constraints;
	bool reset_penalties;

	/*parameters:*/
	int kflag,pflag;
	bool lowmem=0;
	bool kffpartitioning;

	/*Recover parameters: */
	kflag=1; pflag=1;
	femmodel->parameters->FindParam(&kffpartitioning,KffEnum);
	femmodel->parameters->FindParam(&lowmem,LowmemEnum);
	femmodel->parameters->FindParam(&min_thermal_constraints,MinThermalConstraintsEnum);

	count=1;
	converged=false;

	ISSMPRINTF(VerboseSolution(),"%s\n","starting direct shooting method");
	InputUpdateFromConstantx( femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,reset_penalties,ResetPenaltiesEnum);
	InputUpdateFromConstantx( femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,false,ConvergedEnum);
	for(;;){

		if(kffpartitioning){
			SystemMatricesx(NULL,&Kff, &Kfs, NULL,&pf,&melting_offset,femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters);
			Reduceloadx(pf, Kfs, femmodel->ys,femmodel->parameters); MatFree(&Kfs);
		}
		else{
			SystemMatricesx(&Kgg,NULL, NULL, &pg,NULL, &melting_offset,femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters);
			Reducematrixfromgtofx(&Kff,&Kfs,Kgg,femmodel->nodesets,femmodel->parameters); MatFree(&Kgg);
			Reduceloadfromgtofx(&pf, pg, Kfs, femmodel->ys, femmodel->nodesets,femmodel->parameters);VecFree(&pg); MatFree(&Kfs);
		}

		VecFree(&tf);
		Solverx(&tf, Kff, pf,tf_old, femmodel->parameters);
		VecFree(&tf_old); VecDuplicatePatch(&tf_old,tf);
	
		MatFree(&Kff);VecFree(&pf);VecFree(&tg);

		Mergesolutionfromftogx(&tg, tf,femmodel->ys,femmodel->nodesets,femmodel->parameters);
		InputUpdateFromSolutionx(femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,tg);

		PenaltyConstraintsx(&constraints_converged, &num_unstable_constraints, femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters);

		if (!converged){
			ISSMPRINTF(VerboseConvergence(),"%s%i\n","   #unstable constraints = ",num_unstable_constraints);
			if (num_unstable_constraints <= min_thermal_constraints)converged=true;
		}
		count++;

		InputUpdateFromConstantx( femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,converged,ConvergedEnum);
		
		if(converged)break;
	}

	InputUpdateFromSolutionx(femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,tg);
	InputUpdateFromConstantx( femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,melting_offset,MeltingOffsetEnum);

	/*Free ressources: */
	VecFree(&tg);
	VecFree(&tf);
	VecFree(&tf_old);
}
