/*!\file:  gradient_core.cpp
 * \brief compute inverse method gradient direction.
 */ 

#include "../toolkits/toolkits.h"
#include "../objects/objects.h"
#include "../shared/shared.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "./solutions.h"
#include "../modules/modules.h"
#include "../include/include.h"
#include "../solvers/solvers.h"


void gradient_core(FemModel* femmodel,int step, double search_scalar){ //step defaults to 0, search_scalar defaults to 0
	
	/*parameters: */
	bool    control_steady;
	int     num_controls;
	int    *control_type   = NULL;
	double *optscal_list   = NULL;
	double  optscal,norm_grad;

	/*Intermediaries*/
	Vec new_gradient=NULL;
	Vec gradient=NULL;
	Vec old_gradient=NULL;

	/*retrieve parameters:*/
	femmodel->parameters->FindParam(&control_steady,ControlSteadyEnum);
	femmodel->parameters->FindParam(&num_controls,NumControlsEnum);
	femmodel->parameters->FindParam(&control_type,NULL,ControlTypeEnum);
	femmodel->parameters->FindParam(&optscal_list,NULL,NULL,OptScalEnum);

	/*Compute and norm gradient of all controls*/
	for (int i=0;i<num_controls;i++){

		ISSMPRINTF(VerboseControl(),"   compute gradient of J with respect to %s\n",EnumToString(control_type[i]));
		Gradjx(&gradient, femmodel->elements,femmodel->nodes, femmodel->vertices,femmodel->loads, femmodel->materials,femmodel->parameters, control_type[i]);

		if(control_steady)diagnostic_core(femmodel);

		if (step>0 && search_scalar==0){
			ISSMPRINTF(VerboseControl(),"   orthogonalization\n");
			ControlInputGetGradientx(&old_gradient,femmodel->elements,femmodel->nodes, femmodel->vertices,femmodel->loads, femmodel->materials,femmodel->parameters,control_type[i]);
			Orthx(&new_gradient,gradient,old_gradient); VecFree(&old_gradient); VecFree(&gradient);
		}
		else{ 
			ISSMPRINTF(VerboseControl(),"   normalizing directions\n");
			Orthx(&new_gradient,gradient,NULL); VecFree(&gradient);
		}

		/*Get scaling factor of current control:*/
		VecNorm(new_gradient,NORM_INFINITY,&norm_grad);
		if(norm_grad<=0)    ISSMERROR("||∂J/∂α||∞ = 0    gradient norm of J with respect to %s is zero",EnumToString(control_type[i]));
		if(isnan(norm_grad))ISSMERROR("||∂J/∂α||∞ = NaN  gradient norm of J with respect to %s is NaN" ,EnumToString(control_type[i]));
		if(i==0 || (optscal_list[num_controls*step+i]/norm_grad)<optscal) optscal=optscal_list[num_controls*step+i]/norm_grad;

		/*plug back into inputs: */
		ControlInputSetGradientx(femmodel-> elements,femmodel-> nodes, femmodel-> vertices,femmodel-> loads, femmodel-> materials,  femmodel->parameters,control_type[i],new_gradient);
		VecFree(&new_gradient);
	}

	/*Scale Gradients*/
	for (int i=0;i<num_controls;i++) ControlInputScaleGradientx(femmodel-> elements,femmodel-> nodes, femmodel-> vertices,femmodel-> loads, femmodel-> materials,  femmodel->parameters,control_type[i],optscal);

	/*Clean up and return*/
	xfree((void**)&control_type);
	xfree((void**)&optscal_list);
}
