/*!\file:  gradient_core.cpp
 * \brief compute inverse method gradient direction.
 */ 

#include "../toolkits/toolkits.h"
#include "../objects/objects.h"
#include "../shared/shared.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "./solutions.h"
#include "../modules/modules.h"
#include "../include/include.h"
#include "../solvers/solvers.h"


void gradient_core(FemModel* femmodel,int step, double search_scalar){ //step defaults to 0, search_scalar defaults to 0
	
	/*parameters: */
	bool    control_steady;
	int     verbose;
	int     num_controls;
	int    *control_type   = NULL;
	double *optscal_list   = NULL;

	/*Intermediaries*/
	Vec new_gradient=NULL;
	Vec gradient=NULL;
	Vec old_gradient=NULL;

	/*retrieve parameters:*/
	femmodel->parameters->FindParam(&control_steady,ControlSteadyEnum);
	femmodel->parameters->FindParam(&num_controls,NumControlsEnum);
	femmodel->parameters->FindParam(&control_type,NULL,ControlTypeEnum);
	femmodel->parameters->FindParam(&optscal_list,NULL,NULL,OptScalEnum);
	femmodel->parameters->FindParam(&verbose,VerboseEnum);

	for (int i=0;i<num_controls;i++){

		if(verbose)_printf_("      compute gradient of J with respect to %s\n",EnumToString(control_type[i]));
		Gradjx(&gradient, femmodel->elements,femmodel->nodes, femmodel->vertices,femmodel->loads, femmodel->materials,femmodel->parameters, control_type[i]);

		if(control_steady)diagnostic_core(femmodel);

		if (step>0 && search_scalar==0){
			_printf_("%s","      orthogonalization...\n");
			ControlInputGetGradientx(&old_gradient,femmodel->elements,femmodel->nodes, femmodel->vertices,femmodel->loads, femmodel->materials,femmodel->parameters,control_type[i]);
			Orthx(&new_gradient,gradient,old_gradient);
			VecFree(&old_gradient);
		}
		else{ 
			_printf_("%s","      normalizing directions...\n");
			Orthx(&new_gradient,gradient,NULL);
		}
		VecFree(&gradient);

		/*Scale gradient for current step and current parameter*/
		VecScale(new_gradient,optscal_list[num_controls*step+i]);

		/*plug back into inputs: */
		ControlInputSetGradientx(femmodel-> elements,femmodel-> nodes, femmodel-> vertices,femmodel-> loads, femmodel-> materials,  femmodel->parameters,control_type[i],new_gradient);

		/*Free ressources and return:*/
		VecFree(&new_gradient);
	}

	/*Clean up and return*/
	xfree((void**)&control_type);
	xfree((void**)&optscal_list);
}
