/*!\file: CreateParameters.cpp
 * \brief general driver for creating parameters dataset
 */ 

#include "../../Container/Container.h"
#include "../../toolkits/toolkits.h"
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "../../objects/objects.h"
#include "../../shared/shared.h"
#include "../MeshPartitionx/MeshPartitionx.h"
#include "../../io/io.h"
#include "./ModelProcessorx.h"

void CreateParameters(Parameters** pparameters,IoModel* iomodel,ConstDataHandle iomodel_handle,const int solution_type,int analysis_type,int analysis_counter){
	
	int i;
	
	Parameters* parameters = NULL;
	char**   parameteroutput=NULL;
	char*    descriptor=NULL;
	char*    tag=NULL;
	#ifdef _SERIAL_
	mxArray* pfield=NULL;
	mxArray* pfield2=NULL;
	#endif
	int dummy;
	int numpetscoptions;

	/*petsc options: */
	double*  analyses=NULL;
	char**   petscstrings=NULL;

	if(*pparameters)return; //do not create parameters twice!
		
	/*Initialize dataset: */
	parameters = new Parameters(ParametersEnum);

	parameters->AddObject(new    IntParam(SolutionTypeEnum,solution_type));
	parameters->AddObject(new    IntParam(AnalysisTypeEnum,analysis_type));
	parameters->AddObject(new    IntParam(AnalysisCounterEnum,analysis_counter));
	if (iomodel->dim==2) parameters->AddObject(new IntParam(DimEnum,2));
	else parameters->AddObject(new IntParam(DimEnum,3));
	parameters->AddObject(new BoolParam(IsHutterEnum,iomodel->ishutter));
	parameters->AddObject(new BoolParam(IsMacAyealPattynEnum,iomodel->ismacayealpattyn));
	parameters->AddObject(new BoolParam(IsStokesEnum,iomodel->isstokes));
	parameters->AddObject(new IntParam(VerboseEnum,iomodel->verbose));
	parameters->AddObject(new IntParam(OutputFrequencyEnum,iomodel->output_frequency));
	parameters->AddObject(new DoubleParam(EpsResEnum,iomodel->eps_res));
	parameters->AddObject(new DoubleParam(EpsRelEnum,iomodel->eps_rel));
	parameters->AddObject(new DoubleParam(EpsAbsEnum,iomodel->eps_abs));
	parameters->AddObject(new IntParam(MaxNonlinearIterationsEnum,(IssmInt)iomodel->max_nonlinear_iterations));
	parameters->AddObject(new DoubleParam(EpsVelEnum,iomodel->epsvel));
	parameters->AddObject(new DoubleParam(YtsEnum,iomodel->yts));
	parameters->AddObject(new DoubleParam(DtEnum,iomodel->dt*iomodel->yts)); //Dt is in years in iomodel
	parameters->AddObject(new DoubleParam(NdtEnum,iomodel->ndt*iomodel->yts)); //Ndt is in yeats in iomodel
	parameters->AddObject(new BoolParam(TimeAdaptEnum,iomodel->time_adapt)); 
	parameters->AddObject(new DoubleParam(CflCoefficientEnum,iomodel->cfl_coefficient)); 
	parameters->AddObject(new DoubleParam(PenaltyOffsetEnum,iomodel->penalty_offset));
	parameters->AddObject(new DoubleParam(SparsityEnum,iomodel->sparsity));
	parameters->AddObject(new BoolParam(LowmemEnum,iomodel->lowmem));
	parameters->AddObject(new IntParam(ConnectivityEnum,iomodel->connectivity));
	parameters->AddObject(new DoubleParam(BetaEnum,iomodel->beta));
	parameters->AddObject(new DoubleParam(MeltingPointEnum,iomodel->meltingpoint));
	parameters->AddObject(new DoubleParam(LatentHeatEnum,iomodel->latentheat));
	parameters->AddObject(new DoubleParam(HeatCapacityEnum,iomodel->heatcapacity));
	parameters->AddObject(new IntParam(ArtDiffEnum,iomodel->artdiff));
	parameters->AddObject(new DoubleParam(PenaltyMeltingEnum,iomodel->penalty_melting));
	parameters->AddObject(new IntParam(MinThermalConstraintsEnum,iomodel->min_thermal_constraints));
	parameters->AddObject(new IntParam(MinMechanicalConstraintsEnum,iomodel->min_mechanical_constraints));
	parameters->AddObject(new IntParam(StabilizeConstraintsEnum,iomodel->stabilize_constraints));
	parameters->AddObject(new DoubleParam(StokesReconditioningEnum,iomodel->stokesreconditioning));
	parameters->AddObject(new DoubleParam(ViscosityOvershootEnum,iomodel->viscosity_overshoot));
	parameters->AddObject(new BoolParam(WaitOnLockEnum,iomodel->waitonlock));
	parameters->AddObject(new IntParam(NumberOfElementsEnum,iomodel->numberofelements));
	parameters->AddObject(new BoolParam(KffEnum,iomodel->kff));

	/*Deal with more complex parameters*/
	IoModelFetchData(&iomodel->riftinfo,&iomodel->numrifts,NULL,iomodel_handle,"riftinfo");
	parameters->AddObject(new IntParam(NumRiftsEnum,iomodel->numrifts));
	xfree((void**)&iomodel->riftinfo); 

	parameters->AddObject(new IntParam(NumOutputEnum,iomodel->numoutput));
	if(iomodel->numoutput){
		parameteroutput=(char**)xmalloc(iomodel->numoutput*sizeof(char*));

		#ifdef _SERIAL_
		pfield=mxGetField(iomodel_handle,0,"parameteroutput");
		for(i=0;i<iomodel->numoutput;i++){
			pfield2=mxGetCell(pfield,i);
			FetchData(&descriptor,pfield2);
			parameteroutput[i]=descriptor;
		}
		#else
		tag=(char*)xmalloc((strlen("parameteroutput_i")+1)*sizeof(char));
		for(i=0;i<iomodel->numoutput;i++){
			sprintf(tag,"%s%i","parameteroutput_",i);
			IoModelFetchData(&descriptor,iomodel_handle,tag);
			parameteroutput[i]=descriptor;
		}
		#endif

		/*Ok, we have all the parameter output  descriptors. Build a parameter with it: */
		parameters->AddObject(new StringArrayParam(ParameterOutputEnum,parameteroutput,iomodel->numoutput));
	}

	/*Deal with petsc options: */
	#ifdef _SERIAL_
		pfield=mxGetField(iomodel_handle,0,"petscoptions_analyses");
		FetchData(&analyses,&numpetscoptions,pfield);
		pfield=mxGetField(iomodel_handle,0,"petscoptions_strings");
		petscstrings=(char**)xmalloc(numpetscoptions*sizeof(char*)); //allocate
		for (i=0;i<numpetscoptions;i++){
			pfield2=mxGetCell(pfield,i);
			FetchData(&descriptor,pfield2);
			petscstrings[i]=descriptor;
		}
		parameters->AddObject(new StringArrayParam(PetscOptionsStringsEnum,petscstrings,numpetscoptions));
		parameters->AddObject(new DoubleVecParam(PetscOptionsAnalysesEnum,analyses,numpetscoptions));
	#endif

	/*Free data: {{{1*/
	xfree((void**)&tag);
	for(i=0;i<iomodel->numoutput;i++){
		char* descriptor=parameteroutput[i];
		xfree((void**)&descriptor);
	}
	xfree((void**)&parameteroutput);

	#ifdef _SERIAL_
	xfree((void**)&analyses);
	for(i=0;i<numpetscoptions;i++){
		char* descriptor=petscstrings[i];
		xfree((void**)&descriptor);
	}
	xfree((void**)&petscstrings);
	#endif
	/*}}}*/

	/*Before returning, create parameters in case we are running Qmu or control types runs: */
	CreateParametersControl(&parameters,iomodel,iomodel_handle,solution_type,analysis_type);
	CreateParametersQmu(&parameters,iomodel,iomodel_handle,solution_type,analysis_type);

	/*Assign output pointer: */
	*pparameters=parameters;
}
