/*!\file Riftfront.cpp
 * \brief: implementation of the Riftfront object
 */

/*Headers:*/
/*{{{1*/
#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include <string.h>
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "../../shared/shared.h"
#include "../../include/include.h"
#include "../../modules/ModelProcessorx/ModelProcessorx.h"
#include "../objects.h"
/*}}}*/

/*Element macros*/
#define NUMVERTICES 2
#define NDOF1 1
#define NDOF2 2
#define NDOF3 3
#define NDOF4 4

/*Riftfront constructors and destructor*/
/*FUNCTION Riftfront::Riftfront(){{{1*/
Riftfront::Riftfront(){
	this->inputs=NULL;
	this->parameters=NULL;
	this->hnodes=NULL;
	this->helements=NULL;
	this->hmatpar=NULL;
	this->nodes=NULL;
	this->elements=NULL;
	this->matpar=NULL;
}
/*}}}*/
/*FUNCTION Riftfront::Riftfront(int id, int i, IoModel* iomodel,int analysis_type){{{1*/
Riftfront::Riftfront(int riftfront_id,int i, IoModel* iomodel,int riftfront_analysis_type){

	/*data: */
	int    riftfront_node_ids[2];
	int    riftfront_elem_ids[2];
	int    riftfront_matpar_id;
	int    riftfront_type;
	int    riftfront_fill;
	double riftfront_friction;
	double riftfront_fractionincrement;
	bool   riftfront_shelf;

	/*intermediary: */
	int el1    ,el2;
	int grid1  ,grid2;

	/*Ok, retrieve all the data needed to add a penalty between the two grids: */
	el1=(int)*(iomodel->riftinfo+RIFTINFOSIZE*i+2);
	el2=(int)*(iomodel->riftinfo+RIFTINFOSIZE*i+3); 

	grid1=(int)*(iomodel->riftinfo+RIFTINFOSIZE*i+0); 
	grid2=(int)*(iomodel->riftinfo+RIFTINFOSIZE*i+1);

	/*id: */
	this->id=riftfront_id;
	this->analysis_type=riftfront_analysis_type;

	/*hooks: */
	riftfront_node_ids[0]=iomodel->nodecounter+grid1;
	riftfront_node_ids[1]=iomodel->nodecounter+grid2;
	riftfront_elem_ids[0]=el1;
	riftfront_elem_ids[1]=el2;
	riftfront_matpar_id=iomodel->numberofelements+1; //matlab indexing

	/*Hooks: */
	this->hnodes=new Hook(riftfront_node_ids,2);
	this->helements=new Hook(riftfront_elem_ids,2);
	this->hmatpar=new Hook(&riftfront_matpar_id,1);

	/*computational parameters: */
	this->active=0;
	this->frozen=0;
	this->counter=0;
	this->prestable=0;
	this->penalty_lock=iomodel->penalty_lock;
	this->material_converged=0;
	this->normal[0]=*(iomodel->riftinfo+RIFTINFOSIZE*i+4);
	this->normal[1]=*(iomodel->riftinfo+RIFTINFOSIZE*i+5);
	this->length=*(iomodel->riftinfo+RIFTINFOSIZE*i+6);
	this->fraction=*(iomodel->riftinfo+RIFTINFOSIZE*i+9);

	//intialize inputs, and add as many inputs per element as requested: 
	this->inputs=new Inputs();
		
	riftfront_type=SegmentRiftfrontEnum;
	riftfront_fill = (int)*(iomodel->riftinfo+RIFTINFOSIZE*i+7);
	riftfront_friction=*(iomodel->riftinfo+RIFTINFOSIZE*i+8);
	riftfront_fractionincrement=*(iomodel->riftinfo+RIFTINFOSIZE*i+10);
	riftfront_shelf=(bool)iomodel->gridoniceshelf[grid1-1];

	this->inputs->AddInput(new IntInput(TypeEnum,riftfront_type));
	this->inputs->AddInput(new IntInput(FillEnum,riftfront_fill));
	this->inputs->AddInput(new DoubleInput(FrictionEnum,riftfront_friction));
	this->inputs->AddInput(new DoubleInput(FractionIncrementEnum,riftfront_fractionincrement));
	this->inputs->AddInput(new BoolInput(SegmentOnIceShelfEnum,riftfront_shelf));
	
	//parameters and hooked fields: we still can't point to them, they may not even exist. Configure will handle this.
	this->parameters=NULL;
	this->nodes= NULL;
	this->elements= NULL;
	this->matpar= NULL;
		
}
/*}}}1*/
/*FUNCTION Riftfront::~Riftfront(){{{1*/
Riftfront::~Riftfront(){
	delete inputs;
	this->parameters=NULL;

	delete hnodes;
	delete helements;
	delete hmatpar;
}
/*}}}*/

/*Object virtual functions definitions:*/
/*FUNCTION Riftfront::Echo {{{1*/
void Riftfront::Echo(void){

	Input* input=NULL;
	int fill;
	double friction,fractionincrement;

	
	/*recover some inputs first: */
	input=(Input*)this->inputs->GetInput(FillEnum); input->GetParameterValue(&fill);
	input=(Input*)this->inputs->GetInput(FrictionEnum); input->GetParameterValue(&friction);
	input=(Input*)this->inputs->GetInput(FractionIncrementEnum); input->GetParameterValue(&fractionincrement);

	printf("Riftfront:\n");
	printf("   id: %i\n",id);
	printf("   analysis_type: %s\n",EnumToString(analysis_type));
	printf("   hnodes: %p\n",hnodes);
	printf("   helements: %p\n",helements);
	printf("   hmatpar: %p\n",hmatpar);
	printf("   parameters: %p\n",parameters);
	printf("   inputs: %p\n",inputs);
	printf("   internal parameters: \n");
	printf("   normal: %g|%g\n",normal[0],normal[1]);
	printf("   length: %g\n",length);
	printf("   penalty_lock: %i\n",penalty_lock);
	printf("   active: %s\n",active ? "true":"false");
	printf("   counter: %i\n",counter);
	printf("   prestable: %s\n",prestable ? "true":"false");
	printf("   material_converged: %s\n",material_converged ? "true":"false");
	printf("   fill: %i\n",fill);
	printf("   friction: %i\n",friction);
	printf("   fraction: %i\n",fraction);
	printf("   fractionincrement: %i\n",fractionincrement);
	printf("   frozen: %s\n",frozen ? "true":"false");
		
}
/*}}}1*/
/*FUNCTION Riftfront::DeepEcho{{{1*/
void Riftfront::DeepEcho(void){

	printf("Riftfront:\n");
	printf("   id: %i\n",id);
	printf("   analysis_type: %s\n",EnumToString(analysis_type));
	hnodes->DeepEcho();
	helements->DeepEcho();
	hmatpar->DeepEcho();
	printf("   parameters\n");
	if(parameters)parameters->DeepEcho();
	printf("   inputs\n");
	if(inputs)inputs->DeepEcho();
}
/*}}}*/
/*FUNCTION Riftfront::Id {{{1*/
int    Riftfront::Id(void){ return id; }
/*}}}1*/
/*FUNCTION Riftfront::MyRank {{{1*/
int    Riftfront::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}
/*}}}1*/
/*FUNCTION Riftfront::Marshall {{{1*/
void  Riftfront::Marshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   enum_type=0;
	char* marshalled_inputs=NULL;
	int   marshalled_inputs_size;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum type of Riftfront: */
	enum_type=RiftfrontEnum;

	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);

	/*marshall Riftfront data: */
	memcpy(marshalled_dataset,&id,sizeof(id));marshalled_dataset+=sizeof(id);
	memcpy(marshalled_dataset,&analysis_type,sizeof(analysis_type));marshalled_dataset+=sizeof(analysis_type);
	memcpy(marshalled_dataset,&active,sizeof(active));marshalled_dataset+=sizeof(active);
	memcpy(marshalled_dataset,&normal,sizeof(normal));marshalled_dataset+=sizeof(normal);
	memcpy(marshalled_dataset,&length,sizeof(length));marshalled_dataset+=sizeof(length);
	memcpy(marshalled_dataset,&fraction,sizeof(fraction));marshalled_dataset+=sizeof(fraction);
	memcpy(marshalled_dataset,&frozen,sizeof(frozen));marshalled_dataset+=sizeof(frozen);
	memcpy(marshalled_dataset,&counter,sizeof(counter));marshalled_dataset+=sizeof(counter);
	memcpy(marshalled_dataset,&prestable,sizeof(prestable));marshalled_dataset+=sizeof(prestable);
	memcpy(marshalled_dataset,&penalty_lock,sizeof(penalty_lock));marshalled_dataset+=sizeof(penalty_lock);
	memcpy(marshalled_dataset,&material_converged,sizeof(material_converged));marshalled_dataset+=sizeof(material_converged);

	/*Marshall hooks: */
	hnodes->Marshall(&marshalled_dataset);
	helements->Marshall(&marshalled_dataset);
	hmatpar->Marshall(&marshalled_dataset);

	/*Marshall inputs: */
	marshalled_inputs_size=inputs->MarshallSize();
	marshalled_inputs=inputs->Marshall();
	memcpy(marshalled_dataset,marshalled_inputs,marshalled_inputs_size*sizeof(char));
	marshalled_dataset+=marshalled_inputs_size;

	/*parameters: don't do anything about it. parameters are marshalled somewhere else!*/

	xfree((void**)&marshalled_inputs);

	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Riftfront::MarshallSize {{{1*/
int   Riftfront::MarshallSize(){
	
	return sizeof(id)
		+sizeof(analysis_type)
		+sizeof(active)
		+sizeof(normal)
		+sizeof(length)
		+sizeof(fraction)
		+sizeof(frozen)
		+sizeof(counter)
		+sizeof(prestable)
		+sizeof(penalty_lock)
		+sizeof(material_converged)
		+hnodes->MarshallSize()
		+helements->MarshallSize()
		+hmatpar->MarshallSize()
		+inputs->MarshallSize()
		+sizeof(int); //sizeof(int) for enum type
}
/*}}}*/
/*FUNCTION Riftfront::Demarshall {{{1*/
void  Riftfront::Demarshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   i;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum type, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/

	memcpy(&id,marshalled_dataset,sizeof(id));marshalled_dataset+=sizeof(id);
	memcpy(&analysis_type,marshalled_dataset,sizeof(analysis_type));marshalled_dataset+=sizeof(analysis_type);
	memcpy(&active,marshalled_dataset,sizeof(active));marshalled_dataset+=sizeof(active);
	memcpy(&normal,marshalled_dataset,sizeof(normal));marshalled_dataset+=sizeof(normal);
	memcpy(&length,marshalled_dataset,sizeof(length));marshalled_dataset+=sizeof(length);
	memcpy(&fraction,marshalled_dataset,sizeof(fraction));marshalled_dataset+=sizeof(fraction);
	memcpy(&frozen,marshalled_dataset,sizeof(frozen));marshalled_dataset+=sizeof(frozen);
	memcpy(&counter,marshalled_dataset,sizeof(counter));marshalled_dataset+=sizeof(counter);
	memcpy(&prestable,marshalled_dataset,sizeof(prestable));marshalled_dataset+=sizeof(prestable);
	memcpy(&penalty_lock,marshalled_dataset,sizeof(penalty_lock));marshalled_dataset+=sizeof(penalty_lock);
	memcpy(&material_converged,marshalled_dataset,sizeof(material_converged));marshalled_dataset+=sizeof(material_converged);

	/*demarshall hooks: */
	hnodes=new Hook();    hnodes->Demarshall(&marshalled_dataset);
	helements=new Hook(); helements->Demarshall(&marshalled_dataset);
	hmatpar=new Hook();   hmatpar->Demarshall(&marshalled_dataset);

	/*pointers are garbabe, until configuration is carried out: */
	nodes=NULL;
	elements=NULL;
	matpar=NULL;
	
	/*demarshall inputs: */
	inputs=(Inputs*)DataSetDemarshallRaw(&marshalled_dataset); 

	/*parameters: may not exist even yet, so let Configure handle it: */
	this->parameters=NULL;

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
}
/*}}}*/
/*FUNCTION Riftfront::Enum {{{1*/
int Riftfront::Enum(void){

	return RiftfrontEnum;

}
/*}}}1*/
/*FUNCTION Riftfront::copy {{{1*/
Object* Riftfront::copy() {
	
	Riftfront* riftfront=NULL;

	riftfront=new Riftfront();

	/*copy fields: */
	riftfront->id=this->id;
	riftfront->analysis_type=this->analysis_type;
	if(this->inputs){
		riftfront->inputs=(Inputs*)this->inputs->Copy();
	}
	else{
		riftfront->inputs=new Inputs();
	}
	/*point parameters: */
	riftfront->parameters=this->parameters;

	/*now deal with hooks and objects: */
	riftfront->hnodes=(Hook*)this->hnodes->copy();
	riftfront->helements=(Hook*)this->helements->copy();
	riftfront->hmatpar=(Hook*)this->hmatpar->copy();

	/*corresponding fields*/
	riftfront->nodes   =(Node**)riftfront->hnodes->deliverp();
	riftfront->elements=(Element**)riftfront->helements->deliverp();
	riftfront->matpar  =(Matpar*)riftfront->hmatpar->delivers();

	/*internal data: */
	riftfront->penalty_lock=this->penalty_lock;
	riftfront->active=this->active;
	riftfront->frozen=this->frozen;
	riftfront->counter=this->counter;
	riftfront->prestable=this->prestable;
	riftfront->material_converged=this->material_converged;
	riftfront->normal[0]=this->normal[0];
	riftfront->normal[1]=this->normal[1];
	riftfront->length=this->length;
	riftfront->fraction=this->fraction;
	
	return riftfront;

}
/*}}}*/
		
/*Update virtual functions definitions:*/
/*FUNCTION Riftfront::InputUpdateFromConstant(int constant,int name) {{{1*/
void  Riftfront::InputUpdateFromConstant(int constant,int name){

	/*Check that name is a Riftfront input*/
	if (!IsInput(name)) return;

	/*update input*/
	this->inputs->AddInput(new IntInput(name,constant));

}
/*}}}*/
/*FUNCTION Riftfront::InputUpdateFromConstant(double constant,int name) {{{1*/
void  Riftfront::InputUpdateFromConstant(double constant,int name){

	/*Check that name is a Riftfront input*/
	if (!IsInput(name)) return;

	/*update input*/
	this->inputs->AddInput(new DoubleInput(name,constant));

}
/*}}}*/
/*FUNCTION Riftfront::InputUpdateFromConstant(double constant,int name) {{{1*/
void    Riftfront::InputUpdateFromVector(double* vector, int name, int type){

	/*Check that name is a Riftfront input*/
	if (!IsInput(name)) return;

	/*update input*/
	ISSMERROR("not implemented yet");
	//this->inputs->AddInput(new DoubleInput(name,constant));

}
/*}}}*/


/*Load virtual functions definitions:*/
/*FUNCTION Riftfront::Configure {{{1*/
void  Riftfront::Configure(Elements* elementsin,Loads* loadsin,Nodes* nodesin,Vertices* verticesin,Materials* materialsin,Parameters* parametersin){

	/*Take care of hooking up all objects for this element, ie links the objects in the hooks to their respective 
	 * datasets, using internal ids and offsets hidden in hooks: */
	hnodes->configure(nodesin);
	helements->configure(elementsin);
	hmatpar->configure(materialsin);

	/*Initialize hooked fields*/
	this->nodes   =(Node**)hnodes->deliverp();
	this->elements=(Element**)helements->deliverp();
	this->matpar  =(Matpar*)hmatpar->delivers();

	/*point parameters to real dataset: */
	this->parameters=parametersin;

}
/*}}}*/
/*FUNCTION Riftfront::SetCurrentConfiguration {{{1*/
void  Riftfront::SetCurrentConfiguration(Elements* elementsin,Loads* loadsin,Nodes* nodesin,Vertices* verticesin,Materials* materialsin,Parameters* parametersin){

}
/*}}}*/
/*FUNCTION Riftfront::PenaltyCreateKMatrix {{{1*/
void  Riftfront::PenaltyCreateKMatrix(Mat Kgg,Mat Kff, Mat Kfs,double kmax){

	/*Retrieve parameters: */
	ElementMatrix* Ke=NULL;
	int analysis_type;
	this->parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	switch(analysis_type){
		case DiagnosticHorizAnalysisEnum:
			Ke=PenaltyCreateKMatrixDiagnosticHoriz(kmax);
			break;
		case AdjointHorizAnalysisEnum:
			Ke=PenaltyCreateKMatrixDiagnosticHoriz(kmax);
			break;
		default:
			ISSMERROR("analysis %i (%s) not supported yet",analysis_type,EnumToString(analysis_type));
	}

	/*Add to global Vector*/
	if(Ke){
		Ke->AddToGlobal(Kgg,Kff,Kfs);
		delete Ke;
	}
}
/*}}}1*/
/*FUNCTION Riftfront::PenaltyCreatePVector {{{1*/
void  Riftfront::PenaltyCreatePVector(Vec pg,Vec pf,double kmax){

	/*Retrieve parameters: */
	ElementVector* pe=NULL;
	int analysis_type;
	this->parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	switch(analysis_type){
		case DiagnosticHorizAnalysisEnum:
			pe=PenaltyCreatePVectorDiagnosticHoriz(kmax);
			break;
		case AdjointHorizAnalysisEnum:
			/*No penalty applied on load vector*/
			break;
		default:
			ISSMERROR("analysis %i (%s) not supported yet",analysis_type,EnumToString(analysis_type));
	}

	/*Add to global Vector*/
	if(pe){
		pe->AddToGlobal(pg,pf);
		delete pe;
	}
}
/*}}}1*/
/*FUNCTION Riftfront::CreateKMatrix {{{1*/
void  Riftfront::CreateKMatrix(Mat Kgg,Mat Kff, Mat Kfs){
	/*do nothing: */
	return;
}
/*}}}1*/
/*FUNCTION Riftfront::CreatePVector {{{1*/
void  Riftfront::CreatePVector(Vec pg,Vec pf){
	/*do nothing: */
	return;
}
/*}}}1*/
/*FUNCTION Riftfront::InAnalysis{{{1*/
bool Riftfront::InAnalysis(int in_analysis_type){
	if (in_analysis_type==this->analysis_type) return true;
	else return false;
}
/*}}}*/

/*Riftfront numerics*/
/*FUNCTION Riftfront::PenaltyCreateKMatrixDiagnosticHoriz {{{1*/
ElementMatrix* Riftfront::PenaltyCreateKMatrixDiagnosticHoriz(double kmax){

	const int   numdof = NDOF2*NUMVERTICES;
	int         i,j;
	int         dofs[1]             = {0};
	double      Ke_gg[4][4];
	double      thickness;
	double      h[2];
	double      penalty_offset;
	double      friction;

	/*Objects: */
	Tria       *tria1               = NULL;
	Tria       *tria2               = NULL;

	/*enum of element? */
	if(elements[0]->Enum()!=TriaEnum)ISSMERROR(" only Tria element allowed for Riftfront load!");
	tria1=(Tria*)elements[0];
	tria2=(Tria*)elements[1];

	/*Initialize Element Matrix*/
	if(!this->active) return NULL;
	ElementMatrix* Ke=new ElementMatrix(nodes,NUMVERTICES,this->parameters);

	/*Get some parameters: */
	this->parameters->FindParam(&penalty_offset,PenaltyOffsetEnum);
	this->inputs->GetParameterValue(&friction,FrictionEnum);
	tria1->GetParameterValue(&h[0],nodes[0],ThicknessEnum);
	tria2->GetParameterValue(&h[1],nodes[1],ThicknessEnum);
	if (h[0]!=h[1])ISSMERROR(" different thicknesses not supported for rift fronts");
	thickness=h[0];

	/*There is contact, we need to constrain the normal velocities (zero penetration), and the 
	 *contact slip friction. */

	/*From Peter Wriggers book (Computational Contact Mechanics, p191): */
	Ke->values[0*numdof+0]+= +pow(normal[0],2)*kmax*pow(10,penalty_offset);
	Ke->values[0*numdof+1]+= +normal[0]*normal[1]*kmax*pow(10,penalty_offset);
	Ke->values[0*numdof+2]+= -pow(normal[0],2)*kmax*pow(10,penalty_offset);
	Ke->values[0*numdof+3]+= -normal[0]*normal[1]*kmax*pow(10,penalty_offset);

	Ke->values[1*numdof+0]+= +normal[0]*normal[1]*kmax*pow(10,penalty_offset);
	Ke->values[1*numdof+1]+= +pow(normal[1],2)*kmax*pow(10,penalty_offset);
	Ke->values[1*numdof+2]+= -normal[0]*normal[1]*kmax*pow(10,penalty_offset);
	Ke->values[1*numdof+3]+= -pow(normal[1],2)*kmax*pow(10,penalty_offset);

	Ke->values[2*numdof+0]+= -pow(normal[0],2)*kmax*pow(10,penalty_offset);
	Ke->values[2*numdof+1]+= -normal[0]*normal[1]*kmax*pow(10,penalty_offset);
	Ke->values[2*numdof+2]+= +pow(normal[0],2)*kmax*pow(10,penalty_offset);
	Ke->values[2*numdof+3]+= +normal[0]*normal[1]*kmax*pow(10,penalty_offset);

	Ke->values[3*numdof+0]+= -normal[0]*normal[1]*kmax*pow(10,penalty_offset);
	Ke->values[3*numdof+1]+= -pow(normal[1],2)*kmax*pow(10,penalty_offset);
	Ke->values[3*numdof+2]+= +normal[0]*normal[1]*kmax*pow(10,penalty_offset);
	Ke->values[3*numdof+3]+= +pow(normal[1],2)*kmax*pow(10,penalty_offset);

	/*Now take care of the friction: of type sigma=frictiontangent_velocity2-tangent_velocity1)*/

	Ke->values[0*numdof+0]+= +pow(normal[1],2)*thickness*length*friction;
	Ke->values[0*numdof+1]+= -normal[0]*normal[1]*thickness*length*friction;
	Ke->values[0*numdof+2]+= -pow(normal[1],2)*thickness*length*friction;
	Ke->values[0*numdof+3]+= +normal[0]*normal[1]*thickness*length*friction;

	Ke->values[1*numdof+0]+= -normal[0]*normal[1]*thickness*length*friction;
	Ke->values[1*numdof+1]+= +pow(normal[0],2)*thickness*length*friction;
	Ke->values[1*numdof+2]+= +normal[0]*normal[1]*thickness*length*friction;
	Ke->values[1*numdof+3]+= -pow(normal[0],2)*thickness*length*friction;

	Ke->values[2*numdof+0]+= -pow(normal[1],2)*thickness*length*friction;
	Ke->values[2*numdof+1]+= +normal[0]*normal[1]*thickness*length*friction;
	Ke->values[2*numdof+2]+= +pow(normal[1],2)*thickness*length*friction;
	Ke->values[2*numdof+3]+= -normal[0]*normal[1]*thickness*length*friction;

	Ke->values[3*numdof+0]+= +normal[0]*normal[1]*thickness*length*friction;
	Ke->values[3*numdof+1]+= -pow(normal[0],2)*thickness*length*friction;
	Ke->values[3*numdof+2]+= -normal[0]*normal[1]*thickness*length*friction;
	Ke->values[3*numdof+3]+= +pow(normal[0],2)*thickness*length*friction;

	/*Clean up and return*/
	return Ke;
}
/*}}}1*/
/*FUNCTION Riftfront::PenaltyCreatePVectorDiagnosticHoriz {{{1*/
ElementVector* Riftfront::PenaltyCreatePVectorDiagnosticHoriz(double kmax){

	const int   numdof = NDOF2*NUMVERTICES;
	int         i,j;
	double      rho_ice;
	double      rho_water;
	double      gravity;
	double      thickness;
	double      h[2];
	double      bed;
	double      b[2];
	double      pressure;
	double      pressure_litho;
	double      pressure_air;
	double      pressure_melange;
	double      pressure_water;
	int         fill;
	bool        shelf;

	/*Objects: */
	Tria       *tria1               = NULL;
	Tria       *tria2               = NULL;

	/*enum of element? */
	if(elements[0]->Enum()!=TriaEnum)ISSMERROR(" only Tria element allowed for Riftfront load!");
	tria1=(Tria*)elements[0];
	tria2=(Tria*)elements[1];

	/*Initialize Element Matrix*/
	if(this->active) return NULL; /*The penalty is active. No loads implied here.*/
	ElementVector* pe=new ElementVector(nodes,NUMVERTICES,this->parameters);

	/*Get some inputs: */
	this->inputs->GetParameterValue(&fill,FillEnum);
	this->inputs->GetParameterValue(&shelf,SegmentOnIceShelfEnum);
	rho_ice=matpar->GetRhoIce();
	rho_water=matpar->GetRhoWater();
	gravity=matpar->GetG();
	tria1->GetParameterValue(&h[0],nodes[0],ThicknessEnum);
	tria2->GetParameterValue(&h[1],nodes[1],ThicknessEnum);
	if (h[0]!=h[1])ISSMERROR(" different thicknesses not supported for rift fronts");
	thickness=h[0];
	tria1->GetParameterValue(&b[0],nodes[0],BedEnum);
	tria2->GetParameterValue(&b[1],nodes[1],BedEnum);
	if (b[0]!=b[1])ISSMERROR(" different beds not supported for rift fronts");
	bed=b[0];

	/*Ok, this rift is opening. We should put loads on both sides of the rift flanks. Because we are dealing with contact mechanics, 
	 * and we want to avoid zigzagging of the loads, we want lump the loads onto grids, not onto surfaces between grids.:*/

	/*Ok, to compute the pressure, we are going to need material properties, thickness and bed for the two grids. We assume those properties to 
	 * be the same across the rift.: */

	/*Ok, now compute the pressure (in norm) that is being applied to the flanks, depending on the type of fill: */
	if(fill==WaterEnum){
		if(shelf){
			/*We are on an ice shelf, hydrostatic equilibrium is used to determine the pressure for water fill: */
			pressure=rho_ice*gravity*pow(thickness,(double)2)/(double)2  - rho_water*gravity*pow(bed,(double)2)/(double)2; 
		}
		else{
			//We are on an icesheet, we assume the water column fills the entire front: */
			pressure=rho_ice*gravity*pow(thickness,(double)2)/(double)2  - rho_water*gravity*pow(thickness,(double)2)/(double)2; 
		}
	}
	else if(fill==AirEnum){
		pressure=rho_ice*gravity*pow(thickness,(double)2)/(double)2;   //icefront on an ice sheet, pressure imbalance ice vs air.
	}
	else if(fill==IceEnum){ //icefront finding itself against another icefront (pressure imbalance is fully compensated, ice vs ice)
		pressure=0;
	}
	else if(fill==MelangeEnum){ //icefront finding itself against another icefront (pressure imbalance is fully compensated, ice vs ice)

		if(!shelf) ISSMERROR("%s%i%s","fill type ",fill," not supported on ice sheets yet.");

		pressure_litho=rho_ice*gravity*pow(thickness,(double)2)/(double)2;
		pressure_air=0;
		pressure_melange=rho_ice*gravity*pow(fraction*thickness,(double)2)/(double)2;
		pressure_water=1.0/2.0*rho_water*gravity*  ( pow(bed,2.0)-pow(rho_ice/rho_water*fraction*thickness,2.0) );

		pressure=pressure_litho-pressure_air-pressure_melange-pressure_water;
	}
	else{
		ISSMERROR("%s%i%s","fill type ",fill," not supported yet.");
	}

	/*Ok, add contribution to first grid, along the normal i==0: */
	for (j=0;j<2;j++){
		pe->values[j]+=pressure*normal[j]*length;
	}

	/*Add contribution to second grid, along the opposite normal: i==1 */
	for (j=0;j<2;j++){
		pe->values[2+j]+= -pressure*normal[j]*length;
	}	

	/*Clean up and return*/
	return pe;
}
/*}}}1*/
/*FUNCTION Riftfront::Constrain {{{1*/
#define _ZIGZAGCOUNTER_

int Riftfront::Constrain(int* punstable){

	const int   numgrids        = 2;
	double      max_penetration;
	double      penetration;
	int         activate;
	int         found;
	int         unstable;
	double      vx1;
	double      vy1;
	double      vx2;
	double      vy2;
	double      fractionincrement;

	/*Objects: */
	Tria       *tria1           = NULL;
	Tria       *tria2           = NULL;

	/*enum of element? */
	if(elements[0]->Enum()!=TriaEnum)ISSMERROR(" only Tria element allowed for Riftfront load!");

	/*recover elements on both side of rift: */
	tria1=(Tria*)elements[0];
	tria2=(Tria*)elements[1];

	/*Is this constraint frozen? In which case we don't touch: */
	if (this->frozen){
		*punstable=0;
		return 1;
	}

	/*recover parameters: */
	this->inputs->GetParameterValue(&fractionincrement,FractionIncrementEnum);

	/*First recover velocity: */
	tria1->GetParameterValue(&vx1,nodes[0],VxEnum);
	tria2->GetParameterValue(&vx2,nodes[1],VxEnum);
	tria1->GetParameterValue(&vy1,nodes[0],VyEnum);
	tria2->GetParameterValue(&vy2,nodes[1],VyEnum);

	/*Node 1 faces node 2, compute penetration of 2 into 1 (V2-V1).N (with N normal vector, and V velocity vector: */
	penetration=(vx2-vx1)*normal[0]+(vy2-vy1)*normal[1];

	/*activation: */
	if(penetration<0)activate=1;
	else  activate=0;

	/*Here, we try to avoid zigzaging. When a penalty activates and deactivates for more than penalty_lock times, 
	 * we increase the fraction of melange:*/
	if(this->counter>this->penalty_lock){
		/*reset counter: */
		this->counter=0;
		/*increase melange fraction: */
		this->fraction+=fractionincrement;
		if (this->fraction>1)this->fraction=(double)1.0;
		//printf("riftfront %i fraction: %g\n",this->Id(),this->fraction);
	}

	//Figure out stability of this penalty
	if(this->active==activate){
		unstable=0;
	}
	else{
		unstable=1;
		this->counter++;
	}

	//Set penalty flag
	this->active=activate;

	//if ((penetration>0) && (this->active==1))printf("Riftfront %i wants to be released\n",Id());

	/*assign output pointer: */
	*punstable=unstable;
}
/*}}}1*/
/*FUNCTION Riftfront::FreezeConstraints{{{1*/
void   Riftfront::FreezeConstraints(void){

	/*Just set frozen flag to 1: */
	this->frozen=1;

}
/*}}}1*/
/*FUNCTION Riftfront::IsFrozen{{{1*/
bool   Riftfront::IsFrozen(void){

	/*Just set frozen flag to 1: */
	if(this->frozen)return 1;
	else return 0;
}
/*}}}1*/
/*FUNCTION Riftfront::IsMaterialStable {{{1*/
int   Riftfront::IsMaterialStable(void){

	int found=0;
	double converged=0;

	this->inputs->GetParameterValue(&converged,ConvergedEnum);

	if(converged){
		/*ok, material non-linearity has converged. If that was already the case, we keep 
		 * constraining the rift front. If it was not, and this is the first time the material 
		 * has converged, we start constraining now!: */
		this->material_converged=1;
	}

	return this->material_converged;
}
/*}}}1*/
/*FUNCTION Riftfront::MaxPenetration {{{1*/
int   Riftfront::MaxPenetration(double* ppenetration){

	const int     numgrids=2;
	double        max_penetration;
	double        penetration=0;
	int           found;
	double      vx1;
	double      vy1;
	double      vx2;
	double      vy2;

	/*Objects: */
	Tria       *tria1           = NULL;
	Tria       *tria2           = NULL;

	/*enum of element? */
	if(elements[0]->Enum()!=TriaEnum)ISSMERROR(" only Tria element allowed for Riftfront load!");

	/*recover elements on both side of rift: */
	tria1=(Tria*)elements[0];
	tria2=(Tria*)elements[1];

	//initialize: 
	penetration=-1;

	/*recover velocity: */
	tria1->GetParameterValue(&vx1,nodes[0],VxEnum);
	tria2->GetParameterValue(&vx2,nodes[1],VxEnum);
	tria1->GetParameterValue(&vy1,nodes[0],VyEnum);
	tria2->GetParameterValue(&vy2,nodes[1],VyEnum);

	/*Grid 1 faces grid2, compute penetration of 2 into 1 (V2-V1).N (with N normal vector, and V velocity vector: */
	penetration=(vx2-vx1)*normal[0]+(vy2-vy1)*normal[1];

	/*Now, we return penetration only if we are active!: */
	if(this->active==0)penetration=-1;

	/*If we are zigzag locked, same thing: */
	if(this->counter>this->penalty_lock)penetration=-1;
	
	/*assign output pointer: */
	*ppenetration=penetration;

}
/*}}}1*/
/*FUNCTION Riftfront::OutputProperties {{{1*/
void  Riftfront::OutputProperties(Vec riftproperties){

	int row_id=0;
	double value;

	/*recover id of penalty: */
	row_id=this->Id()-1; //c indexing, ids were matlab indexed
	value=(double)this->fraction;

	/*Plug id and fraction  into riftproperties matrix: */
	VecSetValues(riftproperties,1,&row_id,&value,INSERT_VALUES);
}
/*}}}1*/
/*FUNCTION Riftfront::Penetration {{{1*/
int   Riftfront::Penetration(double* ppenetration){

	double    vx1;
	double    vy1;
	double    vx2;
	double    vy2;

	double    penetration;
	int       found;

	/*Objects: */
	Tria     *tria1       = NULL;
	Tria     *tria2       = NULL;

	/*enum of element? */
	if(elements[0]->Enum()!=TriaEnum)ISSMERROR(" only Tria element allowed for Riftfront load!");

	/*recover elements on both side of rift: */
	tria1=(Tria*)elements[0];
	tria2=(Tria*)elements[1];

	/*First recover velocity: */
	tria1->GetParameterValue(&vx1,nodes[0],VxEnum);
	tria2->GetParameterValue(&vx2,nodes[1],VxEnum);
	tria1->GetParameterValue(&vy1,nodes[0],VyEnum);
	tria2->GetParameterValue(&vy2,nodes[1],VyEnum);

	/*Node 1 faces node 2, compute penetration of 2 into 1 (V2-V1).N (with N normal vector, and V velocity vector: */
	penetration=(vx2-vx1)*normal[0]+(vy2-vy1)*normal[1];

	/*Now, we return penetration only if we are active!: */
	if(this->active==0)penetration=0;
	
	/*assign output pointer: */
	*ppenetration=penetration;

}
/*}}}1*/
/*FUNCTION Riftfront::PotentialUnstableConstraint {{{1*/
int   Riftfront::PotentialUnstableConstraint(int* punstable){


	const int   numgrids        = 2;
	double      max_penetration;
	double      penetration;
	int         activate;
	int         unstable;
	int         found;
	double      vx1;
	double      vy1;
	double      vx2;
	double      vy2;

	/*Objects: */
	Tria       *tria1           = NULL;
	Tria       *tria2           = NULL;

	/*enum of element? */
	if(elements[0]->Enum()!=TriaEnum)ISSMERROR(" only Tria element allowed for Riftfront load!");

	/*recover elements on both side of rift: */
	tria1=(Tria*)elements[0];
	tria2=(Tria*)elements[1];

	/*First recover velocity: */
	tria1->GetParameterValue(&vx1,nodes[0],VxEnum);
	tria2->GetParameterValue(&vx2,nodes[1],VxEnum);
	tria1->GetParameterValue(&vy1,nodes[0],VyEnum);
	tria2->GetParameterValue(&vy2,nodes[1],VyEnum);

	/*Node 1 faces node 2, compute penetration of 2 into 1 (V2-V1).N (with N normal vector, and V velocity vector: */
	penetration=(vx2-vx1)*normal[0]+(vy2-vy1)*normal[1];

	/*Ok, we are looking for positive penetration in an active constraint: */
	if(this->active){
		if (penetration>=0){
			unstable=1;
		}
		else{
			unstable=0;
		}
	}
	else{
		unstable=0;
	}

	/*assign output pointer: */
	*punstable=unstable;
}
/*}}}1*/
/*FUNCTION Riftfront::PreConstrain {{{1*/
int   Riftfront::PreConstrain(int* punstable){

	const int   numgrids    = 2;
	double      penetration;
	int         unstable;
	int         found;
	double      vx1;
	double      vy1;
	double      vx2;
	double      vy2;

	/*Objects: */
	Tria       *tria1       = NULL;
	Tria       *tria2       = NULL;

	/*enum of element? */
	if(elements[0]->Enum()!=TriaEnum)ISSMERROR(" only Tria element allowed for Riftfront load!");

	/*recover elements on both side of rift: */
	tria1=(Tria*)elements[0];
	tria2=(Tria*)elements[1];

	/*First recover velocity: */
	tria1->GetParameterValue(&vx1,nodes[0],VxEnum);
	tria2->GetParameterValue(&vx2,nodes[1],VxEnum);
	tria1->GetParameterValue(&vy1,nodes[0],VyEnum);
	tria2->GetParameterValue(&vy2,nodes[1],VyEnum);

	/*Node 1 faces node 2, compute penetration of 2 into 1 (V2-V1).N (with N normal vector, and V velocity vector: */
	penetration=(vx2-vx1)*normal[0]+(vy2-vy1)*normal[1];

	/*Ok, we are preconstraining here. Ie, anything that penetrates is constrained until stability of the entire set 
	 * of constraints is reached.: */
	if(penetration<0){
		if (!this->active){
			/*This is the first time penetration happens: */
			this->active=1;
			unstable=1;
		}
		else{
			/*This constraint was already active: */
			this->active=1;
			unstable=0;
		}
	}
	else{
		/*No penetration happening. : */
		if (!this->active){
			/*This penalty was not active, and no penetration happening. Do nonthing: */
			this->active=0;
			unstable=0; 
		}
		else{
			/*Ok, this penalty wants to get released. But not now, this is preconstraint, not constraint: */
			this->active=1;
			unstable=0;
		}
	}

	/*assign output pointer: */
	*punstable=unstable;
}
/*}}}1*/
/*FUNCTION Riftfront::PreStable {{{1*/
bool  Riftfront::PreStable(){
	return prestable;
}
/*}}}1*/
/*FUNCTION Riftfront::SetPreStable {{{1*/
void Riftfront::SetPreStable(){
	prestable=1;
}
/*}}}1*/
/*FUNCTION Riftfront::IsInput{{{1*/
bool Riftfront::IsInput(int name){
	if (
				name==ConvergedEnum ||
				name==ThicknessEnum ||
				name==SurfaceEnum ||
				name==BedEnum 
		){
		return true;
	}
	else return false;
}
/*}}}*/
