function femmodel=control_core(femmodel)
%CONTROL_CORE - compute the core inversion
%
%   Usage:
%      femmodel=control_core(femmodel);
%

	%recover parameters common to all solutions
	verbose=femmodel.parameters.Verbose;
	nsteps=femmodel.parameters.NSteps;
	control_type=femmodel.parameters.ControlType;
	solution_type=femmodel.parameters.SolutionType;
	responses=femmodel.parameters.CmResponses;
	optscal=femmodel.parameters.OptScal;
	maxiter=femmodel.parameters.MaxIter;
	cm_jump=femmodel.parameters.CmJump;
	eps_cm=femmodel.parameters.EpsCm;
	tolx=femmodel.parameters.TolX;
	cm_gradient=femmodel.parameters.CmGradient;
	dim=femmodel.parameters.Dim;
	isstokes=femmodel.parameters.IsStokes;

	%Initialise options with tolerance and maxiter
	options.TolX=femmodel.parameters.TolX;
	options.MaxIter=femmodel.parameters.MaxIter;

	%Initialize misfits with a vector of zeros
	J=zeros(nsteps,1);
	search_scalar=0;

	%Get core from solution type
	solutioncore=CorePointerFromSolutionEnum(solution_type);
	adjointcore=AdjointCorePointerFromSolutionEnum(solution_type);

	%Launch once a complete solution to set up all inputs
	disp('      preparing initial solution');
	if isstokes,
		eval(['femmodel=' solutioncore '(femmodel);']);
	end

	for n=1:nsteps,

		disp(sprintf('\n%s%s%s%s\n',['   control method step ' num2str(n) '/' num2str(femmodel.parameters.NSteps)]));
		[femmodel.elements,femmodel.loads]=InputUpdateFromConstant(femmodel.elements,femmodel.nodes,femmodel.vertices,femmodel.loads,femmodel.materials,femmodel.parameters,responses(n),CmResponseEnum);

		%In case we are running a steady state control method, compute new temperature field using new parameter distribution: 
		if (solution_type==SteadystateSolutionEnum)
			femmodel=steadystate_core(femmodel);
		end

		displaystring(verbose,'\n%s',['      compute adjoint state...']);
		eval(['femmodel=' adjointcore '(femmodel);']);

		displaystring(verbose,'\n%s',['      computing gradJ...']);
		femmodel=gradient_core(femmodel,n,search_scalar);

		%Return gradient if asked
		if cm_gradient,
			femmodel.elements=InputToResult(femmodel.elements,femmodel.nodes,femmodel.vertices,femmodel.loads,femmodel.materials,femmodel.parameters,GradientEnum);
			return;
		end

		displaystring(verbose,'\n%s',['      optimizing along gradient direction...']);
		[search_scalar J(n)]=ControlOptimization('objectivefunctionC',0,1,options,femmodel,n,femmodel.parameters);

		displaystring('\n%s',['      updating parameter using optimized search scalar:']);
		[femmodel.elements,femmodel.nodes,femmmodel.vertices,femmodel.loads,femmodel.materials,femmodel.parameters]=InputControlUpdate(femmodel.elements,femmodel.nodes,femmodel.vertices,femmodel.loads,femmodel.materials,femmodel.parameters,search_scalar*optscal(n),1);

		disp(['      value of misfit J after optimization #' num2str(n) ':' num2str(J(n))]);

		%Has convergence been reached?
		converged=controlconvergence(J,responses,eps_cm,n);
		if converged,
			break;
		end

	end

	%generate output
	displaystring(verbose,'\n%s',['      preparing final velocity solution...']);
	femmodel.parameters.ControlAnalysis=0;
	eval(['femmodel=' solutioncore '(femmodel);']);

	%Some results not computed by diagnostic or steadystate
	femmodel.elements=InputToResult(femmodel.elements,femmodel.nodes,femmodel.vertices,femmodel.loads,femmodel.materials,femmodel.parameters,control_type);
	femmodel.results=AddExternalResult(femmodel.results,JEnum,J);
	femmodel.results=AddExternalResult(femmodel.results,ControlTypeEnum,control_type);

end %end function
