/*!\file:  prognostic.cpp
 * \brief: prognostic solution
 */ 

#include "../issm.h"
#include "./parallel.h"

#undef __FUNCT__ 
#define __FUNCT__ "prognostic"

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif


int main(int argc,char* *argv){
	
	/*I/O: */
	FILE* fid=NULL;
	char* inputfilename=NULL;
	char* outputfilename=NULL;
	char* lockname=NULL;
	int   numberofnodes;
	int waitonlock=0;

	FemModel fem;
	Vec h_g=NULL;
	Vec u_g=NULL;
	double* u_g_serial=NULL;
	double* h_g_initial=NULL;
	double* melting_g=NULL;
	double* accumulation_g=NULL;
	double  dt;

	
	ParameterInputs* inputs=NULL;
	Param*  param=NULL;

	MODULEBOOT();

	#if !defined(_PARALLEL_) || (defined(_PARALLEL_) && !defined(_HAVE_PETSC_))
	throw ErrorException(__FUNCT__," parallel executable was compiled without support of parallel libraries!");
	#endif

	PetscInitialize(&argc,&argv,(char *)0,"");  

	/*Size and rank: */
	MPI_Comm_rank(MPI_COMM_WORLD,&my_rank);  
	MPI_Comm_size(MPI_COMM_WORLD,&num_procs); 

	_printf_("recover , input file name and output file name:\n");
	inputfilename=argv[2];
	outputfilename=argv[3];
	lockname=argv[4];

	/*Open handle to data on disk: */
	fid=pfopen(inputfilename,"rb");

	_printf_("read and create finite element model:\n");
	CreateFemModel(&fem,fid,"prognostic","");

	//retrieve parameters used to fill inputs
	fem.parameters->FindParam((void*)&u_g_serial,"u_g");
	fem.parameters->FindParam((void*)&h_g_initial,"h_g");
	fem.parameters->FindParam((void*)&melting_g,"m_g");
	fem.parameters->FindParam((void*)&accumulation_g,"a_g");
	fem.parameters->FindParam((void*)&dt,"dt");
	fem.parameters->FindParam((void*)&numberofnodes,"numberofnodes");

	_printf_("depth averaging velocity...");
	u_g=SerialToVec(u_g_serial,numberofnodes*3); xfree((void**)&u_g_serial);//vx,vy and vz should be present at this point.
	VelocityDepthAveragex( u_g, fem.elements,fem.nodes, fem.loads, fem.materials);

	_printf_("initialize inputs:\n");
	inputs=new ParameterInputs;
	inputs->Add("velocity_average",u_g,3,numberofnodes);
	inputs->Add("thickness",h_g_initial,1,numberofnodes);
	inputs->Add("melting",melting_g,1,numberofnodes);
	inputs->Add("accumulation",accumulation_g,1,numberofnodes);
	inputs->Add("dt",dt);

	/*lighten up on parameters : to be done */

	_printf_("call computational core:\n");
	diagnostic_core_linear(&h_g,&fem,inputs,PrognosticAnalysisEnum(),NoneAnalysisEnum());

	_printf_("extrude computed thickness on all layers:\n");
	ThicknessExtrudex( h_g, fem.elements,fem.nodes, fem.loads, fem.materials);

	_printf_("write results to disk:\n");
	OutputPrognostic(h_g,&fem,outputfilename);

	_printf_("write lock file:\n");
	fem.parameters->FindParam((void*)&waitonlock,"waitonlock");
	if (waitonlock){
		WriteLockFile(lockname);
	}
		
	_printf_("closing MPI and Petsc\n");
	MPI_Barrier(MPI_COMM_WORLD);

	/*Close MPI libraries: */
	PetscFinalize(); 


	/*end module: */
	MODULEEND();
	
	return 0; //unix success return;
}
