/*!\file Tria.cpp
 * \brief: implementation of the Tria object
 */

/*Headers:*/
/*{{{1*/
#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include <string.h>
#include "../objects.h"
#include "../../shared/shared.h"
#include "../../Container/Container.h"
#include "../../include/include.h"
/*}}}*/

/*Element macros*/
#define NUMVERTICES 3
#define NDOF1 1
#define NDOF2 2
#define NDOF3 3
#define NDOF4 4

/*Tria constructors and destructor*/
/*FUNCTION Tria::Tria(){{{1*/
Tria::Tria(){

	this->nodes=NULL;
	this->matice=NULL;
	this->matpar=NULL;
	this->inputs=NULL;
	this->parameters=NULL;
	this->results=NULL;

}
/*}}}*/
/*FUNCTION Tria::Tria(int id, int index, IoModel* iomodel,int nummodels){{{1*/
Tria::Tria(int tria_id, int index, IoModel* iomodel,int nummodels)
	:TriaRef(nummodels)
	,TriaHook(nummodels,index+1,iomodel->numberofelements+1){
	/*id: */
	this->id=tria_id;

	//this->parameters: we still can't point to it, it may not even exist. Configure will handle this.
	this->parameters=NULL;

	/*intialize inputs and results: */
	this->inputs=new Inputs();
	this->results=new Results();

	/*initialize pointers:*/
	this->nodes=NULL;
	this->matice=NULL;
	this->matpar=NULL;

}
/*}}}*/
/*FUNCTION Tria::~Tria(){{{1*/
Tria::~Tria(){
	delete inputs;
	delete results;
	this->parameters=NULL;
}
/*}}}*/

/*Object virtual functions definitions:*/
/*FUNCTION Tria::copy {{{1*/
Object* Tria::copy() {

	int i;
	Tria* tria=NULL;

	tria=new Tria();

	//deal with TriaRef mother class
	tria->element_type_list=(int*)xmalloc(this->numanalyses*sizeof(int));
	for(i=0;i<this->numanalyses;i++) tria->element_type_list[i]=this->element_type_list[i];

	//deal with TriaHook mother class
	tria->numanalyses=this->numanalyses;
	tria->hnodes=new Hook*[tria->numanalyses];
	for(i=0;i<tria->numanalyses;i++)tria->hnodes[i]=(Hook*)this->hnodes[i]->copy();
	tria->hmatice=(Hook*)this->hmatice->copy();
	tria->hmatpar=(Hook*)this->hmatpar->copy();

	/*deal with Tria fields: */
	tria->id=this->id;
	if(this->inputs){
		tria->inputs=(Inputs*)this->inputs->Copy();
	}
	else{
		tria->inputs=new Inputs();
	}
	if(this->results){
		tria->results=(Results*)this->results->Copy();
	}
	else{
		tria->results=new Results();
	}
	/*point parameters: */
	tria->parameters=this->parameters;
	
	/*recover objects: */
	tria->nodes=(Node**)xmalloc(3*sizeof(Node*)); //we cannot rely on an analysis_counter to tell us which analysis_type we are running, so we just copy the nodes.
	for(i=0;i<3;i++)tria->nodes[i]=this->nodes[i];
	tria->matice=(Matice*)tria->hmatice->delivers();
	tria->matpar=(Matpar*)tria->hmatpar->delivers();

	return tria;
}
/*}}}*/
/*FUNCTION Tria::DeepEcho{{{1*/
void Tria::DeepEcho(void){

	printf("Tria:\n");
	printf("   id: %i\n",id);
	if(nodes){
		nodes[0]->DeepEcho();
		nodes[1]->DeepEcho();
		nodes[2]->DeepEcho();
	}
	else printf("nodes = NULL\n");

	if (matice) matice->DeepEcho();
	else printf("matice = NULL\n");

	if (matpar) matpar->DeepEcho();
	else printf("matpar = NULL\n");

	printf("   parameters\n");
	if (parameters) parameters->DeepEcho();
	else printf("parameters = NULL\n");

	printf("   inputs\n");
	if (inputs) inputs->DeepEcho();
	else printf("inputs=NULL\n");

	if (results) results->DeepEcho();
	else printf("results=NULL\n");
	
	return;
}
/*}}}*/
/*FUNCTION Tria::Demarshall {{{1*/
void  Tria::Demarshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int i;
	int flaghook;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum type, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/
	memcpy(&id,marshalled_dataset,sizeof(id));marshalled_dataset+=sizeof(id);
	memcpy(&numanalyses,marshalled_dataset,sizeof(numanalyses));marshalled_dataset+=sizeof(numanalyses);

	/*demarshall Ref: */
	this->element_type_list=(int*)xmalloc(this->numanalyses*sizeof(int));
	for(i=0;i<numanalyses;i++){ memcpy(&element_type_list[i],marshalled_dataset,sizeof(int));marshalled_dataset+=sizeof(int);}

	/*allocate dynamic memory: */
	this->hnodes=new Hook*[this->numanalyses];
	/*demarshall hooks: */
	for(i=0;i<numanalyses;i++){
		memcpy(&flaghook,marshalled_dataset,sizeof(flaghook));marshalled_dataset+=sizeof(flaghook);
		if(flaghook){ // there is a hook so demarshall it
			hnodes[i]=new Hook();
			hnodes[i]->Demarshall(&marshalled_dataset);
		}
		else hnodes[i]=NULL; //There is no hook so it is NULL
	}
	hmatice=new Hook(); hmatice->Demarshall(&marshalled_dataset);
	hmatpar=new Hook(); hmatpar->Demarshall(&marshalled_dataset);

	/*pointers are garbabe, until configuration is carried out: */
	nodes=NULL;
	matice=NULL;
	matpar=NULL;
	
	/*demarshall inputs: */
	inputs=(Inputs*)DataSetDemarshallRaw(&marshalled_dataset); 
	results=(Results*)DataSetDemarshallRaw(&marshalled_dataset); 

	/*parameters: may not exist even yet, so let Configure handle it: */
	this->parameters=NULL;

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Tria::Echo{{{1*/
void Tria::Echo(void){
	printf("Tria:\n");
	printf("   id: %i\n",id);
	if(nodes){
		nodes[0]->Echo();
		nodes[1]->Echo();
		nodes[2]->Echo();
	}
	else printf("nodes = NULL\n");

	if (matice) matice->Echo();
	else printf("matice = NULL\n");

	if (matpar) matpar->Echo();
	else printf("matpar = NULL\n");

	printf("   parameters\n");
	if (parameters) parameters->Echo();
	else printf("parameters = NULL\n");

	printf("   inputs\n");
	if (inputs) inputs->Echo();
	else printf("inputs=NULL\n");

	if (results) results->Echo();
	else printf("results=NULL\n");
}
/*}}}*/
/*FUNCTION Tria::Enum {{{1*/
int Tria::Enum(void){

	return TriaEnum;

}
/*}}}*/
/*FUNCTION Tria::Id {{{1*/
int    Tria::Id(){
	
	return id;

}
/*}}}*/
/*FUNCTION Tria::Marshall {{{1*/
void  Tria::Marshall(char** pmarshalled_dataset){

	int   i;
	char* marshalled_dataset=NULL;
	int   enum_type=0;
	char* marshalled_inputs=NULL;
	int   marshalled_inputs_size;
	char* marshalled_results=NULL;
	int   marshalled_results_size;
	int   flaghook; //to indicate if hook is NULL or exists

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum type of Tria: */
	enum_type=TriaEnum;

	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);

	/*marshall Tria data: */
	memcpy(marshalled_dataset,&id,sizeof(id));marshalled_dataset+=sizeof(id);
	memcpy(marshalled_dataset,&numanalyses,sizeof(numanalyses));marshalled_dataset+=sizeof(numanalyses);

	/*Mershall Ref: */
	for(i=0;i<numanalyses;i++){
		memcpy(marshalled_dataset,&element_type_list[i],sizeof(element_type_list[i]));marshalled_dataset+=sizeof(element_type_list[i]);
	}

 /*Marshall hooks: */
	for(i=0;i<numanalyses;i++){
		if(hnodes[i]){
			/*Set flag to 1 as there is a hook */
			flaghook=1;
			memcpy(marshalled_dataset,&flaghook,sizeof(flaghook));marshalled_dataset+=sizeof(flaghook);
			hnodes[i]->Marshall(&marshalled_dataset);
		}
		else{
			/*Set flag to 0 and do not marshall flag as there is no Hook */
			flaghook=0;
			memcpy(marshalled_dataset,&flaghook,sizeof(flaghook));marshalled_dataset+=sizeof(flaghook);
		}
	}
	hmatice->Marshall(&marshalled_dataset);
	hmatpar->Marshall(&marshalled_dataset);

	/*Marshall inputs: */
	marshalled_inputs_size=inputs->MarshallSize();
	marshalled_inputs=inputs->Marshall();
	memcpy(marshalled_dataset,marshalled_inputs,marshalled_inputs_size*sizeof(char));
	marshalled_dataset+=marshalled_inputs_size;

	/*Marshall results: */
	marshalled_results_size=results->MarshallSize();
	marshalled_results=results->Marshall();
	memcpy(marshalled_dataset,marshalled_results,marshalled_results_size*sizeof(char));
	marshalled_dataset+=marshalled_results_size;

	/*parameters: don't do anything about it. parameters are marshalled somewhere else!*/

	xfree((void**)&marshalled_inputs);
	xfree((void**)&marshalled_results);

	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Tria::MarshallSize {{{1*/
int   Tria::MarshallSize(){

	int i;
	int hnodes_size=0;;

	for(i=0;i<numanalyses;i++){
		hnodes_size+=sizeof(int); //Flag 0 or 1
		if (hnodes[i]) hnodes_size+=hnodes[i]->MarshallSize();
	}

	return sizeof(id)
		+hnodes_size
		+sizeof(numanalyses)
		+numanalyses*sizeof(int) //element_type_lists
		+hmatice->MarshallSize()
		+hmatpar->MarshallSize()
		+inputs->MarshallSize()
		+results->MarshallSize()
		+sizeof(int); //sizeof(int) for enum type
}
/*}}}*/
/*FUNCTION Tria::MyRank {{{1*/
int    Tria::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}
/*}}}*/

/*Update virtual functions definitions: */
/*FUNCTION Tria::InputUpdateFromConstant(int value, int name);{{{1*/
void  Tria::InputUpdateFromConstant(int constant, int name){
	/*Check that name is an element input*/
	if (!IsInput(name)) return;

	/*update input*/
	this->inputs->AddInput(new IntInput(name,constant));
}
/*}}}*/
/*FUNCTION Tria::InputUpdateFromConstant(double value, int name);{{{1*/
void  Tria::InputUpdateFromConstant(double constant, int name){
	/*Check that name is an element input*/
	if (!IsInput(name)) return;

	/*update input*/
	this->inputs->AddInput(new DoubleInput(name,constant));
}
/*}}}*/
/*FUNCTION Tria::InputUpdateFromConstant(bool value, int name);{{{1*/
void  Tria::InputUpdateFromConstant(bool constant, int name){
	/*Check that name is an element input*/
	if (!IsInput(name)) return;

	/*update input*/
	this->inputs->AddInput(new BoolInput(name,constant));
}
/*}}}*/
/*FUNCTION Tria::InputUpdateFromSolution {{{1*/
void  Tria::InputUpdateFromSolution(double* solution){

	/*retrive parameters: */
	int analysis_type;
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	/*Just branch to the correct InputUpdateFromSolution generator, according to the type of analysis we are carrying out: */
	switch(analysis_type){
		case DiagnosticHorizAnalysisEnum:
			InputUpdateFromSolutionDiagnosticHoriz( solution);
			break;
		case DiagnosticHutterAnalysisEnum:
			InputUpdateFromSolutionDiagnosticHoriz( solution);
			break;
		case AdjointHorizAnalysisEnum:
			InputUpdateFromSolutionAdjointHoriz( solution);
			break;
		case BedSlopeXAnalysisEnum:
			InputUpdateFromSolutionOneDof(solution,BedSlopeXEnum);
			break;
		case BedSlopeYAnalysisEnum:
			InputUpdateFromSolutionOneDof(solution,BedSlopeYEnum);
			break;
		case SurfaceSlopeXAnalysisEnum:
			InputUpdateFromSolutionOneDof(solution,SurfaceSlopeXEnum);
			break;
		case SurfaceSlopeYAnalysisEnum:
			InputUpdateFromSolutionOneDof(solution,SurfaceSlopeYEnum);
			break;
		case PrognosticAnalysisEnum:
			InputUpdateFromSolutionOneDof(solution,ThicknessEnum);
			break;
		case BalancedthicknessAnalysisEnum:
			InputUpdateFromSolutionOneDof(solution,ThicknessEnum);
			break;
		case AdjointBalancedthicknessAnalysisEnum:
			InputUpdateFromSolutionAdjointBalancedthickness( solution);
			break;
		case BalancedvelocitiesAnalysisEnum:
			InputUpdateFromSolutionOneDof(solution,VelEnum);
			break;
		default:
			ISSMERROR("analysis %i (%s) not supported yet",analysis_type,EnumToString(analysis_type));
	}
}
/*}}}*/
/*FUNCTION Tria::InputUpdateFromVector(double* vector, int name, int type);{{{1*/
void  Tria::InputUpdateFromVector(double* vector, int name, int type){

	/*Check that name is an element input*/
	if (!IsInput(name)) return;

	switch(type){

		case VertexEnum:

			/*New TriaVertexInput*/
			double values[3];

			/*Get values on the 3 vertices*/
			for (int i=0;i<3;i++){
				values[i]=vector[this->nodes[i]->GetVertexDof()];
			}

			/*update input*/
			if (name==RheologyBbarEnum || name==RheologyBEnum){
				matice->inputs->AddInput(new TriaVertexInput(name,values));
			}
			else{
				this->inputs->AddInput(new TriaVertexInput(name,values));
			}
			return;

		default:
			ISSMERROR("type %i (%s) not implemented yet",type,EnumToString(type));
	}
}
/*}}}*/
/*FUNCTION Tria::InputUpdateFromVector(int* vector, int name, int type);{{{1*/
void  Tria::InputUpdateFromVector(int* vector, int name, int type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Tria::InputUpdateFromVector(bool* vector, int name, int type);{{{1*/
void  Tria::InputUpdateFromVector(bool* vector, int name, int type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Tria::InputUpdateFromVectorDakota(double* vector, int name, int type);{{{1*/
void  Tria::InputUpdateFromVectorDakota(double* vector, int name, int type){
	
	int i,j;

	/*Check that name is an element input*/
	if (!IsInput(name)) return;

	switch(type){

		case VertexEnum:

			/*New TriaVertexInput*/
			double values[3];

			/*Get values on the 3 vertices*/
			for (i=0;i<3;i++){
				values[i]=vector[this->nodes[i]->GetVertexDof()];
			}

			/*Branch on the specified type of update: */
			switch(name){
				case ThicknessEnum:
					/*Update thickness + surface: assume bed is constant. On ice shelves, takes hydrostatic equilibrium {{{2*/
					double  thickness[3];
					double  surface[3];
					double  bed[3];
					
					/*retrieve inputs: */
					GetParameterListOnVertices(&bed[0],BedEnum);
					GetParameterListOnVertices(&surface[0],SurfaceEnum);

					/*build new thickness: */
					for(j=0;j<3;j++)thickness[j]=values[j];

					/*build new bed and surface: */
					if (this->IsOnShelf()){
						/*hydrostatic equilibrium: */
						double rho_ice,rho_water,di;
						rho_ice=this->matpar->GetRhoIce();
						rho_water=this->matpar->GetRhoWater();

						di=rho_ice/rho_water;

						for(j=0;j<3;j++){
							surface[j]=(1-di)*thickness[j];
							bed[j]=-di*thickness[j];
						}
					}
					else{
						/*update bed on grounded ice: */
						for(j=0;j<3;j++)surface[j]=bed[j]+thickness[j];
					}

					/*Add new inputs: */
					this->inputs->AddInput(new TriaVertexInput(ThicknessEnum,thickness));
					this->inputs->AddInput(new TriaVertexInput(BedEnum,bed));
					this->inputs->AddInput(new TriaVertexInput(SurfaceEnum,surface));

					/*}}}*/
					break;
				default:
					this->inputs->AddInput(new TriaVertexInput(name,values));
			}
			break;

		default:
			ISSMERROR("type %i (%s) not implemented yet",type,EnumToString(type));
	}

}
/*}}}*/
/*FUNCTION Tria::InputUpdateFromVectorDakota(int* vector, int name, int type);{{{1*/
void  Tria::InputUpdateFromVectorDakota(int* vector, int name, int type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Tria::InputUpdateFromVectorDakota(bool* vector, int name, int type);{{{1*/
void  Tria::InputUpdateFromVectorDakota(bool* vector, int name, int type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/

/*Element virtual functions definitions: */
/*FUNCTION Tria::AverageOntoPartition {{{1*/
void  Tria::AverageOntoPartition(Vec partition_contributions,Vec partition_areas,double* vertex_response,double* qmu_part){

	bool      already=false;
	int       i,j;
	int       partition[NUMVERTICES];
	int       offset[NUMVERTICES];
	double    area;
	double    mean;
	double    values[3];

	/*First, get the area: */
	area=this->GetArea();

	/*Figure out the average for this element: */
	this->GetDofList1(&offset[0]);
	mean=0;
	for(i=0;i<NUMVERTICES;i++){
		partition[i]=(int)qmu_part[offset[i]];
		mean=mean+1.0/NUMVERTICES*vertex_response[offset[i]];
	}

	/*Add contribution: */
	for(i=0;i<NUMVERTICES;i++){
		already=false;
		for(j=0;j<i;j++){
			if (partition[i]==partition[j]){
				already=true;
				break;
			}
		}
		if(!already){
			VecSetValue(partition_contributions,partition[i],mean*area,ADD_VALUES);
			VecSetValue(partition_areas,partition[i],area,ADD_VALUES);
		};
	}
}
/*}}}*/
/*FUNCTION Tria::ComputeBasalStress {{{1*/
void  Tria::ComputeBasalStress(Vec eps){
	ISSMERROR("Not Implemented yet");
}
/*}}}*/
/*FUNCTION Tria::ComputeStrainRate {{{1*/
void  Tria::ComputeStrainRate(Vec eps){
	ISSMERROR("Not Implemented yet");
}
/*}}}*/
/*FUNCTION Tria::SetCurrentConfiguration {{{1*/
void  Tria::SetCurrentConfiguration(Elements* elementsin, Loads* loadsin, DataSet* nodesin, Materials* materialsin, Parameters* parametersin){
	
	/*go into parameters and get the analysis_counter: */
	int analysis_counter;
	parametersin->FindParam(&analysis_counter,AnalysisCounterEnum);

	/*Get Element type*/
	this->element_type=this->element_type_list[analysis_counter];

	/*Pick up nodes*/
	if(this->hnodes[analysis_counter]) this->nodes=(Node**)this->hnodes[analysis_counter]->deliverp();
	else this->nodes=NULL;

}
/*}}}*/
/*FUNCTION Tria::Configure {{{1*/
void  Tria::Configure(Elements* elementsin, Loads* loadsin, DataSet* nodesin, Materials* materialsin, Parameters* parametersin){
	
	/*go into parameters and get the analysis_counter: */
	int analysis_counter;
	parametersin->FindParam(&analysis_counter,AnalysisCounterEnum);

	/*Get Element type*/
	this->element_type=this->element_type_list[analysis_counter];

	/*Take care of hooking up all objects for this element, ie links the objects in the hooks to their respective 
	 * datasets, using internal ids and offsets hidden in hooks: */
	if(this->hnodes[analysis_counter]) this->hnodes[analysis_counter]->configure(nodesin);
	this->hmatice->configure(materialsin);
	this->hmatpar->configure(materialsin);

	/*Now, go pick up the objects inside the hooks: */
	if(this->hnodes[analysis_counter]) this->nodes=(Node**)this->hnodes[analysis_counter]->deliverp();
	else this->nodes=NULL;
	this->matice=(Matice*)this->hmatice->delivers();
	this->matpar=(Matpar*)this->hmatpar->delivers();

	/*point parameters to real dataset: */
	this->parameters=parametersin;

}
/*}}}*/
/*FUNCTION Tria::RegularizeInversion {{{1*/
double Tria::RegularizeInversion(){

	/*constants: */
	const int    numdof=NDOF2*NUMVERTICES;

	/* Intermediaries */
	int        i,j,ig;
	int        control_type;
	double     Jelem = 0;
	double     cm_noisedmp;
	double     Jdet;
	double     xyz_list[NUMVERTICES][3];
	double     dk[NDOF2],dB[NDOF2];
	GaussTria *gauss = NULL;

	/*retrieve parameters and inputs*/
	this->parameters->FindParam(&control_type,ControlTypeEnum);
	this->parameters->FindParam(&cm_noisedmp,CmNoiseDmpEnum);

	/*If on water, return 0: */
	if(IsOnWater()) return 0;

	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);

	/* Start looping on the number of gaussian points: */
	gauss=new GaussTria(2);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss);

		/*Add Tikhonov regularization term to misfit:
		 *
		 * WARNING: for now, the regularization is only taken into account by the gradient
		 * the misfit reflects the response only!
		 *
		 * */
		if (control_type==DragCoefficientEnum){
			Input* drag_input=inputs->GetInput(DragCoefficientEnum);      ISSMASSERT(drag_input);
			drag_input->GetParameterDerivativeValue(&dk[0],&xyz_list[0][0],gauss);
			//Jelem+=cm_noisedmp*1/2*(pow(dk[0],2)+pow(dk[1],2))*Jdet*gauss->weight;
		}
		else if (control_type==RheologyBbarEnum){
			//nothing
		}
		else if (control_type==DhDtEnum){
			//nothing
		}
		else if (control_type==VxEnum){
			//nothing
		}
		else if (control_type==VyEnum){
			//nothing
		}
		else{
			ISSMERROR("unsupported control type: %s",EnumToString(control_type));
		}
	}

	/*clean-up and return: */
	delete gauss;
	return Jelem;
}
/*}}}*/
/*FUNCTION Tria::CreateKMatrix {{{1*/
void  Tria::CreateKMatrix(Mat Kgg, Mat Kff, Mat Kfs){

	/*retreive parameters: */
	ElementMatrix* Ke=NULL;
	int analysis_type;
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	/*Checks in debugging mode{{{2*/
	ISSMASSERT(this->nodes && this->matice && this->matpar && this->parameters && this->inputs);
	/*}}}*/

	/*Just branch to the correct element stiffness matrix generator, according to the type of analysis we are carrying out: */
	switch(analysis_type){
		case DiagnosticHorizAnalysisEnum: case AdjointHorizAnalysisEnum:
			Ke=CreateKMatrixDiagnosticMacAyeal();
			break;
		case DiagnosticHutterAnalysisEnum:
			Ke=CreateKMatrixDiagnosticHutter();
			break;
		case BedSlopeXAnalysisEnum: case SurfaceSlopeXAnalysisEnum: case BedSlopeYAnalysisEnum: case SurfaceSlopeYAnalysisEnum:
			Ke=CreateKMatrixSlope();
			break;
		case PrognosticAnalysisEnum:
			Ke=CreateKMatrixPrognostic();
			break;
		case BalancedthicknessAnalysisEnum:
			Ke=CreateKMatrixBalancedthickness();
			break;
		case AdjointBalancedthicknessAnalysisEnum:
			Ke=CreateKMatrixAdjointBalancedthickness();
			break;
		case BalancedvelocitiesAnalysisEnum:
			Ke=CreateKMatrixBalancedvelocities();
			break;
		default:
			ISSMERROR("analysis %i (%s) not supported yet",analysis_type,EnumToString(analysis_type));
	}

	/*Add to global matrix*/
	if(Ke){
		Ke->AddToGlobal(Kgg,Kff,Kfs);
		delete Ke;
	}
}
/*}}}*/
/*FUNCTION Tria::CreatePVector {{{1*/
void  Tria::CreatePVector(Vec pg, Vec pf){

	/*retrive parameters: */
	ElementVector* pe=NULL;
	int analysis_type;
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	/*asserts: {{{*/
	/*if debugging mode, check that all pointers exist*/
	ISSMASSERT(this->nodes && this->matice && this->matpar && this->parameters && this->inputs);
	/*}}}*/

	/*Just branch to the correct load generator, according to the type of analysis we are carrying out: */
	switch(analysis_type){
		case DiagnosticHorizAnalysisEnum:
			pe=CreatePVectorDiagnosticMacAyeal();
			break;
		case AdjointHorizAnalysisEnum:
			pe=CreatePVectorAdjointHoriz();
			break;
		case DiagnosticHutterAnalysisEnum:
			pe=CreatePVectorDiagnosticHutter();
			break;
		case BedSlopeXAnalysisEnum: case SurfaceSlopeXAnalysisEnum: case BedSlopeYAnalysisEnum: case SurfaceSlopeYAnalysisEnum:
			pe=CreatePVectorSlope();
			break;
		case PrognosticAnalysisEnum:
			pe=CreatePVectorPrognostic();
			break;
		case BalancedthicknessAnalysisEnum:
			pe=CreatePVectorBalancedthickness();
			break;
		case AdjointBalancedthicknessAnalysisEnum:
			pe=CreatePVectorAdjointBalancedthickness();
			break;
		case BalancedvelocitiesAnalysisEnum:
			pe=CreatePVectorBalancedvelocities();
			break;
		default:
			ISSMERROR("analysis %i (%s) not supported yet",analysis_type,EnumToString(analysis_type));
	}

	/*Add to global Vector*/
	if(pe){
		pe->AddToGlobal(pg,pf);
		delete pe;
	}
}
/*}}}*/
/*FUNCTION Tria::DeleteResults {{{1*/
void  Tria::DeleteResults(void){

	/*Delete and reinitialize results*/
	delete this->results;
	this->results=new Results();

}
/*}}}*/
/*FUNCTION Tria::GetNodeIndex {{{1*/
int Tria::GetNodeIndex(Node* node){

	ISSMASSERT(nodes);
	for(int i=0;i<NUMVERTICES;i++){
		if(node==nodes[i])
		 return i;
	}
	ISSMERROR("Node provided not found among element nodes");
}
/*}}}*/
/*FUNCTION Tria::IsOnBed {{{1*/
bool Tria::IsOnBed(){
	
	bool onbed;
	inputs->GetParameterValue(&onbed,ElementOnBedEnum);
	return onbed;
}
/*}}}*/
/*FUNCTION Tria::IsOnShelf {{{1*/
bool   Tria::IsOnShelf(){

	bool shelf;
	inputs->GetParameterValue(&shelf,ElementOnIceShelfEnum);
	return shelf;
}
/*}}}*/
/*FUNCTION Tria::IsOnWater {{{1*/
bool   Tria::IsOnWater(){

	bool water;
	inputs->GetParameterValue(&water,ElementOnWaterEnum);
	return water;
}
/*}}}*/
/*FUNCTION Tria::GetSolutionFromInputs{{{1*/
void  Tria::GetSolutionFromInputs(Vec solution){

	/*retrive parameters: */
	int analysis_type;
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);
	
	/*Just branch to the correct InputUpdateFromSolution generator, according to the type of analysis we are carrying out: */
	if (analysis_type==DiagnosticHorizAnalysisEnum)
	 GetSolutionFromInputsDiagnosticHoriz(solution);
	else if (analysis_type==DiagnosticHutterAnalysisEnum)
	 GetSolutionFromInputsDiagnosticHutter(solution);
	else
	 ISSMERROR("analysis: %s not supported yet",EnumToString(analysis_type));

}
/*}}}*/
/*FUNCTION Tria::GetVectorFromInputs{{{1*/
void  Tria::GetVectorFromInputs(Vec vector,int NameEnum){

	int doflist1[NUMVERTICES];

	/*Find NameEnum input in the inputs dataset, and get it to fill in the vector: */
	for(int i=0;i<this->inputs->Size();i++){
		Input* input=(Input*)this->inputs->GetObjectByOffset(i);
		if(input->EnumType()==NameEnum){
			/*We found the enum.  Use its values to fill into the vector, using the vertices ids: */
			this->GetDofList1(&doflist1[0]);
			input->GetVectorFromInputs(vector,&doflist1[0]);
			break;
		}
	}
}
/*}}}*/
/*FUNCTION Tria::Gradj {{{1*/
void  Tria::Gradj(Vec gradient,int control_type){

	/*If on water, grad = 0: */
	if(IsOnWater())return;

	switch(control_type){
		case DragCoefficientEnum:
			GradjDrag(gradient);
			break;
		case RheologyBbarEnum:
			GradjB(gradient);
			break;
		case DhDtEnum:
			GradjDhDt(gradient);
			break;
		case VxEnum:
			GradjVx(gradient);
			break;
		case VyEnum:
			GradjVy(gradient);
			break;
		default:
			ISSMERROR("%s%i","control type not supported yet: ",control_type);
	}
}
/*}}}*/
/*FUNCTION Tria::GradjB{{{1*/
void  Tria::GradjB(Vec gradient){

	/*Intermediaries*/
	int        i,ig;
	int        doflist[NUMVERTICES];
	double     vx,vy,lambda,mu,thickness,Jdet;
	double     cm_noisedmp,viscosity_complement;
	double     dvx[NDOF2],dvy[NDOF2],dadjx[NDOF2],dadjy[NDOF2],dB[NDOF2]; 
	double     xyz_list[NUMVERTICES][3];
	double     basis[3],epsilon[3];
	double     dbasis[NDOF2][NUMVERTICES];
	double     grad_g[NUMVERTICES];
	double     grad[NUMVERTICES]={0.0};
	GaussTria *gauss = NULL;

	/*retrieve some parameters: */
	this->parameters->FindParam(&cm_noisedmp,CmNoiseDmpEnum);

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	GetDofList1(&doflist[0]);

	/*Retrieve all inputs*/
	Input* thickness_input=inputs->GetInput(ThicknessEnum);            ISSMASSERT(thickness_input);
	Input* vx_input=inputs->GetInput(VxEnum);                          ISSMASSERT(vx_input);
	Input* vy_input=inputs->GetInput(VyEnum);                          ISSMASSERT(vy_input);
	Input* adjointx_input=inputs->GetInput(AdjointxEnum);              ISSMASSERT(adjointx_input);
	Input* adjointy_input=inputs->GetInput(AdjointyEnum);              ISSMASSERT(adjointy_input);
	Input* rheologyb_input=matice->inputs->GetInput(RheologyBbarEnum); ISSMASSERT(rheologyb_input);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussTria(4);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		thickness_input->GetParameterValue(&thickness,gauss);
		rheologyb_input->GetParameterDerivativeValue(&dB[0],&xyz_list[0][0],gauss);
		vx_input->GetParameterDerivativeValue(&dvx[0],&xyz_list[0][0],gauss);
		vy_input->GetParameterDerivativeValue(&dvy[0],&xyz_list[0][0],gauss);
		adjointx_input->GetParameterDerivativeValue(&dadjx[0],&xyz_list[0][0],gauss);
		adjointy_input->GetParameterDerivativeValue(&dadjy[0],&xyz_list[0][0],gauss);

		this->GetStrainRate2d(&epsilon[0],&xyz_list[0][0],gauss,vx_input,vy_input);
		matice->GetViscosityComplement(&viscosity_complement,&epsilon[0]);

		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss);
		GetNodalFunctions(basis,gauss);
		GetNodalFunctionsDerivatives(&dbasis[0][0],&xyz_list[0][0],gauss);

		/*standard gradient dJ/dki*/
		for (i=0;i<NUMVERTICES;i++){
			grad_g[i]=-viscosity_complement*thickness*( (2*dvx[0]+dvy[1])*2*dadjx[0]+(dvx[1]+dvy[0])*(dadjx[1]+dadjy[0])+(2*dvy[1]+dvx[0])*2*dadjy[1])*Jdet*gauss->weight*basis[i]; 
		}
		/*Add regularization term*/
		for (i=0;i<NUMVERTICES;i++) grad_g[i]-=cm_noisedmp*Jdet*gauss->weight*(dbasis[0][i]*dB[0]+dbasis[1][i]*dB[1]);
		for(i=0;i<NUMVERTICES;i++) grad[i]+=grad_g[i];
	}

	VecSetValues(gradient,NUMVERTICES,doflist,(const double*)grad,ADD_VALUES);

	/*clean-up*/
	delete gauss;
}
/*}}}*/
/*FUNCTION Tria::GradjDrag {{{1*/
void  Tria::GradjDrag(Vec gradient){

	int        i,ig;
	int        drag_type,analysis_type;
	int        doflist1[NUMVERTICES];
	double     vx,vy,lambda,mu,alpha_complement,Jdet;
	double     bed,thickness,Neff,drag,cm_noisedmp;
	double     xyz_list[NUMVERTICES][3];
	double     dh1dh3[NDOF2][NUMVERTICES];
	double     dk[NDOF2]; 
	double     grade_g[NUMVERTICES]={0.0};
	double     grade_g_gaussian[NUMVERTICES];
	double     l1l2l3[3];
	double     epsilon[3]; /* epsilon=[exx,eyy,exy];*/
	Friction*  friction=NULL;
	GaussTria  *gauss=NULL;

	/*retrive parameters: */
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	/*retrieve some parameters ands return if iceshelf: */
	this->parameters->FindParam(&cm_noisedmp,CmNoiseDmpEnum);
	if(IsOnShelf())return;
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	GetDofList1(&doflist1[0]);

	/*Build frictoin element, needed later: */
	inputs->GetParameterValue(&drag_type,DragTypeEnum);
	friction=new Friction("2d",inputs,matpar,analysis_type);

	/*Retrieve all inputs we will be needing: */
	Input* adjointx_input=inputs->GetInput(AdjointxEnum);               ISSMASSERT(adjointx_input);
	Input* adjointy_input=inputs->GetInput(AdjointyEnum);               ISSMASSERT(adjointy_input);
	Input* vx_input=inputs->GetInput(VxEnum);                           ISSMASSERT(vx_input);
	Input* vy_input=inputs->GetInput(VyEnum);                           ISSMASSERT(vy_input);
	Input* dragcoefficient_input=inputs->GetInput(DragCoefficientEnum); ISSMASSERT(dragcoefficient_input);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussTria(4);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss);
		GetNodalFunctions(l1l2l3, gauss);
		GetNodalFunctionsDerivatives(&dh1dh3[0][0],&xyz_list[0][0],gauss);

		/*Build alpha_complement_list: */
		if (drag_type==2) friction->GetAlphaComplement(&alpha_complement, gauss,VxEnum,VyEnum);
		else alpha_complement=0;
	
		dragcoefficient_input->GetParameterValue(&drag, gauss);
		adjointx_input->GetParameterValue(&lambda, gauss);
		adjointy_input->GetParameterValue(&mu, gauss);
		vx_input->GetParameterValue(&vx,gauss);
		vy_input->GetParameterValue(&vy,gauss);
		dragcoefficient_input->GetParameterDerivativeValue(&dk[0],&xyz_list[0][0],gauss);

		/*Build gradje_g_gaussian vector (actually -dJ/ddrag): */
		for (i=0;i<NUMVERTICES;i++){

			//standard term dJ/dki
			grade_g_gaussian[i]=-2*drag*alpha_complement*((lambda*vx+mu*vy))*Jdet*gauss->weight*l1l2l3[i];

			//noise dampening d/dki(1/2*(dk/dx)^2)
			grade_g_gaussian[i]+=-cm_noisedmp*Jdet*gauss->weight*(dh1dh3[0][i]*dk[0]+dh1dh3[1][i]*dk[1]);
		}
		
		/*Add gradje_g_gaussian vector to gradje_g: */
		for( i=0; i<NUMVERTICES; i++)grade_g[i]+=grade_g_gaussian[i];
	}

	VecSetValues(gradient,NUMVERTICES,doflist1,(const double*)grade_g,ADD_VALUES);

	/*Clean up and return*/
	delete gauss;
	delete friction;
}
/*}}}*/
/*FUNCTION Tria::GradjDhDt{{{1*/
void  Tria::GradjDhDt(Vec gradient){

	/*Intermediaries*/
	int    doflist1[NUMVERTICES];
	double lambda[NUMVERTICES];
	double gradient_g[NUMVERTICES];

	GetDofList1(&doflist1[0]);

	/*Compute Gradient*/
	GetParameterListOnVertices(&lambda[0],AdjointEnum);
	for(int i=0;i<NUMVERTICES;i++) gradient_g[i]=-lambda[i];

	VecSetValues(gradient,NUMVERTICES,doflist1,(const double*)gradient_g,INSERT_VALUES);
}
/*}}}*/
/*FUNCTION Tria::GradjVx{{{1*/
void  Tria::GradjVx(Vec gradient){

	/*Intermediaries*/
	int        i,ig;
	int        doflist1[NUMVERTICES];
	double     thickness,Jdet;
	double     l1l2l3[3];
	double     Dlambda[2];
	double     xyz_list[NUMVERTICES][3];
	double     grade_g[NUMVERTICES] = {0.0};
	GaussTria *gauss                = NULL;

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	GetDofList1(&doflist1[0]);

	/*Retrieve all inputs we will be needing: */
	Input* adjoint_input=inputs->GetInput(AdjointEnum);     ISSMASSERT(adjoint_input);
	Input* thickness_input=inputs->GetInput(ThicknessEnum); ISSMASSERT(thickness_input);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussTria(2);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss);
		GetNodalFunctions(l1l2l3, gauss);
		
		adjoint_input->GetParameterDerivativeValue(&Dlambda[0],&xyz_list[0][0],gauss);
		thickness_input->GetParameterValue(&thickness, gauss);

		for(i=0;i<NUMVERTICES;i++) grade_g[i]=thickness*Dlambda[0]*Jdet*gauss->weight*l1l2l3[i];
	}

	VecSetValues(gradient,NUMVERTICES,doflist1,(const double*)grade_g,ADD_VALUES);

	/*Clean up and return*/
	delete gauss;
}
/*}}}*/
/*FUNCTION Tria::GradjVy{{{1*/
void  Tria::GradjVy(Vec gradient){

	/*Intermediaries*/
	int        i,ig;
	int        doflist1[NUMVERTICES];
	double     thickness,Jdet;
	double     l1l2l3[3];
	double     Dlambda[2];
	double     xyz_list[NUMVERTICES][3];
	double     grade_g[NUMVERTICES] = {0.0};
	GaussTria *gauss                = NULL;

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	GetDofList1(&doflist1[0]);

	/*Retrieve all inputs we will be needing: */
	Input* adjoint_input=inputs->GetInput(AdjointEnum);     ISSMASSERT(adjoint_input);
	Input* thickness_input=inputs->GetInput(ThicknessEnum); ISSMASSERT(thickness_input);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussTria(2);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		adjoint_input->GetParameterDerivativeValue(&Dlambda[0],&xyz_list[0][0],gauss);
		thickness_input->GetParameterValue(&thickness, gauss);

		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss);
		GetNodalFunctions(l1l2l3, gauss);

		for(i=0;i<NUMVERTICES;i++) grade_g[i]=thickness*Dlambda[1]*Jdet*gauss->weight*l1l2l3[i];
	}

	VecSetValues(gradient,NUMVERTICES,doflist1,(const double*)grade_g,ADD_VALUES);

	/*Clean up and return*/
	delete gauss;
}
/*}}}*/
/*FUNCTION Tria::InputControlUpdate{{{1*/
void  Tria::InputControlUpdate(double scalar,bool save_parameter){

	/*Intermediary*/
	int    control_type;
	Input* input=NULL;
	double cm_min,cm_max;

	/*retrieve some parameters: */
	this->parameters->FindParam(&cm_min,CmMinEnum);
	this->parameters->FindParam(&cm_max,CmMaxEnum);
	this->parameters->FindParam(&control_type,ControlTypeEnum);

	/*Rheology*/
	switch(control_type){

		case RheologyBbarEnum:{

			/*First, get revert to previous parameter value (kept in ControlParameter input)*/
			matice->inputs->DuplicateInput(ControlParameterEnum,RheologyBbarEnum);

			/*Now, update using the gradient new = old + scalar * gradient*/
			//matice->inputs->AXPY(RheologyBbarEnum,scalar,GradientEnum);
			// For now: Gradient is in element (TO BE CHANGED) and parameter in matice
			Input* input_B   =(Input*)matice->inputs->GetInput(RheologyBbarEnum); ISSMASSERT(input_B);
			Input* input_Grad=(Input*)this->inputs->GetInput(GradientEnum); ISSMASSERT(input_Grad);
			input_B->AXPY(input_Grad,scalar);

			/*Constrain constrain input*/
			input=(Input*)matice->inputs->GetInput(RheologyBbarEnum); ISSMASSERT(input);
			input->Constrain(cm_min,cm_max);

			/*Finally: save input if requested*/
			if (save_parameter) matice->inputs->DuplicateInput(RheologyBbarEnum,ControlParameterEnum);
									 }
			break;

		case DragCoefficientEnum:{
			this->inputs->DuplicateInput(ControlParameterEnum,DragCoefficientEnum);
			this->inputs->AXPY(DragCoefficientEnum,scalar,GradientEnum);
			input=(Input*)this->inputs->GetInput(DragCoefficientEnum); ISSMASSERT(input);
			input->Constrain(cm_min,cm_max);
			if (save_parameter) inputs->DuplicateInput(DragCoefficientEnum,ControlParameterEnum);
										 }
			break;
		case DhDtEnum:{
			this->inputs->DuplicateInput(ControlParameterEnum,DhDtEnum);
			this->inputs->AXPY(DhDtEnum,scalar,GradientEnum);
			input=(Input*)this->inputs->GetInput(DhDtEnum); ISSMASSERT(input);
			input->Constrain(cm_min,cm_max);
			if (save_parameter) inputs->DuplicateInput(DhDtEnum,ControlParameterEnum);
						  }
			break;
		case VxEnum:{
			this->inputs->DuplicateInput(ControlParameterEnum,VxEnum);
			this->inputs->AXPY(VxEnum,scalar,GradientEnum);
			input=(Input*)this->inputs->GetInput(VxEnum); ISSMASSERT(input);
			input->Constrain(cm_min,cm_max);
			if (save_parameter) inputs->DuplicateInput(VxEnum,ControlParameterEnum);
						  }
			break;
		case VyEnum:{
			this->inputs->DuplicateInput(ControlParameterEnum,VyEnum);
			this->inputs->AXPY(VyEnum,scalar,GradientEnum);
			input=(Input*)this->inputs->GetInput(VyEnum); ISSMASSERT(input);
			input->Constrain(cm_min,cm_max);
			if (save_parameter) inputs->DuplicateInput(VyEnum,ControlParameterEnum);
						  }
			break;
		default:
			ISSMERROR("control type %s not implemented yet",EnumToString(control_type));
	}
}
/*}}}*/
/*FUNCTION Tria::InputConvergence{{{1*/
bool Tria::InputConvergence(double* eps, int* enums,int num_enums,int* criterionenums,double* criterionvalues,int num_criterionenums){

	bool    converged=true;
	int     i;
	Input** new_inputs=NULL;
	Input** old_inputs=NULL;

	new_inputs=(Input**)xmalloc(num_enums/2*sizeof(Input*)); //half the enums are for the new inputs
	old_inputs=(Input**)xmalloc(num_enums/2*sizeof(Input*)); //half the enums are for the old inputs

	for(i=0;i<num_enums/2;i++){
		new_inputs[i]=(Input*)this->inputs->GetInput(enums[2*i+0]);
		old_inputs[i]=(Input*)this->inputs->GetInput(enums[2*i+1]);
		if(!new_inputs[i])ISSMERROR("%s%s"," could not find input with enum ",EnumToString(enums[2*i+0]));
		if(!old_inputs[i])ISSMERROR("%s%s"," could not find input with enum ",EnumToString(enums[2*i+0]));
	}

	/*ok, we've got the inputs (new and old), now loop throught the number of criterions and fill the eps array:*/
	for(i=0;i<num_criterionenums;i++){
		IsInputConverged(eps+i,new_inputs,old_inputs,num_enums/2,criterionenums[i]);
		if(eps[i]>criterionvalues[i]) converged=false; 
	}

	/*clean up and return*/
	xfree((void**)&new_inputs);
	xfree((void**)&old_inputs);
	return converged;
}
/*}}}*/
/*FUNCTION Tria::InputDepthAverageAtBase {{{1*/
void  Tria::InputDepthAverageAtBase(int enum_type,int average_enum_type,int object_enum){

	/*New input*/
	Input* oldinput=NULL;
	Input* newinput=NULL;

	/*copy input of enum_type*/
	if (object_enum==ElementsEnum)
	 oldinput=(Input*)this->inputs->GetInput(enum_type);
	else if (object_enum==MaterialsEnum)
	 oldinput=(Input*)this->matice->inputs->GetInput(enum_type);
	else
	 ISSMERROR("object %s not supported yet",EnumToString(object_enum));
	if(!oldinput)ISSMERROR("%s%s"," could not find old input with enum: ",EnumToString(enum_type));
	newinput=(Input*)oldinput->copy();

	/*Assign new name (average)*/
	newinput->ChangeEnum(average_enum_type);

	/*Add new input to current element*/
	if (object_enum==ElementsEnum)
	 this->inputs->AddInput((Input*)newinput);
	else if (object_enum==MaterialsEnum)
	 this->matice->inputs->AddInput((Input*)newinput);
	else
	 ISSMERROR("object %s not supported yet",EnumToString(object_enum));
}
/*}}}*/
/*FUNCTION Tria::InputDuplicate{{{1*/
void  Tria::InputDuplicate(int original_enum,int new_enum){

	/*Call inputs method*/
	if (IsInput(original_enum)) inputs->DuplicateInput(original_enum,new_enum);

}
/*}}}*/
/*FUNCTION Tria::InputScale{{{1*/
void  Tria::InputScale(int enum_type,double scale_factor){

	Input* input=NULL;

	/*Make a copy of the original input: */
	input=(Input*)this->inputs->GetInput(enum_type);
	if(!input)ISSMERROR(" could not find old input with enum: %s",EnumToString(enum_type));

	/*Scale: */
	input->Scale(scale_factor);
}
/*}}}*/
/*FUNCTION Tria::InputArtificialNoise{{{1*/
void  Tria::InputArtificialNoise(int enum_type,double min,double max){

	Input* input=NULL;

	/*Make a copy of the original input: */
	input=(Input*)this->inputs->GetInput(enum_type);
	if(!input)ISSMERROR(" could not find old input with enum: %s",EnumToString(enum_type));

	/*ArtificialNoise: */
	input->ArtificialNoise(min,max);
}
/*}}}*/
/*FUNCTION Tria::InputToResult{{{1*/
void  Tria::InputToResult(int enum_type,int step,double time){

	int    i;
	Input *input = NULL;

	/*Go through all the input objects, and find the one corresponding to enum_type, if it exists: */
	if (enum_type==RheologyBbarEnum) input=this->matice->inputs->GetInput(enum_type);
	else input=this->inputs->GetInput(enum_type);

	if (!input) ISSMERROR("Input %s not found in tria->inputs",EnumToString(enum_type));

	/*If we don't find it, no big deal, just don't do the transfer. Otherwise, build a new Result 
	 * object out of the input, with the additional step and time information: */
	this->results->AddObject((Object*)input->SpawnResult(step,time));
}
/*}}}*/
/*FUNCTION Tria::MassFlux {{{1*/
double Tria::MassFlux( double* segment,bool process_units){

	const int    numdofs=2;

	int        i;
	double     mass_flux=0;
	double     xyz_list[NUMVERTICES][3];
	double     normal[2];
	double     length,rho_ice;
	double     x1,y1,x2,y2,h1,h2;
	double     vx1,vx2,vy1,vy2;
	GaussTria* gauss_1=NULL;
	GaussTria* gauss_2=NULL;

	/*Get material parameters :*/
	rho_ice=matpar->GetRhoIce();

	/*First off, check that this segment belongs to this element: */
	if ((int)*(segment+4)!=this->id)ISSMERROR("%s%i%s%i","error message: segment with id ",(int)*(segment+4)," does not belong to element with id:",this->id);

	/*Recover segment node locations: */
	x1=*(segment+0); y1=*(segment+1); x2=*(segment+2); y2=*(segment+3);

	/*Get xyz list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);

	/*get area coordinates of 0 and 1 locations: */
	gauss_1=new GaussTria();
	gauss_1->GaussFromCoords(x1,y1,&xyz_list[0][0]);
	gauss_2=new GaussTria();
	gauss_2->GaussFromCoords(x2,y2,&xyz_list[0][0]);

	normal[0]=cos(atan2(x1-x2,y2-y1));
	normal[1]=sin(atan2(x1-x2,y2-y1));

	length=sqrt(pow(x2-x1,2.0)+pow(y2-y1,2));

	Input* thickness_input=inputs->GetInput(ThicknessEnum); ISSMASSERT(thickness_input);
	Input* vx_input=inputs->GetInput(VxEnum); ISSMASSERT(vx_input);
	Input* vy_input=inputs->GetInput(VyEnum); ISSMASSERT(vy_input);

	thickness_input->GetParameterValue(&h1, gauss_1);
	thickness_input->GetParameterValue(&h2, gauss_2);
	vx_input->GetParameterValue(&vx1,gauss_1);
	vx_input->GetParameterValue(&vx2,gauss_2);
	vy_input->GetParameterValue(&vy1,gauss_1);
	vy_input->GetParameterValue(&vy2,gauss_2);

	mass_flux= rho_ice*length*(  
				(ONETHIRD*(h1-h2)*(vx1-vx2)+0.5*h2*(vx1-vx2)+0.5*(h1-h2)*vx2+h2*vx2)*normal[0]+
				(ONETHIRD*(h1-h2)*(vy1-vy2)+0.5*h2*(vy1-vy2)+0.5*(h1-h2)*vy2+h2*vy2)*normal[1]
				);

	/*Process units: */
	mass_flux=UnitConversion(mass_flux,IuToExtEnum,MassFluxEnum,this->parameters);

	/*clean up and return:*/
	delete gauss_1;
	delete gauss_2;
	return mass_flux;
}
/*}}}*/
/*FUNCTION Tria::MaxAbsVx{{{1*/
void  Tria::MaxAbsVx(double* pmaxabsvx, bool process_units){

	/*Get maximum:*/
	double maxabsvx=this->inputs->MaxAbs(VxEnum);

	/*process units if requested: */
	if(process_units) maxabsvx=UnitConversion(maxabsvx,IuToExtEnum,VxEnum,this->parameters);

	/*Assign output pointers:*/
	*pmaxabsvx=maxabsvx;
}
/*}}}*/
/*FUNCTION Tria::MaxAbsVy{{{1*/
void  Tria::MaxAbsVy(double* pmaxabsvy, bool process_units){

	/*Get maximum:*/
	double maxabsvy=this->inputs->MaxAbs(VyEnum);

	/*process units if requested: */
	if(process_units) maxabsvy=UnitConversion(maxabsvy,IuToExtEnum,VyEnum,this->parameters);

	/*Assign output pointers:*/
	*pmaxabsvy=maxabsvy;
}
/*}}}*/
/*FUNCTION Tria::MaxAbsVz{{{1*/
void  Tria::MaxAbsVz(double* pmaxabsvz, bool process_units){

	/*Get maximum:*/
	double maxabsvz=this->inputs->MaxAbs(VzEnum);

	/*process units if requested: */
	if(process_units) maxabsvz=UnitConversion(maxabsvz,IuToExtEnum,VyEnum,this->parameters);

	/*Assign output pointers:*/
	*pmaxabsvz=maxabsvz;
}
/*}}}*/
/*FUNCTION Tria::MaxVel{{{1*/
void  Tria::MaxVel(double* pmaxvel, bool process_units){

	/*Get maximum:*/
	double maxvel=this->inputs->Max(VelEnum);

	/*process units if requested: */
	if(process_units) maxvel=UnitConversion(maxvel,IuToExtEnum,VelEnum,this->parameters);

	/*Assign output pointers:*/
	*pmaxvel=maxvel;
}
/*}}}*/
/*FUNCTION Tria::MaxVx{{{1*/
void  Tria::MaxVx(double* pmaxvx, bool process_units){

	/*Get maximum:*/
	double maxvx=this->inputs->Max(VxEnum);

	/*process units if requested: */
	if(process_units) maxvx=UnitConversion(maxvx,IuToExtEnum,VxEnum,this->parameters);

	/*Assign output pointers:*/
	*pmaxvx=maxvx;
}
/*}}}*/
/*FUNCTION Tria::MaxVy{{{1*/
void  Tria::MaxVy(double* pmaxvy, bool process_units){

	/*Get maximum:*/
	double maxvy=this->inputs->Max(VyEnum);

	/*process units if requested: */
	if(process_units) maxvy=UnitConversion(maxvy,IuToExtEnum,VyEnum,this->parameters);

	/*Assign output pointers:*/
	*pmaxvy=maxvy;

}
/*}}}*/
/*FUNCTION Tria::MaxVz{{{1*/
void  Tria::MaxVz(double* pmaxvz, bool process_units){

	/*Get maximum:*/
	double maxvz=this->inputs->Max(VzEnum);

	/*process units if requested: */
	if(process_units) maxvz=UnitConversion(maxvz,IuToExtEnum,VzEnum,this->parameters);

	/*Assign output pointers:*/
	*pmaxvz=maxvz;
}
/*}}}*/
/*FUNCTION Tria::MinVel{{{1*/
void  Tria::MinVel(double* pminvel, bool process_units){

	/*Get minimum:*/
	double minvel=this->inputs->Min(VelEnum);

	/*process units if requested: */
	if(process_units) minvel=UnitConversion(minvel,IuToExtEnum,VelEnum,this->parameters);

	/*Assign output pointers:*/
	*pminvel=minvel;
}
/*}}}*/
/*FUNCTION Tria::MinVx{{{1*/
void  Tria::MinVx(double* pminvx, bool process_units){

	/*Get minimum:*/
	double minvx=this->inputs->Min(VxEnum);

	/*process units if requested: */
	if(process_units) minvx=UnitConversion(minvx,IuToExtEnum,VxEnum,this->parameters);

	/*Assign output pointers:*/
	*pminvx=minvx;
}
/*}}}*/
/*FUNCTION Tria::MinVy{{{1*/
void  Tria::MinVy(double* pminvy, bool process_units){

	/*Get minimum:*/
	double minvy=this->inputs->Min(VyEnum);

	/*process units if requested: */
	if(process_units) minvy=UnitConversion(minvy,IuToExtEnum,VyEnum,this->parameters);

	/*Assign output pointers:*/
	*pminvy=minvy;
}
/*}}}*/
/*FUNCTION Tria::MinVz{{{1*/
void  Tria::MinVz(double* pminvz, bool process_units){

	/*Get minimum:*/
	double minvz=this->inputs->Min(VzEnum);

	/*process units if requested: */
	if(process_units) minvz=UnitConversion(minvz,IuToExtEnum,VzEnum,this->parameters);

	/*Assign output pointers:*/
	*pminvz=minvz;
}
/*}}}*/
/*FUNCTION Tria::TimeAdapt{{{1*/
double  Tria::TimeAdapt(void){

	/*intermediary: */
	int    i;
	double C,dt;
	double dx,dy;
	double maxx,minx;
	double maxy,miny;
	double maxabsvx,maxabsvy;
	double xyz_list[NUMVERTICES][3];

	/*get CFL coefficient:*/
	this->parameters->FindParam(&C,CflCoefficientEnum);

	/*Get for Vx and Vy, the max of abs value: */
	this->MaxAbsVx(&maxabsvx,false);
	this->MaxAbsVy(&maxabsvy,false);

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], this->nodes, NUMVERTICES);

	minx=xyz_list[0][0];
	maxx=xyz_list[0][0];
	miny=xyz_list[0][1];
	maxy=xyz_list[0][1];
	
	for(i=1;i<NUMVERTICES;i++){
		if (xyz_list[i][0]<minx)minx=xyz_list[i][0];
		if (xyz_list[i][0]>maxx)maxx=xyz_list[i][0];
		if (xyz_list[i][1]<miny)miny=xyz_list[i][1];
		if (xyz_list[i][1]>maxy)maxy=xyz_list[i][1];
	}
	dx=maxx-minx;
	dy=maxy-miny;

	/*CFL criterion: */
	dt=C/(maxabsvy/dx+maxabsvy/dy);

	return dt;
}
/*}}}*/
/*FUNCTION Tria::ThicknessAbsMisfit {{{1*/
double Tria::ThicknessAbsMisfit(bool process_units){

	/* Constants */
	const int    numdof=1*NUMVERTICES;

	/*Intermediaries*/
	int        i,ig;
	double     thickness,thicknessobs,weight;
	double     Jdet;
	double     Jelem = 0;
	double     xyz_list[NUMVERTICES][3];
	GaussTria *gauss = NULL;

	/*If on water, return 0: */
	if(IsOnWater())return 0;

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);

	/*Retrieve all inputs we will be needing: */
	Input* thickness_input   =inputs->GetInput(ThicknessEnum);   ISSMASSERT(thickness_input);
	Input* thicknessobs_input=inputs->GetInput(ThicknessObsEnum);ISSMASSERT(thicknessobs_input);
	Input* weights_input     =inputs->GetInput(WeightsEnum);     ISSMASSERT(weights_input);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussTria(2);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss);

		/*Get parameters at gauss point*/
		thickness_input->GetParameterValue(&thickness,gauss);
		thicknessobs_input->GetParameterValue(&thicknessobs,gauss);
		weights_input->GetParameterValue(&weight,gauss);

		/*compute ThicknessAbsMisfit*/
		Jelem+=0.5*pow(thickness-thicknessobs,2.0)*weight*Jdet*gauss->weight;
	}

	/* clean up and Return: */
	delete gauss;
	return Jelem;
}
/*}}}*/
/*FUNCTION Tria::SurfaceAbsVelMisfit {{{1*/
double Tria::SurfaceAbsVelMisfit(bool process_units){

	const int    numdof=NDOF2*NUMVERTICES;

	int        i,ig;
	double     Jelem=0;
	double     velocity_mag,obs_velocity_mag;
	double     meanvel, epsvel,misfit,Jdet;
	double     xyz_list[NUMVERTICES][3];
	double     vx_list[NUMVERTICES];
	double     vy_list[NUMVERTICES];
	double     obs_vx_list[NUMVERTICES];
	double     obs_vy_list[NUMVERTICES];
	double     misfit_square_list[NUMVERTICES];
	double     misfit_list[NUMVERTICES];
	double     weights_list[NUMVERTICES];
	GaussTria *gauss=NULL;

	/*If on water, return 0: */
	if(IsOnWater())return 0;

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);

	/* Recover input data: */
	GetParameterListOnVertices(&obs_vx_list[0],VxObsEnum);
	GetParameterListOnVertices(&obs_vy_list[0],VyObsEnum);
	GetParameterListOnVertices(&vx_list[0],VxEnum);
	GetParameterListOnVertices(&vy_list[0],VyEnum);
	GetParameterListOnVertices(&weights_list[0],WeightsEnum);

	/*retrieve some parameters: */
	this->parameters->FindParam(&meanvel,MeanVelEnum);
	this->parameters->FindParam(&epsvel,EpsVelEnum);
	
	/* Compute SurfaceAbsVelMisfit at the 3 nodes
	 * Here we integrate linearized functions:
	 *               
	 * J(E) = int_E   sum_{i=1}^3  J_i Phi_i
	 *
	 * where J_i are the misfits at the 3 nodes of the triangle
	 *       Phi_i is the nodal function (P1) with respect to 
	 *       the vertex i
	 */

	/*We are using an absolute misfit:
	 *
	 *      1  [           2              2 ]
	 * J = --- | (u - u   )  +  (v - v   )  |
	 *      2  [       obs            obs   ]
	 *
	 */
	for (i=0;i<NUMVERTICES;i++){
		misfit_list[i]=0.5*(pow((vx_list[i]-obs_vx_list[i]),(double)2)+pow((vy_list[i]-obs_vy_list[i]),(double)2));
	}
	/*Process units: */
	if(process_units)UnitConversion(&misfit_list[0],NUMVERTICES,IuToExtEnum,SurfaceAbsVelMisfitEnum,this->parameters);

	/*Apply weights to misfits*/
	for (i=0;i<NUMVERTICES;i++) misfit_list[i]=weights_list[i]*misfit_list[i];

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussTria(2);
	for (ig=gauss->begin();ig<gauss->end(); ig++){

		gauss->GaussPoint(ig);

		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss);
		TriaRef::GetParameterValue(&misfit, &misfit_list[0],gauss);
		Jelem+=misfit*Jdet*gauss->weight;
	}

	/*clean up and Return: */
	delete gauss;
	return Jelem;
}
/*}}}*/
/*FUNCTION Tria::SurfaceRelVelMisfit {{{1*/
double Tria::SurfaceRelVelMisfit(bool process_units){

	const int    numdof=2*NUMVERTICES;

	int        i,ig;
	double     Jelem=0;
	double     scalex=1,scaley=1;
	double     meanvel, epsvel,misfit,Jdet;
	double     velocity_mag,obs_velocity_mag;
	double     xyz_list[NUMVERTICES][3];
	double     vx_list[NUMVERTICES];
	double     vy_list[NUMVERTICES];
	double     obs_vx_list[NUMVERTICES];
	double     obs_vy_list[NUMVERTICES];
	double     misfit_square_list[NUMVERTICES];
	double     misfit_list[NUMVERTICES];
	double     weights_list[NUMVERTICES];
	GaussTria *gauss=NULL;

	/*If on water, return 0: */
	if(IsOnWater())return 0;

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);

	/* Recover input data: */
	GetParameterListOnVertices(&obs_vx_list[0],VxObsEnum);
	GetParameterListOnVertices(&obs_vy_list[0],VyObsEnum);
	GetParameterListOnVertices(&vx_list[0],VxEnum);
	GetParameterListOnVertices(&vy_list[0],VyEnum);
	GetParameterListOnVertices(&weights_list[0],WeightsEnum);

	/*retrieve some parameters: */
	this->parameters->FindParam(&meanvel,MeanVelEnum);
	this->parameters->FindParam(&epsvel,EpsVelEnum);

	/* Compute SurfaceRelVelMisfit at the 3 nodes
	 * Here we integrate linearized functions:
	 *               
	 * J(E) = int_E   sum_{i=1}^3  J_i Phi_i
	 *
	 * where J_i are the misfits at the 3 nodes of the triangle
	 *       Phi_i is the nodal function (P1) with respect to 
	 *       the vertex i
	 */

	/*We are using a relative misfit: 
	 *                        
	 *      1  [     \bar{v}^2             2   \bar{v}^2              2 ]
	 * J = --- | -------------  (u - u   ) + -------------  (v - v   )  |
	 *      2  [  (u   + eps)^2       obs    (v   + eps)^2       obs    ]
	 *              obs                        obs                      
	 */
	for (i=0;i<NUMVERTICES;i++){
		scalex=pow(meanvel/(obs_vx_list[i]+epsvel),(double)2);
		scaley=pow(meanvel/(obs_vy_list[i]+epsvel),(double)2);
		if(obs_vx_list[i]==0)scalex=0;
		if(obs_vy_list[i]==0)scaley=0;
		misfit_list[i]=0.5*(scalex*pow((vx_list[i]-obs_vx_list[i]),2)+scaley*pow((vy_list[i]-obs_vy_list[i]),2));
	}

	/*Process units: */
	if(process_units)UnitConversion(&misfit_list[0],NUMVERTICES,IuToExtEnum,SurfaceRelVelMisfitEnum,this->parameters);

	/*Apply weights to misfits*/
	for (i=0;i<NUMVERTICES;i++) misfit_list[i]=weights_list[i]*misfit_list[i];

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussTria(2);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss);
		TriaRef::GetParameterValue(&misfit, &misfit_list[0],gauss);
		Jelem+=misfit*Jdet*gauss->weight;
	}

	/*clean up and Return: */
	delete gauss;
	return Jelem;
}
/*}}}*/
/*FUNCTION Tria::SurfaceLogVelMisfit {{{1*/
double Tria::SurfaceLogVelMisfit(bool process_units){

	const int    numdof=NDOF2*NUMVERTICES;

	int        i,ig;
	double     Jelem=0;
	double     scalex=1,scaley=1;
	double     meanvel, epsvel,misfit,Jdet;
	double     velocity_mag,obs_velocity_mag;
	double     xyz_list[NUMVERTICES][3];
	double     vx_list[NUMVERTICES];
	double     vy_list[NUMVERTICES];
	double     obs_vx_list[NUMVERTICES];
	double     obs_vy_list[NUMVERTICES];
	double     misfit_square_list[NUMVERTICES];
	double     misfit_list[NUMVERTICES];
	double     weights_list[NUMVERTICES];
	GaussTria *gauss=NULL;

	/*If on water, return 0: */
	if(IsOnWater())return 0;

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);

	/* Recover input data: */
	GetParameterListOnVertices(&obs_vx_list[0],VxObsEnum);
	GetParameterListOnVertices(&obs_vy_list[0],VyObsEnum);
	GetParameterListOnVertices(&vx_list[0],VxEnum);
	GetParameterListOnVertices(&vy_list[0],VyEnum);
	GetParameterListOnVertices(&weights_list[0],WeightsEnum);

	/*retrieve some parameters: */
	this->parameters->FindParam(&meanvel,MeanVelEnum);
	this->parameters->FindParam(&epsvel,EpsVelEnum);
	
	/* Compute SurfaceLogVelMisfit at the 3 nodes
	 * Here we integrate linearized functions:
	 *               
	 * J(E) = int_E   sum_{i=1}^3  J_i Phi_i
	 *
	 * where J_i are the misfits at the 3 nodes of the triangle
	 *       Phi_i is the nodal function (P1) with respect to 
	 *       the vertex i
	 */

	/*We are using a logarithmic misfit:
	 *                        
	 *                 [        vel + eps     ] 2
	 * J = 4 \bar{v}^2 | log ( -----------  ) |  
	 *                 [       vel   + eps    ]
	 *                            obs
	 */
	for (i=0;i<NUMVERTICES;i++){
		velocity_mag=sqrt(pow(vx_list[i],(double)2)+pow(vy_list[i],(double)2))+epsvel; //epsvel to avoid velocity being nil.
		obs_velocity_mag=sqrt(pow(obs_vx_list[i],(double)2)+pow(obs_vy_list[i],(double)2))+epsvel; //epsvel to avoid observed velocity being nil.
		misfit_list[i]=4*pow(meanvel,(double)2)*pow(log(velocity_mag/obs_velocity_mag),(double)2);
	}

	/*Process units: */
	if(process_units)UnitConversion(&misfit_list[0],NUMVERTICES,IuToExtEnum,SurfaceLogVelMisfitEnum,this->parameters);

	/*Apply weights to misfits*/
	for (i=0;i<NUMVERTICES;i++) misfit_list[i]=weights_list[i]*misfit_list[i];

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussTria(2);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss);
		TriaRef::GetParameterValue(&misfit, &misfit_list[0],gauss);
		Jelem+=misfit*Jdet*gauss->weight;
	}

	/*clean-up and Return: */
	delete gauss;
	return Jelem;
}
/*}}}*/
/*FUNCTION Tria::SurfaceLogVxVyMisfit {{{1*/
double Tria::SurfaceLogVxVyMisfit(bool process_units){

	const int    numdof=NDOF2*NUMVERTICES;

	int        i,ig;
	int        fit=-1;
	double     Jelem=0, S=0;
	double     scalex=1,scaley=1;
	double     meanvel, epsvel, misfit, Jdet;
	double     velocity_mag,obs_velocity_mag;
	double     xyz_list[NUMVERTICES][3];
	double     vx_list[NUMVERTICES];
	double     vy_list[NUMVERTICES];
	double     obs_vx_list[NUMVERTICES];
	double     obs_vy_list[NUMVERTICES];
	double     misfit_square_list[NUMVERTICES];
	double     misfit_list[NUMVERTICES];
	double     weights_list[NUMVERTICES];
	GaussTria *gauss=NULL;

	/*If on water, return 0: */
	if(IsOnWater())return 0;

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);

	/* Recover input data: */
	GetParameterListOnVertices(&obs_vx_list[0],VxObsEnum);
	GetParameterListOnVertices(&obs_vy_list[0],VyObsEnum);
	GetParameterListOnVertices(&vx_list[0],VxEnum);
	GetParameterListOnVertices(&vy_list[0],VyEnum);
	GetParameterListOnVertices(&weights_list[0],WeightsEnum);

	/*retrieve some parameters: */
	this->parameters->FindParam(&meanvel,MeanVelEnum);
	this->parameters->FindParam(&epsvel,EpsVelEnum);
	
	/* Compute SurfaceLogVxVyMisfit at the 3 nodes
	 * Here we integrate linearized functions:
	 *               
	 * J(E) = int_E   sum_{i=1}^3  J_i Phi_i
	 *
	 * where J_i are the misfits at the 3 nodes of the triangle
	 *       Phi_i is the nodal function (P1) with respect to 
	 *       the vertex i
	 */

	/*We are using an logarithmic 2 misfit:
	 *
	 *      1            [        |u| + eps     2          |v| + eps     2  ]
	 * J = --- \bar{v}^2 | log ( -----------  )   +  log ( -----------  )   |  
	 *      2            [       |u    |+ eps              |v    |+ eps     ]
	 *                              obs                       obs
	 */
	for (i=0;i<NUMVERTICES;i++){
		misfit_list[i]=0.5*pow(meanvel,(double)2)*(
					pow(log((fabs(vx_list[i])+epsvel)/(fabs(obs_vx_list[i])+epsvel)),(double)2) +
					pow(log((fabs(vy_list[i])+epsvel)/(fabs(obs_vy_list[i])+epsvel)),(double)2) );
	}

	/*Process units: */
	if(process_units)UnitConversion(&misfit_list[0],NUMVERTICES,IuToExtEnum,SurfaceLogVxVyMisfitEnum,this->parameters);

	/*Apply weights to misfits*/
	for (i=0;i<NUMVERTICES;i++) misfit_list[i]=weights_list[i]*misfit_list[i];

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussTria(2);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss);
		TriaRef::GetParameterValue(&misfit, &misfit_list[0],gauss);
		Jelem+=misfit*Jdet*gauss->weight;
	}

	/*clean-up and Return: */
	delete gauss;
	return Jelem;
}
/*}}}*/
/*FUNCTION Tria::SurfaceAverageVelMisfit {{{1*/
double Tria::SurfaceAverageVelMisfit(bool process_units){

	const int    numdof=2*NUMVERTICES;

	int        i,ig;
	int        fit=-1;
	double     Jelem=0,S=0;
	double     scalex=1, scaley=1;
	double     meanvel, epsvel,Jdet;
	double     velocity_mag,obs_velocity_mag,misfit;
	double     xyz_list[NUMVERTICES][3];
	double     vx_list[NUMVERTICES];
	double     vy_list[NUMVERTICES];
	double     obs_vx_list[NUMVERTICES];
	double     obs_vy_list[NUMVERTICES];
	double     misfit_square_list[NUMVERTICES];
	double     misfit_list[NUMVERTICES];
	double     weights_list[NUMVERTICES];
	GaussTria *gauss=NULL;

	/*If on water, return 0: */
	if(IsOnWater())return 0;

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);

	/* Recover input data: */
	inputs->GetParameterValue(&S,SurfaceAreaEnum);
	GetParameterListOnVertices(&obs_vx_list[0],VxObsEnum);
	GetParameterListOnVertices(&obs_vy_list[0],VyObsEnum);
	GetParameterListOnVertices(&vx_list[0],VxEnum);
	GetParameterListOnVertices(&vy_list[0],VyEnum);
	GetParameterListOnVertices(&weights_list[0],WeightsEnum);

	/*retrieve some parameters: */
	this->parameters->FindParam(&meanvel,MeanVelEnum);
	this->parameters->FindParam(&epsvel,EpsVelEnum);

	/* Compute SurfaceAverageVelMisfit at the 3 nodes
	 * Here we integrate linearized functions:
	 *               
	 * J(E) = int_E   sum_{i=1}^3  J_i Phi_i
	 *
	 * where J_i are the misfits at the 3 nodes of the triangle
	 *       Phi_i is the nodal function (P1) with respect to 
	 *       the vertex i
	 */

	/*We are using a spacially average absolute misfit:
	 *
	 *      1                    2              2
	 * J = ---  sqrt(  (u - u   )  +  (v - v   )  )
	 *      S                obs            obs
	 */
	for (i=0;i<NUMVERTICES;i++) misfit_square_list[i]=pow(vx_list[i]-obs_vx_list[i],2)+pow(vy_list[i]-obs_vx_list[i],2);

	/*Process units: */
	if(process_units)UnitConversion(&misfit_square_list[0],NUMVERTICES,IuToExtEnum,SurfaceAverageVelMisfitEnum,this->parameters);

	/*Take the square root, and scale by surface: */
	for (i=0;i<NUMVERTICES;i++)misfit_list[i]=pow(misfit_square_list[i],2)/S;

	/*Apply weights to misfits*/
	for (i=0;i<NUMVERTICES;i++) misfit_list[i]=weights_list[i]*misfit_list[i];

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussTria(2);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss);
		TriaRef::GetParameterValue(&misfit, &misfit_list[0],gauss);
		Jelem+=misfit*Jdet*gauss->weight;
	}

	/*clean-up and Return: */
	delete gauss;
	return Jelem;
}
/*}}}*/
/*FUNCTION Tria::PatchFill{{{1*/
void  Tria::PatchFill(int* prow, Patch* patch){

	int i,row;
	int vertices_ids[3];

	/*recover pointer: */
	row=*prow;
		
	for(i=0;i<3;i++) vertices_ids[i]=nodes[i]->GetVertexId(); //vertices id start at column 3 of the patch.

	for(i=0;i<this->results->Size();i++){
		ElementResult* elementresult=(ElementResult*)this->results->GetObjectByOffset(i);

		/*For this result,fill the information in the Patch object (element id + vertices ids), and then hand 
		 *it to the result object, to fill the rest: */
		patch->fillelementinfo(row,this->id,vertices_ids,3);
		elementresult->PatchFill(row,patch);

		/*increment rower: */
		row++;
	}

	/*Assign output pointers:*/
	*prow=row;
}
/*}}}*/
/*FUNCTION Tria::PatchSize{{{1*/
void  Tria::PatchSize(int* pnumrows, int* pnumvertices,int* pnumnodes){

	int     i;
	int     numrows     = 0;
	int     numnodes    = 0;

	/*Go through all the results objects, and update the counters: */
	for (i=0;i<this->results->Size();i++){
		ElementResult* elementresult=(ElementResult*)this->results->GetObjectByOffset(i);
		/*first, we have one more result: */
		numrows++;
		/*now, how many vertices and how many nodal values for this result? :*/
		numnodes=elementresult->NumberOfNodalValues(); //ask result object.
	}

	/*Assign output pointers:*/
	*pnumrows=numrows;
	*pnumvertices=NUMVERTICES;
	*pnumnodes=numnodes;
}
/*}}}*/
/*FUNCTION Tria::ProcessResultsUnits{{{1*/
void  Tria::ProcessResultsUnits(void){

	int i;

	for(i=0;i<this->results->Size();i++){
		ElementResult* elementresult=(ElementResult*)this->results->GetObjectByOffset(i);
		elementresult->ProcessUnits(this->parameters);
	}
}
/*}}}*/
/*FUNCTION Tria::SurfaceArea {{{1*/
double Tria::SurfaceArea(void){

	int    i;
	double S;
	double normal[3];
	double v13[3],v23[3];
	double xyz_list[NUMVERTICES][3];

	/*If on water, return 0: */
	if(IsOnWater())return 0;

	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);

	for (i=0;i<3;i++){
		v13[i]=xyz_list[0][i]-xyz_list[2][i];
		v23[i]=xyz_list[1][i]-xyz_list[2][i];
	}

	normal[0]=v13[1]*v23[2]-v13[2]*v23[1];
	normal[1]=v13[2]*v23[0]-v13[0]*v23[2];
	normal[2]=v13[0]*v23[1]-v13[1]*v23[0];

	S = 0.5 * sqrt(pow(normal[0],(double)2)+pow(normal[1],(double)2)+pow(normal[2],(double)2));

	/*Return: */
	return S;
}
/*}}}*/
/*FUNCTION Tria::Update{{{1*/
void Tria::Update(int index, IoModel* iomodel,int analysis_counter,int analysis_type){ //i is the element index

	/*Intermediaries*/
	int    i;
	int    tria_node_ids[3];
	int    tria_vertex_ids[3];
	int    tria_type;
	double nodeinputs[3];

	/*Checks if debuging*/
	/*{{{2*/
	ISSMASSERT(iomodel->elements);
	/*}}}*/

	/*Recover element type*/
	if ((analysis_type==PrognosticAnalysisEnum || analysis_type==BalancedthicknessAnalysisEnum) && iomodel->prognostic_DG){

		/*P1 Discontinuous Galerkin*/
		tria_type=P1DGEnum;
	}
	else{
		/*P1 Continuous Galerkin*/
		tria_type=P1Enum;
	}
	this->SetElementType(tria_type,analysis_counter);

	/*Recover vertices ids needed to initialize inputs*/
	for(i=0;i<3;i++){ 
		tria_vertex_ids[i]=(int)iomodel->elements[3*index+i]; //ids for vertices are in the elements array from Matlab
	}

	/*Recover nodes ids needed to initialize the node hook.*/
	if (tria_type==P1DGEnum){
		/*Discontinuous Galerkin*/
		tria_node_ids[0]=iomodel->nodecounter+3*index+1;
		tria_node_ids[1]=iomodel->nodecounter+3*index+2;
		tria_node_ids[2]=iomodel->nodecounter+3*index+3;
	}
	else{
		/*Continuous Galerkin*/
		for(i=0;i<3;i++){ 
			tria_node_ids[i]=iomodel->nodecounter+(int)*(iomodel->elements+3*index+i); //ids for vertices are in the elements array from Matlab
		}
	}

	/*hooks: */
	this->SetHookNodes(tria_node_ids,analysis_counter); this->nodes=NULL; //set hook to nodes, for this analysis type
	
	/*add as many inputs per element as requested:*/
	if (iomodel->thickness) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->thickness[tria_vertex_ids[i]-1];
		this->inputs->AddInput(new TriaVertexInput(ThicknessEnum,nodeinputs));
	}
	if (iomodel->surface) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->surface[tria_vertex_ids[i]-1];
		this->inputs->AddInput(new TriaVertexInput(SurfaceEnum,nodeinputs));
	}
	if (iomodel->bed) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->bed[tria_vertex_ids[i]-1];
		this->inputs->AddInput(new TriaVertexInput(BedEnum,nodeinputs));
	}
	if (iomodel->drag_coefficient) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->drag_coefficient[tria_vertex_ids[i]-1];
		this->inputs->AddInput(new TriaVertexInput(DragCoefficientEnum,nodeinputs));

		if (iomodel->drag_p) this->inputs->AddInput(new DoubleInput(DragPEnum,iomodel->drag_p[index]));
		if (iomodel->drag_q) this->inputs->AddInput(new DoubleInput(DragQEnum,iomodel->drag_q[index]));
		this->inputs->AddInput(new IntInput(DragTypeEnum,iomodel->drag_type));
	}
	if (iomodel->control_parameter) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->control_parameter[tria_vertex_ids[i]-1];
		this->inputs->AddInput(new TriaVertexInput(ControlParameterEnum,nodeinputs));
	}
	if (iomodel->thickness_obs) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->thickness_obs[tria_vertex_ids[i]-1];
		this->inputs->AddInput(new TriaVertexInput(ThicknessObsEnum,nodeinputs));
	}
	if (iomodel->melting_rate) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->melting_rate[tria_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new TriaVertexInput(MeltingRateEnum,nodeinputs));
	}
	if (iomodel->accumulation_rate) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->accumulation_rate[tria_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new TriaVertexInput(AccumulationRateEnum,nodeinputs));
	}
	if (iomodel->geothermalflux) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->geothermalflux[tria_vertex_ids[i]-1];
		this->inputs->AddInput(new TriaVertexInput(GeothermalFluxEnum,nodeinputs));
	}
	if (iomodel->dhdt) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->dhdt[tria_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new TriaVertexInput(DhDtEnum,nodeinputs));
	}
	if (iomodel->pressure) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->pressure[tria_vertex_ids[i]-1];
		this->inputs->AddInput(new TriaVertexInput(PressureEnum,nodeinputs));
	}
	if (iomodel->temperature) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->temperature[tria_vertex_ids[i]-1];
		this->inputs->AddInput(new TriaVertexInput(TemperatureEnum,nodeinputs));
	}
	/*vx,vy and vz: */
	if (iomodel->vx) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->vx[tria_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new TriaVertexInput(VxEnum,nodeinputs));
		this->inputs->AddInput(new TriaVertexInput(VxOldEnum,nodeinputs));
		if(iomodel->qmu_analysis)this->inputs->AddInput(new TriaVertexInput(QmuVxEnum,nodeinputs));
	}
	if (iomodel->vy) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->vy[tria_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new TriaVertexInput(VyEnum,nodeinputs));
		this->inputs->AddInput(new TriaVertexInput(VyOldEnum,nodeinputs));
		if(iomodel->qmu_analysis)this->inputs->AddInput(new TriaVertexInput(QmuVyEnum,nodeinputs));
	}
	if (iomodel->vz) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->vz[tria_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new TriaVertexInput(VzEnum,nodeinputs));
		this->inputs->AddInput(new TriaVertexInput(VzOldEnum,nodeinputs));
		if(iomodel->qmu_analysis)this->inputs->AddInput(new TriaVertexInput(QmuVzEnum,nodeinputs));
	}
	if (iomodel->vx_obs) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->vx_obs[tria_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new TriaVertexInput(VxObsEnum,nodeinputs));
	}
	if (iomodel->vy_obs) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->vy_obs[tria_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new TriaVertexInput(VyObsEnum,nodeinputs));
	}
	if (iomodel->vz_obs) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->vz_obs[tria_vertex_ids[i]-1]/iomodel->yts;
		this->inputs->AddInput(new TriaVertexInput(VzObsEnum,nodeinputs));
	}
	if (iomodel->weights) {
		for(i=0;i<3;i++)nodeinputs[i]=iomodel->weights[tria_vertex_ids[i]-1];
		this->inputs->AddInput(new TriaVertexInput(WeightsEnum,nodeinputs));
	}
	if (iomodel->elementoniceshelf) this->inputs->AddInput(new BoolInput(ElementOnIceShelfEnum,(IssmBool)iomodel->elementoniceshelf[index]));
	if (iomodel->elementonbed) this->inputs->AddInput(new BoolInput(ElementOnBedEnum,(IssmBool)iomodel->elementonbed[index]));
	if (iomodel->elementonwater) this->inputs->AddInput(new BoolInput(ElementOnWaterEnum,(IssmBool)iomodel->elementonwater[index]));
	if (iomodel->elementonsurface) this->inputs->AddInput(new BoolInput(ElementOnSurfaceEnum,(IssmBool)iomodel->elementonsurface[index]));

	/*Defaults if not provided in iomodel*/
	switch(analysis_type){

		case DiagnosticHorizAnalysisEnum:

			/*default vx,vy and vz: either observation or 0 */
			if(!iomodel->vx){
				if (iomodel->vx_obs) for(i=0;i<3;i++)nodeinputs[i]=iomodel->vx_obs[tria_vertex_ids[i]-1]/iomodel->yts;
				else                 for(i=0;i<3;i++)nodeinputs[i]=0;
				this->inputs->AddInput(new TriaVertexInput(VxEnum,nodeinputs));
				this->inputs->AddInput(new TriaVertexInput(VxOldEnum,nodeinputs));
				if(iomodel->qmu_analysis) this->inputs->AddInput(new TriaVertexInput(QmuVxEnum,nodeinputs));
			}
			if(!iomodel->vy){
				if (iomodel->vy_obs) for(i=0;i<3;i++)nodeinputs[i]=iomodel->vy_obs[tria_vertex_ids[i]-1]/iomodel->yts;
				else                 for(i=0;i<3;i++)nodeinputs[i]=0;
				this->inputs->AddInput(new TriaVertexInput(VyEnum,nodeinputs));
				this->inputs->AddInput(new TriaVertexInput(VyOldEnum,nodeinputs));
				if(iomodel->qmu_analysis) this->inputs->AddInput(new TriaVertexInput(QmuVyEnum,nodeinputs));
			}
			if(!iomodel->vz){
				if (iomodel->vz_obs) for(i=0;i<3;i++)nodeinputs[i]=iomodel->vz_obs[tria_vertex_ids[i]-1]/iomodel->yts;
				else                 for(i=0;i<3;i++)nodeinputs[i]=0;
				this->inputs->AddInput(new TriaVertexInput(VzEnum,nodeinputs));
				this->inputs->AddInput(new TriaVertexInput(VzOldEnum,nodeinputs));
				if(iomodel->qmu_analysis) this->inputs->AddInput(new TriaVertexInput(QmuVzEnum,nodeinputs));
			}
			if(!iomodel->pressure){
				for(i=0;i<3;i++)nodeinputs[i]=0;
				if(iomodel->qmu_analysis){
					this->inputs->AddInput(new TriaVertexInput(PressureEnum,nodeinputs));
					this->inputs->AddInput(new TriaVertexInput(QmuPressureEnum,nodeinputs));
				}
			}
			break;

		default:
			/*No update for other solution types*/
			break;

	}

	//this->parameters: we still can't point to it, it may not even exist. Configure will handle this.
	this->parameters=NULL;

}
/*}}}*/
/*FUNCTION Tria::UpdateGeometry{{{1*/
void  Tria::UpdateGeometry(void){

	/*Intermediaries*/
	double rho_ice,rho_water;

	/*Correct thickness*/
	this->inputs->ConstrainMin(ThicknessEnum,1.0);

	/*If shelf: hydrostatic equilibrium*/
	if (this->IsOnShelf()){

		/*recover material parameters: */
		rho_ice=matpar->GetRhoIce();
		rho_water=matpar->GetRhoWater();

		/*Create New Surface: s = (1-rho_ice/rho_water) h*/
		this->inputs->DuplicateInput(ThicknessEnum,SurfaceEnum);     //1: copy thickness into surface
		InputScale(SurfaceEnum,(1.0-rho_ice/rho_water)); //2: surface = surface * (1-di)

		/*Create New Bed b = -rho_ice/rho_water h*/
		this->inputs->DuplicateInput(ThicknessEnum,BedEnum);         //1: copy thickness into bed
		InputScale(BedEnum, -rho_ice/rho_water);       //2: bed = bed * (-di)
	}

	/*If sheet: surface = bed + thickness*/
	else{
		
		/*The bed does not change, update surface only s = b + h*/
		this->inputs->DuplicateInput(BedEnum,SurfaceEnum);          //1: copy bed into surface
		this->inputs->AXPY(SurfaceEnum,1.0,ThicknessEnum);     //2: surface = surface + 1 * thickness
	}
}
/*}}}*/

/*Tria specific routines: */
/*FUNCTION Tria::CreateKMatrixAdjointBalancedthickness {{{1*/
ElementMatrix* Tria::CreateKMatrixAdjointBalancedthickness(void){

	ElementMatrix* Ke=NULL;

	/*Get Element Matrix of the forward model*/
	switch(GetElementType()){
		case P1Enum:
			Ke=CreateKMatrixBalancedthickness_CG();
			break;
		case P1DGEnum:
			Ke=CreateKMatrixBalancedthickness_DG();
			break;
		default:
			ISSMERROR("Element type %s not supported yet",EnumToString(GetElementType()));
	}

	/*Transpose and return Ke*/
	Ke->Transpose();
	return Ke;
}
/*}}}*/
/*FUNCTION Tria::CreateKMatrixBalancedthickness {{{1*/
ElementMatrix* Tria::CreateKMatrixBalancedthickness(void){

	switch(GetElementType()){
		case P1Enum:
			return CreateKMatrixBalancedthickness_CG();
		case P1DGEnum:
			return CreateKMatrixBalancedthickness_DG();
		default:
			ISSMERROR("Element type %s not supported yet",EnumToString(GetElementType()));
	}

}
/*}}}*/
/*FUNCTION Tria::CreateKMatrixBalancedthickness_CG {{{1*/
ElementMatrix* Tria::CreateKMatrixBalancedthickness_CG(void){

	/*Constants*/
	const int    numdof=NDOF1*NUMVERTICES;

	/*Intermediaries */
	bool       artdiff;
	int        i,j,ig,dim;
	double     Jdettria ,vx,vy,dvxdx,dvydy;
	double     dvx[2],dvy[2];
	double     xyz_list[NUMVERTICES][3];
	double     L[NUMVERTICES];
	double     B[2][NUMVERTICES];
	double     Bprime[2][NUMVERTICES];
	double     K[2][2]                          = {0.0};
	double     KDL[2][2]                        = {0.0};
	double     DL[2][2]                         = {0.0};
	double     DLprime[2][2]                    = {0.0};
	double     DL_scalar;
	double     Ke_gg_gaussian[numdof][numdof]   = {0.0};
	double     Ke_gg_thickness1[numdof][numdof] = {0.0};
	double     Ke_gg_thickness2[numdof][numdof] = {0.0};
	GaussTria *gauss                            = NULL;

	/*Initialize Element matrix and return if necessary*/
	if(IsOnWater()) return NULL;
	ElementMatrix* Ke=new ElementMatrix(nodes,NUMVERTICES,this->parameters,NoneApproximationEnum);

	/*Retrieve all Inputs and parameters: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	this->parameters->FindParam(&artdiff,ArtDiffEnum);
	this->parameters->FindParam(&dim,DimEnum);
	Input* vxaverage_input=NULL;
	Input* vyaverage_input=NULL;
	if(dim==2){
		vxaverage_input=inputs->GetInput(VxEnum); ISSMASSERT(vxaverage_input);
		vyaverage_input=inputs->GetInput(VyEnum); ISSMASSERT(vyaverage_input);
	}
	else{
		vxaverage_input=inputs->GetInput(VxAverageEnum); ISSMASSERT(vxaverage_input);
		vyaverage_input=inputs->GetInput(VyAverageEnum); ISSMASSERT(vyaverage_input);
	}

	/*Create Artificial diffusivity once for all if requested*/
	if(artdiff){
		gauss=new GaussTria();
		gauss->GaussCenter();
		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss);
		delete gauss;

		vxaverage_input->GetParameterAverage(&vx);
		vyaverage_input->GetParameterAverage(&vy);
		K[0][0]=pow(Jdettria,(double).5)/2.0*fabs(vx);
		K[1][1]=pow(Jdettria,(double).5)/2.0*fabs(vy);
	}

	/*Start looping on the number of gaussian points:*/
	gauss=new GaussTria(2);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss);
		GetBPrognostic(&B[0][0], &xyz_list[0][0], gauss);
		GetBprimePrognostic(&Bprime[0][0], &xyz_list[0][0], gauss);

		vxaverage_input->GetParameterValue(&vx,gauss);
		vyaverage_input->GetParameterValue(&vy,gauss);
		vxaverage_input->GetParameterDerivativeValue(&dvx[0],&xyz_list[0][0],gauss);
		vyaverage_input->GetParameterDerivativeValue(&dvy[0],&xyz_list[0][0],gauss);

		dvxdx=dvx[0];
		dvydy=dvy[1];
		DL_scalar=gauss->weight*Jdettria;

		DL[0][0]=DL_scalar*dvxdx;
		DL[1][1]=DL_scalar*dvydy;

		DLprime[0][0]=DL_scalar*vx;
		DLprime[1][1]=DL_scalar*vy;

		TripleMultiply( &B[0][0],2,numdof,1,
					&DL[0][0],2,2,0,
					&B[0][0],2,numdof,0,
					&Ke_gg_thickness1[0][0],0);

		TripleMultiply( &B[0][0],2,numdof,1,
					&DLprime[0][0],2,2,0,
					&Bprime[0][0],2,numdof,0,
					&Ke_gg_thickness2[0][0],0);

		for(i=0;i<numdof;i++) for(j=0;j<numdof;j++) Ke->values[i*numdof+j]+=Ke_gg_thickness1[i][j];
		for(i=0;i<numdof;i++) for(j=0;j<numdof;j++) Ke->values[i*numdof+j]+=Ke_gg_thickness2[i][j];

		if(artdiff){
			KDL[0][0]=DL_scalar*K[0][0];
			KDL[1][1]=DL_scalar*K[1][1];

			TripleMultiply( &Bprime[0][0],2,numdof,1,
						&KDL[0][0],2,2,0,
						&Bprime[0][0],2,numdof,0,
						&Ke_gg_gaussian[0][0],0);

			for(i=0;i<numdof;i++) for(j=0;j<numdof;j++) Ke->values[i*numdof+j]+=Ke_gg_gaussian[i][j];
		}
	}

	/*Clean up and return*/
	delete gauss;
	return Ke;
}
/*}}}*/
/*FUNCTION Tria::CreateKMatrixBalancedthickness_DG {{{1*/
ElementMatrix* Tria::CreateKMatrixBalancedthickness_DG(void){

	/*Constants*/
	const int  numdof=NDOF1*NUMVERTICES;

	/*Intermediaries*/
	int        i,j,ig,dim;
	double     vx,vy,Jdettria;
	double     xyz_list[NUMVERTICES][3];
	double     B[2][NUMVERTICES];
	double     Bprime[2][NUMVERTICES];
	double     DL[2][2]={0.0};
	double     DL_scalar;
	double     Ke_gg[numdof][numdof]={0.0};
	GaussTria  *gauss=NULL;

	/*Initialize Element matrix and return if necessary*/
	if(IsOnWater()) return NULL;
	ElementMatrix* Ke=new ElementMatrix(nodes,NUMVERTICES,this->parameters,NoneApproximationEnum);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	this->parameters->FindParam(&dim,DimEnum);
	Input* vx_input=inputs->GetInput(VxEnum); ISSMASSERT(vx_input);
	Input* vy_input=inputs->GetInput(VyEnum); ISSMASSERT(vy_input);

	/*Start looping on the number of gaussian points:*/
	gauss=new GaussTria(2);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss);
		/*WARNING: B and Bprime are inverted compared to usual prognostic!!!!*/
		GetBPrognostic(&Bprime[0][0], &xyz_list[0][0], gauss);
		GetBprimePrognostic(&B[0][0], &xyz_list[0][0], gauss);

		vx_input->GetParameterValue(&vx,gauss);
		vy_input->GetParameterValue(&vy,gauss);

		DL_scalar=-gauss->weight*Jdettria;
		DL[0][0]=DL_scalar*vx;
		DL[1][1]=DL_scalar*vy;

		TripleMultiply( &B[0][0],2,numdof,1,
					&DL[0][0],2,2,0,
					&Bprime[0][0],2,numdof,0,
					&Ke_gg[0][0],0);

		for(i=0;i<numdof;i++) for(j=0;j<numdof;j++) Ke->values[i*numdof+j]+=Ke_gg[i][j];
	}

	/*Clean up and return*/
	delete gauss;
	return Ke;
}
/*}}}*/
/*FUNCTION Tria::CreateKMatrixBalancedvelocities {{{1*/
ElementMatrix* Tria::CreateKMatrixBalancedvelocities(void){

	/*Constants*/
	const int    numdof=NDOF1*NUMVERTICES;

	/*Intermediaries */
	bool    artdiff;
	int     i,j,ig,dim;
	double  nx,ny,norm,Jdettria;
	double  dvx[2],dvy[2];
	double  vx,vy,dvxdx,dvydy;
	double  v_gauss[2]={0.0};
	double  surface_normal[3];
	double  surface_list[3];
	double  xyz_list[NUMVERTICES][3];
	double  B[2][NUMVERTICES];
	double  Bprime[2][NUMVERTICES];
	double  K[2][2]={0.0};
	double  KDL[2][2]={0.0};
	double  DLprime[2][2]={0.0};
	double  DL_scalar;
	double  Ke_gg_gaussian[numdof][numdof]   = {0.0}; 
	double  Ke_gg_velocities[numdof][numdof] = {0.0}; 
	GaussTria *gauss=NULL;

	/*Initialize Element matrix and return if necessary*/
	if(IsOnWater()) return NULL;
	ElementMatrix* Ke=new ElementMatrix(nodes,NUMVERTICES,this->parameters,NoneApproximationEnum);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	this->parameters->FindParam(&artdiff,ArtDiffEnum);
	this->parameters->FindParam(&dim,DimEnum);
	Input* surface_input=inputs->GetInput(SurfaceEnum); ISSMASSERT(surface_input);
	Input* vxaverage_input=NULL;
	Input* vyaverage_input=NULL;
	if(dim==2){
		vxaverage_input=inputs->GetInput(VxEnum); ISSMASSERT(vxaverage_input);
		vyaverage_input=inputs->GetInput(VyEnum); ISSMASSERT(vyaverage_input);
	}
	else{
		vxaverage_input=inputs->GetInput(VxAverageEnum); ISSMASSERT(vxaverage_input);
		vyaverage_input=inputs->GetInput(VyAverageEnum); ISSMASSERT(vyaverage_input);
	}

	/*Modify z so that it reflects the surface*/
	GetParameterListOnVertices(&surface_list[0],SurfaceEnum);
	for(i=0;i<NUMVERTICES;i++) xyz_list[i][2]=surface_list[i];

	/*Get normal vector to the surface*/
	vxaverage_input->GetParameterAverage(&nx);
	vyaverage_input->GetParameterAverage(&ny);
	if(nx==0 && ny==0){
		SurfaceNormal(&surface_normal[0],xyz_list);
		nx=surface_normal[0];
		ny=surface_normal[1];
	}
	if(nx==0 && ny==0){
		nx=0; ny=1;
	}
	norm=pow( pow(nx,2)+pow(ny,2) , (double).5);
	nx=nx/norm; ny=ny/norm;

	//Create Artificial diffusivity once for all if requested
	if(artdiff){
		gauss=new GaussTria();
		gauss->GaussCenter();
		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss);
		delete gauss;

		vxaverage_input->GetParameterAverage(&v_gauss[0]);
		vyaverage_input->GetParameterAverage(&v_gauss[1]);
		K[0][0]=pow(10,2)*pow(Jdettria,(double).5)/2.0*fabs(v_gauss[0]); //pow should be zero!!
		K[1][1]=pow(10,2)*pow(Jdettria,(double).5)/2.0*fabs(v_gauss[1]);
	}

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussTria(2);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss);
		GetBPrognostic(&B[0][0], &xyz_list[0][0], gauss);
		GetBprimePrognostic(&Bprime[0][0], &xyz_list[0][0], gauss);

		vxaverage_input->GetParameterValue(&vx,gauss);
		vyaverage_input->GetParameterValue(&vy,gauss);
		vxaverage_input->GetParameterDerivativeValue(&dvx[0],&xyz_list[0][0],gauss);
		vyaverage_input->GetParameterDerivativeValue(&dvy[0],&xyz_list[0][0],gauss);

		dvxdx=dvx[0];
		dvydy=dvy[1];
		DL_scalar=gauss->weight*Jdettria;

		DLprime[0][0]=DL_scalar*nx;
		DLprime[1][1]=DL_scalar*ny;

		TripleMultiply( &B[0][0],2,numdof,1,
					&DLprime[0][0],2,2,0,
					&Bprime[0][0],2,numdof,0,
					&Ke_gg_velocities[0][0],0);

		for(i=0;i<numdof;i++) for(j=0;j<numdof;j++) Ke->values[i*numdof+j]+=Ke_gg_velocities[i][j];

		if(artdiff){
			KDL[0][0]=DL_scalar*K[0][0];
			KDL[1][1]=DL_scalar*K[1][1];

			TripleMultiply( &Bprime[0][0],2,numdof,1,
						&KDL[0][0],2,2,0,
						&Bprime[0][0],2,numdof,0,
						&Ke_gg_gaussian[0][0],0);

			for(i=0;i<numdof;i++) for(j=0;j<numdof;j++) Ke->values[i*numdof+j]+=Ke_gg_gaussian[i][j];
		}
	}

	/*Clean up and return*/
	delete gauss;
	return Ke;
}
/*}}}*/
/*FUNCTION Tria::CreateKMatrixDiagnosticMacAyeal {{{1*/
ElementMatrix* Tria::CreateKMatrixDiagnosticMacAyeal(void){
	
	/*compute all stiffness matrices for this element*/
	ElementMatrix* Ke1=CreateKMatrixDiagnosticMacAyealViscous();
	ElementMatrix* Ke2=CreateKMatrixDiagnosticMacAyealFriction();
	ElementMatrix* Ke =new ElementMatrix(Ke1,Ke2);

	/*clean-up and return*/
	delete Ke1;
	delete Ke2;
	return Ke;

}
/*}}}*/
/*FUNCTION Tria::CreateKMatrixDiagnosticMacAyealViscous{{{1*/
ElementMatrix* Tria::CreateKMatrixDiagnosticMacAyealViscous(void){

	/*Constants*/
	const int  numdof=NDOF2*NUMVERTICES;

	/*Intermediaries*/
	int        i,j,ig;
	double     xyz_list[NUMVERTICES][3];
	double     viscosity,newviscosity,oldviscosity;
	double     viscosity_overshoot,thickness,Jdet;
	double     epsilon[3],oldepsilon[3];    /* epsilon=[exx,eyy,exy];    */
	double     B[3][numdof];
	double     Bprime[3][numdof];
	double     D[3][3]   = {0.0};
	double     D_scalar;
	double     Ke_g[numdof][numdof];
	GaussTria *gauss = NULL;

	/*Initialize Element matrix and return if necessary*/
	if(IsOnWater()) return NULL;
	ElementMatrix* Ke=new ElementMatrix(nodes,NUMVERTICES,this->parameters,MacAyealApproximationEnum);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	Input* thickness_input=inputs->GetInput(ThicknessEnum); ISSMASSERT(thickness_input);
	Input* vx_input=inputs->GetInput(VxEnum);               ISSMASSERT(vx_input);
	Input* vy_input=inputs->GetInput(VyEnum);               ISSMASSERT(vy_input);
	Input* vxold_input=inputs->GetInput(VxOldEnum);         ISSMASSERT(vxold_input);
	Input* vyold_input=inputs->GetInput(VyOldEnum);         ISSMASSERT(vyold_input);
	this->parameters->FindParam(&viscosity_overshoot,ViscosityOvershootEnum);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussTria(2);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss);
		GetBMacAyeal(&B[0][0], &xyz_list[0][0], gauss);
		GetBprimeMacAyeal(&Bprime[0][0], &xyz_list[0][0], gauss);

		this->GetStrainRate2d(&epsilon[0],&xyz_list[0][0],gauss,vx_input,vy_input);
		this->GetStrainRate2d(&oldepsilon[0],&xyz_list[0][0],gauss,vxold_input,vyold_input);
		matice->GetViscosity2d(&viscosity, &epsilon[0]);
		matice->GetViscosity2d(&oldviscosity, &oldepsilon[0]);
		thickness_input->GetParameterValue(&thickness, gauss);

		newviscosity=viscosity+viscosity_overshoot*(viscosity-oldviscosity);
		D_scalar=2*newviscosity*thickness*gauss->weight*Jdet;
		for (i=0;i<3;i++) D[i][i]=D_scalar;

		TripleMultiply(&B[0][0],3,numdof,1,
					&D[0][0],3,3,0,
					&Bprime[0][0],3,numdof,0,
					&Ke_g[0][0],0);

		for(i=0;i<numdof;i++) for(j=0;j<numdof;j++) Ke->values[i*numdof+j]+=Ke_g[i][j];
	}

	/*Clean up and return*/
	delete gauss;
	return Ke;
}
/*}}}*/
/*FUNCTION Tria::CreateKMatrixDiagnosticMacAyealFriction {{{1*/
ElementMatrix* Tria::CreateKMatrixDiagnosticMacAyealFriction(void){

	/*Constants*/
	const int  numdof=NDOF2*NUMVERTICES;

	/*Intermediaries*/
	int        i,j,ig;
	int        analysis_type,drag_type;
	double     MAXSLOPE  = .06; // 6 %
	double     MOUNTAINKEXPONENT = 10;
	double     slope_magnitude,alpha2;
	double     Jdet;
	double     L[2][numdof];
	double     DL[2][2]  = {{ 0,0 },{0,0}};
	double     Ke_g[numdof][numdof];
	double     DL_scalar;
	double     slope[2]  = {0.0,0.0};
	double     xyz_list[NUMVERTICES][3];
	Friction  *friction = NULL;
	GaussTria *gauss    = NULL;

	/*Initialize Element matrix and return if necessary*/
	if(IsOnWater() || IsOnShelf()) return NULL;
	ElementMatrix* Ke=new ElementMatrix(nodes,NUMVERTICES,this->parameters,MacAyealApproximationEnum);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	Input* surface_input=inputs->GetInput(SurfaceEnum); ISSMASSERT(surface_input);
	Input* vx_input=inputs->GetInput(VxEnum);           ISSMASSERT(vx_input);
	Input* vy_input=inputs->GetInput(VyEnum);           ISSMASSERT(vy_input);
	Input* vz_input=inputs->GetInput(VzEnum);           ISSMASSERT(vz_input);
	inputs->GetParameterValue(&drag_type,DragTypeEnum);
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	/*build friction object, used later on: */
	if (drag_type!=2) ISSMERROR(" non-viscous friction not supported yet!");
	friction=new Friction("2d",inputs,matpar,analysis_type);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussTria(2);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		// If we have a slope > 6% for this element,  it means  we are on a mountain. In this particular case, 
		//velocity should be = 0. To achieve this result, we set alpha2_list to a very high value: */
		surface_input->GetParameterDerivativeValue(&slope[0],&xyz_list[0][0],gauss);
		slope_magnitude=sqrt(pow(slope[0],2)+pow(slope[1],2));
		if(slope_magnitude>MAXSLOPE) alpha2=pow((double)10,MOUNTAINKEXPONENT);
		else friction->GetAlpha2(&alpha2, gauss,VxEnum,VyEnum,VzEnum);

		GetL(&L[0][0], &xyz_list[0][0], gauss,NDOF2);
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss);
		DL_scalar=alpha2*gauss->weight*Jdet;
		for (i=0;i<2;i++) DL[i][i]=DL_scalar;
		
		TripleMultiply( &L[0][0],2,numdof,1,
					&DL[0][0],2,2,0,
					&L[0][0],2,numdof,0,
					&Ke_g[0][0],0);

		for(i=0;i<numdof;i++) for(j=0;j<numdof;j++) Ke->values[i*numdof+j]+=Ke_g[i][j]; 
	}

	/*Clean up and return*/
	delete gauss;
	delete friction;
	return Ke;
}
/*}}}*/
/*FUNCTION Tria::CreateKMatrixCouplingMacAyealPattynFriction {{{1*/
ElementMatrix* Tria::CreateKMatrixCouplingMacAyealPattynFriction(void){

	/*Constants*/
	const int numdof        = NDOF2 *NUMVERTICES;
	const int numdoftotal   = NDOF4 *NUMVERTICES;
	
	/*Intermediaries */
	int       i,j,ig,analysis_type,drag_type;
	double    Jdet,slope_magnitude,alpha2;
	double    xyz_list[NUMVERTICES][3];
	double    slope[2]={0.0,0.0};
	double    MAXSLOPE=.06; // 6 %
	double    MOUNTAINKEXPONENT=10;
	double    L[2][numdof];
	double    DL[2][2]                  ={{ 0,0 },{0,0}}; //for basal drag
	double    DL_scalar;
	double    Ke_gg[numdof][numdof]     ={0.0};
	double    Ke_gg_gaussian[numdof][numdof]; //stiffness matrix contribution from drag
	Friction  *friction = NULL;
	GaussTria *gauss=NULL;

	/*Initialize Element matrix and return if necessary*/
	if(IsOnWater() || IsOnShelf()) return NULL;
	ElementMatrix* Ke1=new ElementMatrix(nodes,NUMVERTICES,this->parameters,MacAyealApproximationEnum);
	ElementMatrix* Ke2=new ElementMatrix(nodes,NUMVERTICES,this->parameters,PattynApproximationEnum);
	ElementMatrix* Ke=new ElementMatrix(Ke1,Ke2);
	delete Ke1; delete Ke2;

	/*retrieve inputs :*/
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);
	inputs->GetParameterValue(&drag_type,DragTypeEnum);
	Input* surface_input=inputs->GetInput(SurfaceEnum); ISSMASSERT(surface_input);
	Input* vx_input=inputs->GetInput(VxEnum);           ISSMASSERT(vx_input);
	Input* vy_input=inputs->GetInput(VyEnum);           ISSMASSERT(vy_input);
	Input* vz_input=inputs->GetInput(VzEnum);           ISSMASSERT(vz_input);

	/*build friction object, used later on: */
	if (drag_type!=2)ISSMERROR(" non-viscous friction not supported yet!");
	friction=new Friction("2d",inputs,matpar,analysis_type);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussTria(2);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		/*Friction: */
		friction->GetAlpha2(&alpha2, gauss,VxEnum,VyEnum,VzEnum);

		// If we have a slope > 6% for this element,  it means  we are on a mountain. In this particular case, 
		//velocity should be = 0. To achieve this result, we set alpha2_list to a very high value: */
		surface_input->GetParameterDerivativeValue(&slope[0],&xyz_list[0][0],gauss);
		slope_magnitude=sqrt(pow(slope[0],2)+pow(slope[1],2));

		if (slope_magnitude>MAXSLOPE){
			alpha2=pow((double)10,MOUNTAINKEXPONENT);
		}

		/* Get Jacobian determinant: */
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss);

		/*Get L matrix: */
		GetL(&L[0][0], &xyz_list[0][0], gauss,NDOF2);

		
		DL_scalar=alpha2*gauss->weight*Jdet;
		for (i=0;i<2;i++){
			DL[i][i]=DL_scalar;
		}
		
		/*  Do the triple producte tL*D*L: */
		TripleMultiply( &L[0][0],2,numdof,1,
					&DL[0][0],2,2,0,
					&L[0][0],2,numdof,0,
					&Ke_gg_gaussian[0][0],0);

		for(i=0;i<numdof;i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_gaussian[i][j];
	}

	for(i=0;i<numdof;i++) for(j=0;j<numdof;j++) Ke->values[i*numdoftotal+(numdof+j)]+=Ke_gg[i][j];
	for(i=0;i<numdof;i++) for(j=0;j<numdof;j++) Ke->values[(i+numdof)*numdoftotal+j]+=Ke_gg[i][j];

	/*Clean up and return*/
	delete gauss;
	delete friction;
	return Ke;
}	
/*}}}*/
/*FUNCTION Tria::CreateKMatrixDiagnosticPattynFriction {{{1*/
ElementMatrix* Tria::CreateKMatrixDiagnosticPattynFriction(void){

	/*Constants*/
	const int numdof   = NDOF2*NUMVERTICES;
	
	/*Intermediaries */
	int       i,j,ig;
	int       analysis_type,drag_type;
	double    xyz_list[NUMVERTICES][3];
	double    slope_magnitude,alpha2,Jdet;
	double    slope[2]={0.0,0.0};
	double    MAXSLOPE=.06; // 6 %
	double    MOUNTAINKEXPONENT=10;
	double    L[2][numdof];
	double    DL[2][2]={{ 0,0 },{0,0}}; //for basal drag
	double    DL_scalar;
	double    Ke_gg_gaussian[numdof][numdof]; //stiffness matrix contribution from drag
	Friction  *friction = NULL;
	GaussTria *gauss=NULL;

	/*Initialize Element matrix and return if necessary*/
	if(IsOnWater() || IsOnShelf()) return NULL;
	ElementMatrix* Ke=new ElementMatrix(nodes,NUMVERTICES,this->parameters,PattynApproximationEnum);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);
	inputs->GetParameterValue(&drag_type,DragTypeEnum);
	Input* surface_input=inputs->GetInput(SurfaceEnum); ISSMASSERT(surface_input);
	Input* vx_input=inputs->GetInput(VxEnum);           ISSMASSERT(vx_input);
	Input* vy_input=inputs->GetInput(VyEnum);           ISSMASSERT(vy_input);
	Input* vz_input=inputs->GetInput(VzEnum);           ISSMASSERT(vz_input);

	/*build friction object, used later on: */
	if (drag_type!=2)ISSMERROR(" non-viscous friction not supported yet!");
	friction=new Friction("2d",inputs,matpar,analysis_type);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussTria(2);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss);
		GetL(&L[0][0], &xyz_list[0][0], gauss,NDOF2);

		surface_input->GetParameterDerivativeValue(&slope[0],&xyz_list[0][0],gauss);
		friction->GetAlpha2(&alpha2, gauss,VxEnum,VyEnum,VzEnum); // TO UNCOMMENT
		slope_magnitude=sqrt(pow(slope[0],2)+pow(slope[1],2));

		// If we have a slope > 6% for this element,  it means  we are on a mountain. In this particular case, 
		//velocity should be = 0. To achieve this result, we set alpha2_list to a very high value: */
		if (slope_magnitude>MAXSLOPE){
			alpha2=pow((double)10,MOUNTAINKEXPONENT);
		}
		
		DL_scalar=alpha2*gauss->weight*Jdet;
		for (i=0;i<2;i++) DL[i][i]=DL_scalar;
		
		TripleMultiply( &L[0][0],2,numdof,1,
					&DL[0][0],2,2,0,
					&L[0][0],2,numdof,0,
					&Ke_gg_gaussian[0][0],0);

		for(i=0;i<numdof;i++) for(j=0;j<numdof;j++) Ke->values[i*numdof+j]+=Ke_gg_gaussian[i][j];
	}

	/*Clean up and return*/
	delete gauss;
	delete friction;
	return Ke;
}	
/*}}}*/
/*FUNCTION Tria::CreateKMatrixDiagnosticHutter{{{1*/
ElementMatrix* Tria::CreateKMatrixDiagnosticHutter(void){

	/*Intermediaries*/
	const int numdof=NUMVERTICES*NDOF2;
	int    i,connectivity;

	/*Initialize Element matrix and return if necessary*/
	if(IsOnWater()) return NULL;
	ElementMatrix* Ke=new ElementMatrix(nodes,NUMVERTICES,this->parameters,NoneApproximationEnum);

	/*Create Element matrix*/
	for(i=0;i<NUMVERTICES;i++){
		connectivity=nodes[i]->GetConnectivity();
		Ke->values[(2*i)*numdof  +(2*i)  ]=1/(double)connectivity;
		Ke->values[(2*i+1)*numdof+(2*i+1)]=1/(double)connectivity;
	}

	/*Clean up and return*/
	return Ke;
}
/*}}}*/
/*FUNCTION Tria::CreateKMatrixDiagnosticVertSurface {{{1*/
ElementMatrix* Tria::CreateKMatrixDiagnosticVertSurface(void){

	/*Constants*/
	const int numdof=NDOF1*NUMVERTICES;

	/*Intermediaries */
	int       i,j,ig;
	double    xyz_list[NUMVERTICES][3];
	double    surface_normal[3];
	double    Jdet,DL_scalar;
	double    L[3];
	GaussTria *gauss=NULL;
	double Ke_g[numdof][numdof]; //stiffness matrix evaluated at the gaussian point.

	/*Initialize Element matrix and return if necessary*/
	if(IsOnWater()) return NULL;
	ElementMatrix* Ke=new ElementMatrix(nodes,NUMVERTICES,this->parameters,NoneApproximationEnum);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	SurfaceNormal(&surface_normal[0],xyz_list);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussTria(2);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		GetJacobianDeterminant3d(&Jdet, &xyz_list[0][0],gauss);
		GetL(&L[0], &xyz_list[0][0], gauss,NDOF1);

		/**********************Do not forget the sign**********************************/
		DL_scalar=- gauss->weight*Jdet*surface_normal[2]; 
		/******************************************************************************/

		TripleMultiply( L,1,3,1,
					&DL_scalar,1,1,0,
					L,1,3,0,
					&Ke_g[0][0],0);

		for(i=0;i<numdof;i++) for(j=0;j<numdof;j++) Ke->values[i*numdof+j]+=Ke_g[i][j];
	}

	/*Clean up and return*/
	delete gauss;
	return Ke;
}
/*}}}*/
/*FUNCTION Tria::CreateKMatrixMelting {{{1*/
ElementMatrix* Tria::CreateKMatrixMelting(void){

	/*Constants*/
	const int  numdof=NUMVERTICES*NDOF1;

	/*Intermediaries */
	int        i,j,ig;
	double     heatcapacity,latentheat;
	double     Jdet,D_scalar;
	double     xyz_list[NUMVERTICES][3];
	double     L[3];
	double     tLD[3];
	double     Ke_gaussian[numdof][numdof]={0.0};
	GaussTria *gauss=NULL;

	/*Initialize Element matrix and return if necessary*/
	if(IsOnWater()) return NULL;
	ElementMatrix* Ke=new ElementMatrix(nodes,NUMVERTICES,this->parameters,NoneApproximationEnum);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	latentheat=matpar->GetLatentHeat();
	heatcapacity=matpar->GetHeatCapacity();

	/* Start looping on the number of gauss  (nodes on the bedrock) */
	gauss=new GaussTria(2);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		GetL(&L[0], &xyz_list[0][0], gauss,NDOF1);
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0], gauss);

		D_scalar=latentheat/heatcapacity*gauss->weight*Jdet;

		MatrixMultiply(&L[0],numdof,1,0,&D_scalar,1,1,0,&tLD[0],0);
		MatrixMultiply(&tLD[0],numdof,1,0,&L[0],1,numdof,0,&Ke_gaussian[0][0],0);

		for(i=0;i<numdof;i++) for(j=0;j<numdof;j++) Ke->values[i*numdof+j]+=Ke_gaussian[i][j];
	}

	/*Clean up and return*/
	delete gauss;
	return Ke;
}
/*}}}*/
/*FUNCTION Tria::CreateKMatrixPrognostic {{{1*/
ElementMatrix* Tria::CreateKMatrixPrognostic(void){

	switch(GetElementType()){
		case P1Enum:
			return CreateKMatrixPrognostic_CG();
		case P1DGEnum:
			return CreateKMatrixPrognostic_DG();
		default:
			ISSMERROR("Element type %s not supported yet",EnumToString(GetElementType()));
	}

}
/*}}}*/
/*FUNCTION Tria::CreateKMatrixPrognostic_CG {{{1*/
ElementMatrix* Tria::CreateKMatrixPrognostic_CG(void){

	/*Constants*/
	const int    numdof=NDOF1*NUMVERTICES;

	/*Intermediaries */
	bool       artdiff;
	int        i,j,ig,dim;
	double     Jdettria,DL_scalar,dt;
	double     vx,vy,dvxdx,dvydy;
	double     dvx[2],dvy[2];
	double     v_gauss[2]={0.0};
	double     xyz_list[NUMVERTICES][3];
	double     L[NUMVERTICES];
	double     B[2][NUMVERTICES];
	double     Bprime[2][NUMVERTICES];
	double     K[2][2]                        ={0.0};
	double     KDL[2][2]                      ={0.0};
	double     DL[2][2]                        ={0.0};
	double     DLprime[2][2]                   ={0.0};
	double     Ke_gg_gaussian[numdof][numdof]  ={0.0};
	double     Ke_gg_thickness1[numdof][numdof]={0.0};
	double     Ke_gg_thickness2[numdof][numdof]={0.0};
	GaussTria *gauss=NULL;

	/*Initialize Element matrix and return if necessary*/
	if(IsOnWater()) return NULL;
	ElementMatrix* Ke=new ElementMatrix(nodes,NUMVERTICES,this->parameters,NoneApproximationEnum);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	this->parameters->FindParam(&dt,DtEnum);
	this->parameters->FindParam(&dim,DimEnum);
	this->parameters->FindParam(&artdiff,ArtDiffEnum);
	Input* vxaverage_input=NULL;
	Input* vyaverage_input=NULL;
	if(dim==2){
		vxaverage_input=inputs->GetInput(VxEnum); ISSMASSERT(vxaverage_input);
		vyaverage_input=inputs->GetInput(VyEnum); ISSMASSERT(vyaverage_input);
	}
	else{
		vxaverage_input=inputs->GetInput(VxAverageEnum); ISSMASSERT(vxaverage_input);
		vyaverage_input=inputs->GetInput(VyAverageEnum); ISSMASSERT(vyaverage_input);
	}

	//Create Artificial diffusivity once for all if requested
	if(artdiff){
		gauss=new GaussTria();
		gauss->GaussCenter();
		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss);
		delete gauss;

		vxaverage_input->GetParameterAverage(&v_gauss[0]);
		vyaverage_input->GetParameterAverage(&v_gauss[1]);

		K[0][0]=pow(Jdettria,(double).5)/2.0*fabs(v_gauss[0]);
		K[1][1]=pow(Jdettria,(double).5)/2.0*fabs(v_gauss[1]);
	}

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussTria(2);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss);
		GetL(&L[0], &xyz_list[0][0], gauss,NDOF1);

		vxaverage_input->GetParameterValue(&vx,gauss);
		vyaverage_input->GetParameterValue(&vy,gauss);
		vxaverage_input->GetParameterDerivativeValue(&dvx[0],&xyz_list[0][0],gauss);
		vyaverage_input->GetParameterDerivativeValue(&dvy[0],&xyz_list[0][0],gauss);

		DL_scalar=gauss->weight*Jdettria;

		TripleMultiply( &L[0],1,numdof,1,
					&DL_scalar,1,1,0,
					&L[0],1,numdof,0,
					&Ke_gg_gaussian[0][0],0);

		GetBPrognostic(&B[0][0], &xyz_list[0][0], gauss);
		GetBprimePrognostic(&Bprime[0][0], &xyz_list[0][0], gauss);

		dvxdx=dvx[0];
		dvydy=dvy[1];
		DL_scalar=dt*gauss->weight*Jdettria;

		DL[0][0]=DL_scalar*dvxdx;
		DL[1][1]=DL_scalar*dvydy;
		DLprime[0][0]=DL_scalar*vx;
		DLprime[1][1]=DL_scalar*vy;

		TripleMultiply( &B[0][0],2,numdof,1,
					&DL[0][0],2,2,0,
					&B[0][0],2,numdof,0,
					&Ke_gg_thickness1[0][0],0);

		TripleMultiply( &B[0][0],2,numdof,1,
					&DLprime[0][0],2,2,0,
					&Bprime[0][0],2,numdof,0,
					&Ke_gg_thickness2[0][0],0);

		for(i=0;i<numdof;i++) for(j=0;j<numdof;j++) Ke->values[i*numdof+j]+=Ke_gg_gaussian[i][j];
		for(i=0;i<numdof;i++) for(j=0;j<numdof;j++) Ke->values[i*numdof+j]+=Ke_gg_thickness1[i][j];
		for(i=0;i<numdof;i++) for(j=0;j<numdof;j++) Ke->values[i*numdof+j]+=Ke_gg_thickness2[i][j];

		if(artdiff){
			KDL[0][0]=DL_scalar*K[0][0];
			KDL[1][1]=DL_scalar*K[1][1];

			TripleMultiply( &Bprime[0][0],2,numdof,1,
						&KDL[0][0],2,2,0,
						&Bprime[0][0],2,numdof,0,
						&Ke_gg_gaussian[0][0],0);

			for(i=0;i<numdof;i++) for(j=0;j<numdof;j++) Ke->values[i*numdof+j]+=Ke_gg_gaussian[i][j];
		}
	}

	/*Clean up and return*/
	delete gauss;
	return Ke;
}
/*}}}*/
/*FUNCTION Tria::CreateKMatrixPrognostic_DG {{{1*/
ElementMatrix* Tria::CreateKMatrixPrognostic_DG(void){

	/*Constants*/
	const int    numdof=NDOF1*NUMVERTICES;

	/*Intermediaries */
	int        i,j,ig,dim;
	double     xyz_list[NUMVERTICES][3];
	double     Jdettria,dt,vx,vy;
	double     L[NUMVERTICES];
	double     B[2][NUMVERTICES];
	double     Bprime[2][NUMVERTICES];
	double     DL[2][2]={0.0};
	double     DLprime[2][2]={0.0};
	double     DL_scalar;
	double     Ke_gg1[numdof][numdof]={0.0};
	double     Ke_gg2[numdof][numdof]={0.0};
	GaussTria  *gauss=NULL;

	/*Initialize Element matrix and return if necessary*/
	if(IsOnWater()) return NULL;
	ElementMatrix* Ke=new ElementMatrix(nodes,NUMVERTICES,this->parameters,NoneApproximationEnum);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	this->parameters->FindParam(&dt,DtEnum);
	this->parameters->FindParam(&dim,DimEnum);
	Input* vxaverage_input=NULL;
	Input* vyaverage_input=NULL;
	if(dim==2){
		vxaverage_input=inputs->GetInput(VxEnum); ISSMASSERT(vxaverage_input);
		vyaverage_input=inputs->GetInput(VyEnum); ISSMASSERT(vyaverage_input);
	}
	else{
		vxaverage_input=inputs->GetInput(VxAverageEnum); ISSMASSERT(vxaverage_input);
		vyaverage_input=inputs->GetInput(VyAverageEnum); ISSMASSERT(vyaverage_input);
	}

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussTria(2);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		vxaverage_input->GetParameterValue(&vx,gauss);
		vyaverage_input->GetParameterValue(&vy,gauss);

		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss);
		GetL(&L[0], &xyz_list[0][0], gauss,NDOF1);

		DL_scalar=gauss->weight*Jdettria;

		TripleMultiply( &L[0],1,numdof,1,
					&DL_scalar,1,1,0,
					&L[0],1,numdof,0,
					&Ke_gg1[0][0],0);

		/*WARNING: B and Bprime are inverted compared to usual prognostic!!!!*/
		GetBPrognostic(&Bprime[0][0], &xyz_list[0][0], gauss);
		GetBprimePrognostic(&B[0][0], &xyz_list[0][0], gauss);

		DL_scalar=-dt*gauss->weight*Jdettria;

		DLprime[0][0]=DL_scalar*vx;
		DLprime[1][1]=DL_scalar*vy;

		TripleMultiply( &B[0][0],2,numdof,1,
					&DLprime[0][0],2,2,0,
					&Bprime[0][0],2,numdof,0,
					&Ke_gg2[0][0],0);

		for(i=0;i<numdof;i++) for(j=0;j<numdof;j++) Ke->values[i*numdof+j]+=Ke_gg1[i][j];
		for(i=0;i<numdof;i++) for(j=0;j<numdof;j++) Ke->values[i*numdof+j]+=Ke_gg2[i][j];
	}

	/*Clean up and return*/
	delete gauss;
	return Ke;
}
/*}}}*/
/*FUNCTION Tria::CreateKMatrixSlope {{{1*/
ElementMatrix* Tria::CreateKMatrixSlope(void){

	/*constants: */
	const int    numdof=NDOF1*NUMVERTICES;

	/* Intermediaries */
	int        i,j,ig;
	double     DL_scalar,Jdet;
	double     xyz_list[NUMVERTICES][3];
	double     L[1][3];
	double     Ke_g[numdof][numdof];
	GaussTria *gauss = NULL;

	/*Initialize Element matrix and return if necessary*/
	if(IsOnWater()) return NULL;
	ElementMatrix* Ke=new ElementMatrix(nodes,NUMVERTICES,this->parameters,NoneApproximationEnum);

	GetVerticesCoordinates(&xyz_list[0][0],nodes,NUMVERTICES);

	/* Start looping on the number of gaussian points: */
	gauss=new GaussTria(2);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);
		
		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss);
		DL_scalar=gauss->weight*Jdet;

		GetL(&L[0][0], &xyz_list[0][0], gauss,NDOF1);

		TripleMultiply(&L[0][0],1,3,1,
					&DL_scalar,1,1,0,
					&L[0][0],1,3,0,
					&Ke_g[0][0],0);

		for(i=0;i<numdof;i++) for(j=0;j<numdof;j++) Ke->values[i*numdof+j]+=Ke_g[i][j];
	}

	/*Clean up and return*/
	delete gauss;
	return Ke;
}
/*}}}*/
/*FUNCTION Tria::CreateKMatrixThermal {{{1*/
ElementMatrix* Tria::CreateKMatrixThermal(void){

	/*Constants*/
	const int    numdof=NDOF1*NUMVERTICES;

	/*Intermediaries */
	int       i,j,ig;
	double    mixed_layer_capacity,thermal_exchange_velocity;
	double    rho_ice,rho_water,heatcapacity;
	double    Jdet,dt;
	double    xyz_list[NUMVERTICES][3];
	double    l1l2l3[NUMVERTICES];
	double    tl1l2l3D[3];
	double    D_scalar;
	double    Ke_gaussian[numdof][numdof]={0.0};
	GaussTria *gauss=NULL;

	/*Initialize Element matrix and return if necessary*/
	if(IsOnWater() || !IsOnShelf()) return NULL;
	ElementMatrix* Ke=new ElementMatrix(nodes,NUMVERTICES,this->parameters,NoneApproximationEnum);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	this->parameters->FindParam(&dt,DtEnum);
	mixed_layer_capacity=matpar->GetMixedLayerCapacity();
	thermal_exchange_velocity=matpar->GetThermalExchangeVelocity();
	rho_water=matpar->GetRhoWater();
	rho_ice=matpar->GetRhoIce();
	heatcapacity=matpar->GetHeatCapacity();

	/* Start looping on the number of gauss (nodes on the bedrock) */
	gauss=new GaussTria(2);
	for (ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);
		
		GetJacobianDeterminant3d(&Jdet, &xyz_list[0][0], gauss);
		GetNodalFunctions(&l1l2l3[0], gauss);
				
		D_scalar=gauss->weight*Jdet*rho_water*mixed_layer_capacity*thermal_exchange_velocity/(heatcapacity*rho_ice);
		if(dt) D_scalar=dt*D_scalar;

		MatrixMultiply(&l1l2l3[0],numdof,1,0,&D_scalar,1,1,0,&tl1l2l3D[0],0);
		MatrixMultiply(&tl1l2l3D[0],numdof,1,0,&l1l2l3[0],1,numdof,0,&Ke_gaussian[0][0],0);

		for(i=0;i<numdof;i++) for(j=0;j<numdof;j++) Ke->values[i*numdof+j]+=Ke_gaussian[i][j];
	}
	
	/*Clean up and return*/
	delete gauss;
	return Ke;
}
/*}}}*/
/*FUNCTION Tria::CreatePVectorBalancedthickness{{{1*/
ElementVector* Tria::CreatePVectorBalancedthickness(void){

	switch(GetElementType()){
		case P1Enum:
			return CreatePVectorBalancedthickness_CG();
			break;
		case P1DGEnum:
			return CreatePVectorBalancedthickness_DG();
		default:
			ISSMERROR("Element type %s not supported yet",EnumToString(GetElementType()));
	}
}
/*}}}*/
/*FUNCTION Tria::CreatePVectorBalancedthickness_CG{{{1*/
ElementVector* Tria::CreatePVectorBalancedthickness_CG(void){

	/*Constants*/
	const int    numdof=NDOF1*NUMVERTICES;
	
	/*Intermediaries */
	int        i,j,ig;
	double     xyz_list[NUMVERTICES][3];
	double     dhdt_g,melting_g,accumulation_g,Jdettria;
	double     L[NUMVERTICES];
	GaussTria* gauss=NULL;

	/*Initialize Element vector and return if necessary*/
	if(IsOnWater()) return NULL;
	ElementVector* pe=new ElementVector(nodes,NUMVERTICES,this->parameters);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	Input* accumulation_input=inputs->GetInput(AccumulationRateEnum); ISSMASSERT(accumulation_input);
	Input* melting_input=inputs->GetInput(MeltingRateEnum);           ISSMASSERT(melting_input);
	Input* dhdt_input=inputs->GetInput(DhDtEnum);                     ISSMASSERT(dhdt_input);
	
	/* Start  looping on the number of gaussian points: */
	gauss=new GaussTria(2);
	for(ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		accumulation_input->GetParameterValue(&accumulation_g,gauss);
		melting_input->GetParameterValue(&melting_g,gauss);
		dhdt_input->GetParameterValue(&dhdt_g,gauss);

		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss);
		GetL(&L[0], &xyz_list[0][0], gauss,NDOF1);

		for(i=0;i<numdof;i++) pe->values[i]+=Jdettria*gauss->weight*(accumulation_g-melting_g-dhdt_g)*L[i];
	}

	/*Clean up and return*/
	delete gauss;
	return pe;
}
/*}}}*/
/*FUNCTION Tria::CreatePVectorBalancedthickness_DG {{{1*/
ElementVector* Tria::CreatePVectorBalancedthickness_DG(void){

	/*Constants*/
	const int    numdof=NDOF1*NUMVERTICES;

	/*Intermediaries */
	int        i,j,ig;
	double     xyz_list[NUMVERTICES][3];
	double     melting_g,accumulation_g,dhdt_g,Jdettria;
	double     L[NUMVERTICES];
	GaussTria* gauss=NULL;

	/*Initialize Element vector and return if necessary*/
	if(IsOnWater()) return NULL;
	ElementVector* pe=new ElementVector(nodes,NUMVERTICES,this->parameters);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	Input* accumulation_input=inputs->GetInput(AccumulationRateEnum); ISSMASSERT(accumulation_input);
	Input* melting_input=inputs->GetInput(MeltingRateEnum);           ISSMASSERT(melting_input);
	Input* dhdt_input=inputs->GetInput(DhDtEnum);                     ISSMASSERT(dhdt_input);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussTria(2);
	for(ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		accumulation_input->GetParameterValue(&accumulation_g,gauss);
		melting_input->GetParameterValue(&melting_g,gauss);
		dhdt_input->GetParameterValue(&dhdt_g,gauss);

		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss);
		GetL(&L[0], &xyz_list[0][0], gauss,NDOF1);

		for(i=0;i<numdof;i++) pe->values[i]+=Jdettria*gauss->weight*(accumulation_g-melting_g-dhdt_g)*L[i];
	}

	/*Clean up and return*/
	delete gauss;
	return pe;
}
/*}}}*/
/*FUNCTION Tria::CreatePVectorBalancedvelocities {{{1*/
ElementVector* Tria::CreatePVectorBalancedvelocities(void){

	/*Constants*/
	const int    numdof=NDOF1*NUMVERTICES;

	/*Intermediaries */
	int        i,j,ig;
	double     xyz_list[NUMVERTICES][3];
	double     Jdettria,accumulation_g,melting_g;
	double     L[NUMVERTICES];
	GaussTria* gauss=NULL;

	/*Initialize Element vector and return if necessary*/
	if(IsOnWater()) return NULL;
	ElementVector* pe=new ElementVector(nodes,NUMVERTICES,this->parameters);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	Input* accumulation_input=inputs->GetInput(AccumulationRateEnum); ISSMASSERT(accumulation_input);
	Input* melting_input=inputs->GetInput(MeltingRateEnum);           ISSMASSERT(melting_input);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussTria(2);
	for(ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss);
		GetL(&L[0], &xyz_list[0][0], gauss,NDOF1);

		accumulation_input->GetParameterValue(&accumulation_g,gauss);
		melting_input->GetParameterValue(&melting_g,gauss);

		for(i=0;i<numdof;i++) pe->values[i]+=Jdettria*gauss->weight*(accumulation_g-melting_g)*L[i];
	}

	/*Clean up and return*/
	delete gauss;
	return pe;
}
/*}}}*/
/*FUNCTION Tria::CreatePVectorDiagnosticBaseVert {{{1*/
ElementVector* Tria::CreatePVectorDiagnosticBaseVert(void){

	/*Constants*/
	const int    numdof=NDOF1*NUMVERTICES;

	/*Intermediaries */
	int        i,j,ig;
	int        approximation;
	double     xyz_list[NUMVERTICES][3];
	double     Jdet;
	double     vx,vy,vz,dbdx,dbdy,meltingvalue;
	double     slope[2];
	double     L[NUMVERTICES];
	GaussTria* gauss=NULL;

	/*Initialize Element vector and return if necessary*/
	if(IsOnWater()) return NULL;
	ElementVector* pe=new ElementVector(nodes,NUMVERTICES,this->parameters);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	inputs->GetParameterValue(&approximation,ApproximationEnum);
	Input* bed_input=inputs->GetInput(BedEnum);             ISSMASSERT(bed_input);
	Input* melting_input=inputs->GetInput(MeltingRateEnum); ISSMASSERT(melting_input);
	Input* vx_input=inputs->GetInput(VxEnum);               ISSMASSERT(vx_input);
	Input* vy_input=inputs->GetInput(VyEnum);               ISSMASSERT(vy_input);
	Input* vzstokes_input=NULL;
	if(approximation==PattynStokesApproximationEnum){
		vzstokes_input=inputs->GetInput(VzStokesEnum);       ISSMASSERT(vzstokes_input);
	}

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussTria(2);
	for(ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		melting_input->GetParameterValue(&meltingvalue, gauss);
		bed_input->GetParameterDerivativeValue(&slope[0],&xyz_list[0][0],gauss);
		vx_input->GetParameterValue(&vx, gauss);
		vy_input->GetParameterValue(&vy, gauss);
		if(approximation==PattynStokesApproximationEnum){
			vzstokes_input->GetParameterValue(&vz, gauss);
		}
		else vz=0;

		dbdx=slope[0];
		dbdy=slope[1];

		GetJacobianDeterminant3d(&Jdet, &xyz_list[0][0],gauss);
		GetL(&L[0], &xyz_list[0][0], gauss,NDOF1);

		for(i=0;i<numdof;i++) pe->values[i]+=-Jdet*gauss->weight*(vx*dbdx+vy*dbdy-vz-meltingvalue)*L[i];
	}

	/*Clean up and return*/
	delete gauss;
	return pe;
}
/*}}}*/
/*FUNCTION Tria::CreatePVectorDiagnosticMacAyeal {{{1*/
ElementVector* Tria::CreatePVectorDiagnosticMacAyeal(){

	/*Constants*/
	const int    numdof=NDOF2*NUMVERTICES;

	/*Intermediaries */
	int            i,j,ig,drag_type;
	double         plastic_stress,driving_stress_baseline,thickness;
	double         Jdet;
	double         xyz_list[NUMVERTICES][3];
	double         slope[2];
	double         l1l2l3[3];
	double         pe_g_gaussian[numdof];
	GaussTria*     gauss=NULL;

	/*Initialize Element vector and return if necessary*/
	if(IsOnWater()) return NULL;
	ElementVector* pe=new ElementVector(nodes,NUMVERTICES,this->parameters,MacAyealApproximationEnum);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	inputs->GetParameterValue(&drag_type,DragTypeEnum);
	Input* thickness_input=inputs->GetInput(ThicknessEnum); ISSMASSERT(thickness_input); 
	Input* surface_input=inputs->GetInput(SurfaceEnum);     ISSMASSERT(surface_input);
	Input* drag_input=inputs->GetInput(DragCoefficientEnum);ISSMASSERT(drag_input);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussTria(2);
	for(ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss);
		GetNodalFunctions(l1l2l3, gauss);

		thickness_input->GetParameterValue(&thickness,gauss);
		surface_input->GetParameterDerivativeValue(&slope[0],&xyz_list[0][0],gauss);
		
		/*In case we have plastic basal drag, compute plastic stress at gaussian point from k1, k2 and k3 fields in the 
		 * element itself: */
		if(drag_type==1) drag_input->GetParameterValue(&plastic_stress,gauss);

		driving_stress_baseline=matpar->GetRhoIce()*matpar->GetG()*thickness;

		/*Build pe_g_gaussian vector: */
		if(drag_type==1){
			for (i=0;i<NUMVERTICES;i++){
				for (j=0;j<NDOF2;j++){
					pe->values[i*NDOF2+j]+=(-driving_stress_baseline*slope[j]-plastic_stress)*Jdet*gauss->weight*l1l2l3[i]; 
				}
			}
		}
		else {
			for (i=0;i<NUMVERTICES;i++){
				for (j=0;j<NDOF2;j++){
					pe->values[i*NDOF2+j]+=-driving_stress_baseline*slope[j]*Jdet*gauss->weight*l1l2l3[i];
				}
			}
		}
	}

	/*Clean up and return*/
	delete gauss;
	return pe;
}
/*}}}*/
/*FUNCTION Tria::CreatePVectorAdjointBalancedthickness{{{1*/
ElementVector* Tria::CreatePVectorAdjointBalancedthickness(void){

	/*Constants*/
	const int    numdof=1*NUMVERTICES;

	/*Intermediaries */
	int         i,ig;
	double      Jdet;
	double      thickness,thicknessobs,weight;
	double      xyz_list[NUMVERTICES][3];
	double      l1l2l3[3];
	GaussTria*  gauss=NULL;

	/*Initialize Element vector and return if necessary*/
	if(IsOnWater()) return NULL;
	ElementVector* pe=new ElementVector(nodes,NUMVERTICES,this->parameters);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	Input* thickness_input   =inputs->GetInput(ThicknessEnum);   ISSMASSERT(thickness_input);
	Input* thicknessobs_input=inputs->GetInput(ThicknessObsEnum);ISSMASSERT(thicknessobs_input);
	Input* weights_input     =inputs->GetInput(WeightsEnum);     ISSMASSERT(weights_input);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussTria(2);
	for(ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss);
		GetNodalFunctions(l1l2l3, gauss);

		thickness_input->GetParameterValue(&thickness, gauss);
		thicknessobs_input->GetParameterValue(&thicknessobs, gauss);
		weights_input->GetParameterValue(&weight, gauss);

		for(i=0;i<numdof;i++) pe->values[i]+=(thicknessobs-thickness)*weight*Jdet*gauss->weight*l1l2l3[i];
	}

	/*Clean up and return*/
	delete gauss;
	return pe;
}
/*}}}*/
/*FUNCTION Tria::CreatePVectorAdjointHoriz{{{1*/
ElementVector* Tria::CreatePVectorAdjointHoriz(void){

	/*Constants*/
	const int    numdof=NDOF2*NUMVERTICES;

	/*Intermediaries */
	int        i,ig,response;
	double     Jdet;
	double     obs_velocity_mag,velocity_mag;
	double     dux,duy,meanvel,epsvel;
	double     scalex=0,scaley=0,scale=0,S=0;
	double     xyz_list[NUMVERTICES][3];
	double     vx_list[NUMVERTICES];
	double     vy_list[NUMVERTICES];
	double     obs_vx_list[NUMVERTICES];
	double     obs_vy_list[NUMVERTICES];
	double     dux_list[NUMVERTICES];
	double     duy_list[NUMVERTICES];
	double     weights_list[NUMVERTICES];
	double     l1l2l3[3];
	GaussTria* gauss=NULL;

	/*Initialize Element vector and return if necessary*/
	if(IsOnWater()) return NULL;
	ElementVector* pe=new ElementVector(nodes,NUMVERTICES,this->parameters);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	this->parameters->FindParam(&meanvel,MeanVelEnum);
	this->parameters->FindParam(&epsvel,EpsVelEnum);
	GetParameterListOnVertices(&obs_vx_list[0],VxObsEnum);
	GetParameterListOnVertices(&obs_vy_list[0],VyObsEnum);
	GetParameterListOnVertices(&vx_list[0],VxEnum);
	GetParameterListOnVertices(&vy_list[0],VyEnum);
	GetParameterListOnVertices(&weights_list[0],WeightsEnum);
	inputs->GetParameterValue(&response,CmResponseEnum);
	if(response==SurfaceAverageVelMisfitEnum){
		inputs->GetParameterValue(&S,SurfaceAreaEnum);
	}

	/*Get Du at the 3 nodes (integration of the linearized function)
	 * Here we integrate linearized functions:
	 *               
	 * J(E) = int_E   sum_{i=1}^3  J_i Phi_i
	 *
	 *       d J                  dJ_i
	 * DU= - --- = sum_{i=1}^3  - ---  Phi_i = sum_{i=1}^3 DU_i Phi_i
	 *       d u                  du_i
	 *
	 * where J_i are the misfits at the 3 nodes of the triangle
	 *       Phi_i is the nodal function (P1) with respect to 
	 *       the vertex i
	 */
	if(response==SurfaceAbsVelMisfitEnum){
		/*We are using an absolute misfit:
		 *
		 *      1  [           2              2 ]
		 * J = --- | (u - u   )  +  (v - v   )  |
		 *      2  [       obs            obs   ]
		 *
		 *        dJ
		 * DU = - -- = (u   - u )
		 *        du     obs
		 */
		for (i=0;i<NUMVERTICES;i++){
			dux_list[i]=obs_vx_list[i]-vx_list[i];
			duy_list[i]=obs_vy_list[i]-vy_list[i];
		}
	}
	else if(response==SurfaceRelVelMisfitEnum){
		/*We are using a relative misfit: 
		 *                        
		 *      1  [     \bar{v}^2             2   \bar{v}^2              2 ]
		 * J = --- | -------------  (u - u   ) + -------------  (v - v   )  |
		 *      2  [  (u   + eps)^2       obs    (v   + eps)^2       obs    ]
		 *              obs                        obs                      
		 *
		 *        dJ     \bar{v}^2
		 * DU = - -- = ------------- (u   - u )
		 *        du   (u   + eps)^2    obs
		 *               obs
		 */
		for (i=0;i<NUMVERTICES;i++){
			scalex=pow(meanvel/(obs_vx_list[i]+epsvel),2);
			scaley=pow(meanvel/(obs_vy_list[i]+epsvel),2);
			if(obs_vx_list[i]==0)scalex=0;
			if(obs_vy_list[i]==0)scaley=0;
			dux_list[i]=scalex*(obs_vx_list[i]-vx_list[i]);
			duy_list[i]=scaley*(obs_vy_list[i]-vy_list[i]);
		}
	}
	else if(response==SurfaceLogVelMisfitEnum){
		/*We are using a logarithmic misfit:
		 *                        
		 *                 [        vel + eps     ] 2
		 * J = 4 \bar{v}^2 | log ( -----------  ) |  
		 *                 [       vel   + eps    ]
		 *                            obs
		 *
		 *        dJ                 2 * log(...)
		 * DU = - -- = - 4 \bar{v}^2 -------------  u
		 *        du                 vel^2 + eps
		 *            
		 */
		for (i=0;i<NUMVERTICES;i++){
			velocity_mag=sqrt(pow(vx_list[i],2)+pow(vy_list[i],2))+epsvel; //epsvel to avoid velocity being nil.
			obs_velocity_mag=sqrt(pow(obs_vx_list[i],2)+pow(obs_vy_list[i],2))+epsvel; //epsvel to avoid observed velocity being nil.
			scale=-8*pow(meanvel,2)/pow(velocity_mag,2)*log(velocity_mag/obs_velocity_mag);
			dux_list[i]=scale*vx_list[i];
			duy_list[i]=scale*vy_list[i];
		}
	}
	else if(response==SurfaceAverageVelMisfitEnum){
		/*We are using an spacially average absolute misfit:
		 *
		 *      1                    2              2
		 * J = ---  sqrt(  (u - u   )  +  (v - v   )  )
		 *      S                obs            obs
		 *
		 *        dJ      1       1 
		 * DU = - -- = - --- ----------- * 2 (u - u   )
		 *        du      S  2 sqrt(...)           obs
		 */
		for (i=0;i<NUMVERTICES;i++){
			scale=1.0/(S*sqrt(pow(vx_list[i]-obs_vx_list[i],2)+pow(vy_list[i]-obs_vx_list[i],2))+epsvel);
			dux_list[i]=scale*(obs_vx_list[i]-vx_list[i]);
			duy_list[i]=scale*(obs_vy_list[i]-vy_list[i]);
		}
	}
	else if(response==SurfaceLogVxVyMisfitEnum){
		/*We are using an logarithmic 2 misfit:
		 *
		 *      1            [        |u| + eps     2          |v| + eps     2  ]
		 * J = --- \bar{v}^2 | log ( -----------  )   +  log ( -----------  )   |  
		 *      2            [       |u    |+ eps              |v    |+ eps     ]
		 *                              obs                       obs
		 *        dJ                              1      u                             1
		 * DU = - -- = - \bar{v}^2 log(u...) --------- ----  ~ - \bar{v}^2 log(u...) ------
		 *        du                         |u| + eps  |u|                           u + eps
		 */
		for (i=0;i<NUMVERTICES;i++){
			dux_list[i] = - pow(meanvel,(double)2)*(
						log((fabs(vx_list[i])+epsvel)/(fabs(obs_vx_list[i])+epsvel)) * 1/(vx_list[i]+epsvel));
			duy_list[i] = - pow(meanvel,(double)2)*(
						log((fabs(vy_list[i])+epsvel)/(fabs(obs_vy_list[i])+epsvel)) * 1/(vy_list[i]+epsvel));
		}
	}
	else{
		/*Not supported yet! : */
		ISSMERROR("response %s not supported yet",EnumToString(response));
	}

	/*Apply weights to DU*/
	for (i=0;i<NUMVERTICES;i++){
		dux_list[i]=weights_list[i]*dux_list[i];
		duy_list[i]=weights_list[i]*duy_list[i];
	}

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussTria(2);
	for(ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss);
		GetNodalFunctions(l1l2l3, gauss);

		TriaRef::GetParameterValue(&dux, &dux_list[0],gauss);
		TriaRef::GetParameterValue(&duy, &duy_list[0],gauss);

		for (i=0;i<NUMVERTICES;i++){
			pe->values[i*NDOF2+0]+=dux*Jdet*gauss->weight*l1l2l3[i]; 
			pe->values[i*NDOF2+1]+=duy*Jdet*gauss->weight*l1l2l3[i]; 
		}
	}

	/*Clean up and return*/
	delete gauss;
	return pe;
}
/*}}}*/
/*FUNCTION Tria::CreatePVectorAdjointStokes{{{1*/
ElementVector* Tria::CreatePVectorAdjointStokes(void){

	/*Intermediaries */
	int        i,ig;
	int        fit=-1;
	int        response;
	double     Jdet;
	double     obs_velocity_mag,velocity_mag;
	double     dux,duy,meanvel,epsvel;
	double     scalex=0,scaley=0,scale=0,S=0;
	double     xyz_list[NUMVERTICES][3];
	double     vx_list[NUMVERTICES];
	double     vy_list[NUMVERTICES];
	double     obs_vx_list[NUMVERTICES];
	double     obs_vy_list[NUMVERTICES];
	double     dux_list[NUMVERTICES];
	double     duy_list[NUMVERTICES];
	double     weights_list[NUMVERTICES];
	double     l1l2l3[3];
	GaussTria* gauss=NULL;

	/*Initialize Element vector and return if necessary*/
	if(IsOnWater()) return NULL;
	ElementVector* pe=new ElementVector(nodes,NUMVERTICES,this->parameters,StokesApproximationEnum);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	this->parameters->FindParam(&meanvel,MeanVelEnum);
	this->parameters->FindParam(&epsvel,EpsVelEnum);
	GetParameterListOnVertices(&obs_vx_list[0],VxObsEnum);
	GetParameterListOnVertices(&obs_vy_list[0],VyObsEnum);
	GetParameterListOnVertices(&vx_list[0],VxEnum);
	GetParameterListOnVertices(&vy_list[0],VyEnum);
	GetParameterListOnVertices(&weights_list[0],WeightsEnum);
	inputs->GetParameterValue(&response,CmResponseEnum);
	if(response==SurfaceAverageVelMisfitEnum){
		inputs->GetParameterValue(&S,SurfaceAreaEnum);
	}

	/*Get Du at the 3 nodes (integration of the linearized function)
	 * Here we integrate linearized functions:
	 *               
	 * J(E) = int_E   sum_{i=1}^3  J_i Phi_i
	 *
	 *       d J                  dJ_i
	 * DU= - --- = sum_{i=1}^3  - ---  Phi_i = sum_{i=1}^3 DU_i Phi_i
	 *       d u                  du_i
	 *
	 * where J_i are the misfits at the 3 nodes of the triangle
	 *       Phi_i is the nodal function (P1) with respect to 
	 *       the vertex i
	 */
	if(response==SurfaceAbsVelMisfitEnum){
		/*We are using an absolute misfit:
		 *
		 *      1  [           2              2 ]
		 * J = --- | (u - u   )  +  (v - v   )  |
		 *      2  [       obs            obs   ]
		 *
		 *        dJ             2
		 * DU = - -- = (u   - u )
		 *        du     obs
		 */
		for (i=0;i<NUMVERTICES;i++){
			dux_list[i]=obs_vx_list[i]-vx_list[i];
			duy_list[i]=obs_vy_list[i]-vy_list[i];
		}
	}
	else if(response==SurfaceRelVelMisfitEnum){
		/*We are using a relative misfit: 
		 *                        
		 *      1  [     \bar{v}^2             2   \bar{v}^2              2 ]
		 * J = --- | -------------  (u - u   ) + -------------  (v - v   )  |
		 *      2  [  (u   + eps)^2       obs    (v   + eps)^2       obs    ]
		 *              obs                        obs                      
		 *
		 *        dJ     \bar{v}^2
		 * DU = - -- = ------------- (u   - u )
		 *        du   (u   + eps)^2    obs
		 *               obs
		 */
		for (i=0;i<NUMVERTICES;i++){
			scalex=pow(meanvel/(obs_vx_list[i]+epsvel),2);
			scaley=pow(meanvel/(obs_vy_list[i]+epsvel),2);
			if(obs_vx_list[i]==0)scalex=0;
			if(obs_vy_list[i]==0)scaley=0;
			dux_list[i]=scalex*(obs_vx_list[i]-vx_list[i]);
			duy_list[i]=scaley*(obs_vy_list[i]-vy_list[i]);
		}
	}
	else if(response==SurfaceLogVelMisfitEnum){
		/*We are using a logarithmic misfit:
		 *                        
		 *                 [        vel + eps     ] 2
		 * J = 4 \bar{v}^2 | log ( -----------  ) |  
		 *                 [       vel   + eps    ]
		 *                            obs
		 *
		 *        dJ                 2 * log(...)
		 * DU = - -- = - 4 \bar{v}^2 -------------  u
		 *        du                 vel^2 + eps
		 *            
		 */
		for (i=0;i<NUMVERTICES;i++){
			velocity_mag=sqrt(pow(vx_list[i],2)+pow(vy_list[i],2))+epsvel; //epsvel to avoid velocity being nil.
			obs_velocity_mag=sqrt(pow(obs_vx_list[i],2)+pow(obs_vy_list[i],2))+epsvel; //epsvel to avoid observed velocity being nil.
			scale=-8*pow(meanvel,2)/pow(velocity_mag,2)*log(velocity_mag/obs_velocity_mag);
			dux_list[i]=scale*vx_list[i];
			duy_list[i]=scale*vy_list[i];
		}
	}
	else if(response==SurfaceAverageVelMisfitEnum){
		/*We are using an spacially average absolute misfit:
		 *
		 *      1                    2              2
		 * J = ---  sqrt(  (u - u   )  +  (v - v   )  )
		 *      S                obs            obs
		 *
		 *        dJ      1       1 
		 * DU = - -- = - --- ----------- * 2 (u - u   )
		 *        du      S  2 sqrt(...)           obs
		 */
		for (i=0;i<NUMVERTICES;i++){
			scale=1.0/(S*sqrt(pow(vx_list[i]-obs_vx_list[i],2)+pow(vy_list[i]-obs_vx_list[i],2))+epsvel);
			dux_list[i]=scale*(obs_vx_list[i]-vx_list[i]);
			duy_list[i]=scale*(obs_vy_list[i]-vy_list[i]);
		}
	}
	else if(response==SurfaceLogVxVyMisfitEnum){
		/*We are using an logarithmic 2 misfit:
		 *
		 *      1            [        |u| + eps     2          |v| + eps     2  ]
		 * J = --- \bar{v}^2 | log ( -----------  )   +  log ( -----------  )   |  
		 *      2            [       |u    |+ eps              |v    |+ eps     ]
		 *                              obs                       obs
		 *        dJ                              1      u                             1
		 * DU = - -- = - \bar{v}^2 log(u...) --------- ----  ~ - \bar{v}^2 log(u...) ------
		 *        du                         |u| + eps  |u|                           u + eps
		 */
		for (i=0;i<NUMVERTICES;i++){
			dux_list[i] = - pow(meanvel,(double)2)*(
						log((fabs(vx_list[i])+epsvel)/(fabs(obs_vx_list[i])+epsvel)) * 1/(vx_list[i]+epsvel));
			duy_list[i] = - pow(meanvel,(double)2)*(
						log((fabs(vy_list[i])+epsvel)/(fabs(obs_vy_list[i])+epsvel)) * 1/(vy_list[i]+epsvel));
		}
	}
	else{
		/*Not supported yet! : */
		ISSMERROR("response %s not supported yet",EnumToString(response));
	}

	/*Apply weights to DU*/
	for (i=0;i<NUMVERTICES;i++){
		dux_list[i]=weights_list[i]*dux_list[i];
		duy_list[i]=weights_list[i]*duy_list[i];
	}

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussTria(2);
	for(ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss);
		GetNodalFunctions(l1l2l3, gauss);

		TriaRef::GetParameterValue(&dux, &dux_list[0],gauss);
		TriaRef::GetParameterValue(&duy, &duy_list[0],gauss);

		for (i=0;i<NUMVERTICES;i++){
			pe->values[i*NDOF4+0]+=dux*Jdet*gauss->weight*l1l2l3[i];
			pe->values[i*NDOF4+1]+=duy*Jdet*gauss->weight*l1l2l3[i]; 
		}
	}

	/*Clean up and return*/
	delete gauss;
	return pe;
}
/*}}}*/
/*FUNCTION Tria::CreatePVectorDiagnosticHutter{{{1*/
ElementVector* Tria::CreatePVectorDiagnosticHutter(void){

	/*Intermediaries */
	int        i,connectivity;
	double     constant_part,ub,vb;
	double     rho_ice,gravity,n,B;
	double     slope2,thickness;
	double     slope[2];
	GaussTria* gauss=NULL;

	/*Initialize Element vector and return if necessary*/
	if(IsOnWater()) return NULL;
	ElementVector* pe=new ElementVector(nodes,NUMVERTICES,this->parameters);

	/*Retrieve all inputs and parameters*/
	rho_ice=matpar->GetRhoIce();
	gravity=matpar->GetG();
	n=matice->GetN();
	B=matice->GetBbar();
	Input* slopex_input=inputs->GetInput(SurfaceSlopeXEnum); ISSMASSERT(slopex_input);
	Input* slopey_input=inputs->GetInput(SurfaceSlopeYEnum); ISSMASSERT(slopey_input);
	Input* thickness_input=inputs->GetInput(ThicknessEnum);  ISSMASSERT(thickness_input);

	/*Spawn 3 sing elements: */
	gauss=new GaussTria();
	for(i=0;i<NUMVERTICES;i++){

		gauss->GaussVertex(i);

		connectivity=nodes[i]->GetConnectivity();

		thickness_input->GetParameterValue(&thickness,gauss);
		slopex_input->GetParameterValue(&slope[0],gauss);
		slopey_input->GetParameterValue(&slope[1],gauss);
		slope2=pow(slope[0],2)+pow(slope[1],2);

		constant_part=-2*pow(rho_ice*gravity,n)*pow(slope2,((n-1)/2));

		ub=-1.58*pow((double)10.0,(double)-10.0)*rho_ice*gravity*thickness*slope[0];
		vb=-1.58*pow((double)10.0,(double)-10.0)*rho_ice*gravity*thickness*slope[1];

		pe->values[2*i]  =(ub-2.0*pow(rho_ice*gravity,n)*pow(slope2,((n-1)/2.0))*pow(thickness,n)/(pow(B,n)*(n+1))*slope[0])/(double)connectivity;
		pe->values[2*i+1]=(vb-2.0*pow(rho_ice*gravity,n)*pow(slope2,((n-1)/2.0))*pow(thickness,n)/(pow(B,n)*(n+1))*slope[1])/(double)connectivity;
	}

	/*Clean up and return*/
	delete gauss;
	return pe;
}
/*}}}*/
/*FUNCTION Tria::CreatePVectorPrognostic{{{1*/
ElementVector* Tria::CreatePVectorPrognostic(void){

	switch(GetElementType()){
		case P1Enum:
			return CreatePVectorPrognostic_CG();
		case P1DGEnum:
			return CreatePVectorPrognostic_DG();
		default:
			ISSMERROR("Element type %s not supported yet",EnumToString(GetElementType()));
	}
}
/*}}}*/
/*FUNCTION Tria::CreatePVectorPrognostic_CG {{{1*/
ElementVector* Tria::CreatePVectorPrognostic_CG(void){

	/*Constants*/
	const int    numdof=NDOF1*NUMVERTICES;

	/*Intermediaries */
	int        i,j,ig;
	double     Jdettria,dt;
	double     accumulation_g,melting_g,thickness_g;
	double     xyz_list[NUMVERTICES][3];
	double     L[NUMVERTICES];
	GaussTria* gauss=NULL;

	/*Initialize Element vector and return if necessary*/
	if(IsOnWater()) return NULL;
	ElementVector* pe=new ElementVector(nodes,NUMVERTICES,this->parameters);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	this->parameters->FindParam(&dt,DtEnum);
	Input* accumulation_input=inputs->GetInput(AccumulationRateEnum); ISSMASSERT(accumulation_input);
	Input* melting_input=inputs->GetInput(MeltingRateEnum);           ISSMASSERT(melting_input);
	Input* thickness_input=inputs->GetInput(ThicknessEnum);           ISSMASSERT(thickness_input);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussTria(2);
	for(ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss);
		GetL(&L[0], &xyz_list[0][0], gauss,NDOF1);

		accumulation_input->GetParameterValue(&accumulation_g,gauss);
		melting_input->GetParameterValue(&melting_g,gauss);
		thickness_input->GetParameterValue(&thickness_g,gauss);

		for(i=0;i<numdof;i++) pe->values[i]+=Jdettria*gauss->weight*(thickness_g+dt*(accumulation_g-melting_g))*L[i];
	}

	/*Clean up and return*/
	delete gauss;
	return pe;
}
/*}}}*/
/*FUNCTION Tria::CreatePVectorPrognostic_DG {{{1*/
ElementVector* Tria::CreatePVectorPrognostic_DG(void){

	/*Constants*/
	const int    numdof=NDOF1*NUMVERTICES;

	/*Intermediaries */
	int        i,j,ig;
	double     Jdettria,dt;
	double     accumulation_g,melting_g,thickness_g;
	double     xyz_list[NUMVERTICES][3];
	double     L[NUMVERTICES];
	GaussTria* gauss=NULL;

	/*Initialize Element vector and return if necessary*/
	if(IsOnWater()) return NULL;
	ElementVector* pe=new ElementVector(nodes,NUMVERTICES,this->parameters);

	/*Retrieve all inputs and parameters*/
	this->parameters->FindParam(&dt,DtEnum);
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	Input* accumulation_input=inputs->GetInput(AccumulationRateEnum); ISSMASSERT(accumulation_input);
	Input* melting_input=inputs->GetInput(MeltingRateEnum);           ISSMASSERT(melting_input);
	Input* thickness_input=inputs->GetInput(ThicknessEnum);           ISSMASSERT(thickness_input);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussTria(2);
	for(ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		GetJacobianDeterminant2d(&Jdettria, &xyz_list[0][0],gauss);
		GetL(&L[0], &xyz_list[0][0], gauss,NDOF1);

		accumulation_input->GetParameterValue(&accumulation_g,gauss);
		melting_input->GetParameterValue(&melting_g,gauss);
		thickness_input->GetParameterValue(&thickness_g,gauss);

		for(i=0;i<numdof;i++) pe->values[i]+=Jdettria*gauss->weight*(thickness_g+dt*(accumulation_g-melting_g))*L[i];
	}

	/*Clean up and return*/
	delete gauss;
	return pe;
}
/*}}}*/
/*FUNCTION Tria::CreatePVectorSlope {{{1*/
ElementVector* Tria::CreatePVectorSlope(void){

	/*Constants*/
	const int    numdof=NDOF1*NUMVERTICES;
	
	/*Intermediaries */
	int        i,j,ig;
	int        analysis_type;
	double     Jdet;
	double     xyz_list[NUMVERTICES][3];
	double     slope[2];
	double     l1l2l3[3];
	GaussTria* gauss=NULL;

	/*Initialize Element vector and return if necessary*/
	if(IsOnWater()) return NULL;
	ElementVector* pe=new ElementVector(nodes,NUMVERTICES,this->parameters);

	/*Retrieve all inputs and parameters*/
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	Input* slope_input=NULL;
	if ( (analysis_type==SurfaceSlopeXAnalysisEnum) || (analysis_type==SurfaceSlopeYAnalysisEnum)){
		slope_input=inputs->GetInput(SurfaceEnum); ISSMASSERT(slope_input);
	}
	if ( (analysis_type==BedSlopeXAnalysisEnum) || (analysis_type==BedSlopeYAnalysisEnum)){
		slope_input=inputs->GetInput(BedEnum);     ISSMASSERT(slope_input);
	}
		
	/* Start  looping on the number of gaussian points: */
	gauss=new GaussTria(2);
	for(ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0],gauss);
		GetNodalFunctions(l1l2l3, gauss);

		slope_input->GetParameterDerivativeValue(&slope[0],&xyz_list[0][0],gauss);

		if ( (analysis_type==SurfaceSlopeXAnalysisEnum) || (analysis_type==BedSlopeXAnalysisEnum)){
			for(i=0;i<numdof;i++) pe->values[i]+=Jdet*gauss->weight*slope[0]*l1l2l3[i];
		}
		if ( (analysis_type==SurfaceSlopeYAnalysisEnum) || (analysis_type==BedSlopeYAnalysisEnum)){
			for(i=0;i<numdof;i++) pe->values[i]+=Jdet*gauss->weight*slope[1]*l1l2l3[i];
		}
	}

	/*Clean up and return*/
	delete gauss;
	return pe;
}
/*}}}*/
/*FUNCTION Tria::CreatePVectorThermalShelf {{{1*/
ElementVector* Tria::CreatePVectorThermalShelf(void){
	
	/*Constants*/
	const int  numdof=NUMVERTICES*NDOF1;

	/*Intermediaries */
	int        i,ig;
	double     Jdet;
	double     mixed_layer_capacity,thermal_exchange_velocity;
	double     rho_ice,rho_water,pressure,dt,scalar_ocean;
	double     meltingpoint,beta,heatcapacity,t_pmp;
	double     xyz_list[NUMVERTICES][3];
	double     l1l2l3[NUMVERTICES];
	GaussTria* gauss=NULL;

	/*Initialize Element vector and return if necessary*/
	if(IsOnWater()) return NULL;
	ElementVector* pe=new ElementVector(nodes,NUMVERTICES,this->parameters);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	mixed_layer_capacity=matpar->GetMixedLayerCapacity();
	thermal_exchange_velocity=matpar->GetThermalExchangeVelocity();
	rho_water=matpar->GetRhoWater();
	rho_ice=matpar->GetRhoIce();
	heatcapacity=matpar->GetHeatCapacity();
	beta=matpar->GetBeta();
	meltingpoint=matpar->GetMeltingPoint();
	this->parameters->FindParam(&dt,DtEnum);
	Input* pressure_input=inputs->GetInput(PressureEnum); ISSMASSERT(pressure_input);

	/* Start looping on the number of gauss 2d (nodes on the bedrock) */
	gauss=new GaussTria(2);
	for(ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		GetJacobianDeterminant3d(&Jdet, &xyz_list[0][0], gauss);
		GetNodalFunctions(&l1l2l3[0], gauss);

		pressure_input->GetParameterValue(&pressure,gauss);
		t_pmp=meltingpoint-beta*pressure;

		scalar_ocean=gauss->weight*Jdet*rho_water*mixed_layer_capacity*thermal_exchange_velocity*(t_pmp)/(heatcapacity*rho_ice);
		if(dt) scalar_ocean=dt*scalar_ocean;

		for(i=0;i<numdof;i++) pe->values[i]+=scalar_ocean*l1l2l3[i];
	}

	/*Clean up and return*/
	delete gauss;
	return pe;
}
/*}}}*/
/*FUNCTION Tria::CreatePVectorThermalSheet {{{1*/
ElementVector* Tria::CreatePVectorThermalSheet(void){

	/*Constants*/
	const int  numdof=NUMVERTICES*NDOF1;

	/*Intermediaries */
	int        i,ig;
	int        analysis_type,drag_type;
	double     xyz_list[NUMVERTICES][3];
	double     Jdet,dt;
	double     rho_ice,heatcapacity,geothermalflux_value;
	double     basalfriction,alpha2,vx,vy,pressure;
	double     pressure_list[3];
	double     scalar;
	double     l1l2l3[NUMVERTICES];
	Friction*  friction=NULL;
	GaussTria* gauss=NULL;

	/*Initialize Element vector and return if necessary*/
	if(IsOnWater()) return NULL;
	ElementVector* pe=new ElementVector(nodes,NUMVERTICES,this->parameters);

	/*Retrieve all inputs and parameters*/
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);
	rho_ice=matpar->GetRhoIce();
	heatcapacity=matpar->GetHeatCapacity();
	this->parameters->FindParam(&dt,DtEnum);
	Input* vx_input=inputs->GetInput(VxEnum);                         ISSMASSERT(vx_input);
	Input* vy_input=inputs->GetInput(VyEnum);                         ISSMASSERT(vy_input);
	Input* vz_input=inputs->GetInput(VzEnum);                         ISSMASSERT(vz_input);
	Input* geothermalflux_input=inputs->GetInput(GeothermalFluxEnum); ISSMASSERT(geothermalflux_input);

	/*Build frictoin element, needed later: */
	inputs->GetParameterValue(&drag_type,DragTypeEnum);
	if (drag_type!=2)ISSMERROR(" non-viscous friction not supported yet!");
	friction=new Friction("3d",inputs,matpar,analysis_type);

	/* Start looping on the number of gauss 2d (nodes on the bedrock) */
	gauss=new GaussTria(2);
	for(ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		GetJacobianDeterminant2d(&Jdet, &xyz_list[0][0], gauss);
		GetNodalFunctions(&l1l2l3[0], gauss);

		geothermalflux_input->GetParameterValue(&geothermalflux_value,gauss);
		friction->GetAlpha2(&alpha2,gauss,VxEnum,VyEnum,VzEnum);
		
		scalar=gauss->weight*Jdet*(basalfriction+geothermalflux_value)/(heatcapacity*rho_ice);
		if(dt) scalar=dt*scalar;

		for(i=0;i<numdof;i++) pe->values[i]+=scalar*l1l2l3[i];
	}

	/*Clean up and return*/
	delete gauss;
	delete friction;
	return pe;
}
/*}}}*/
/*FUNCTION Tria::GetArea {{{1*/
double Tria::GetArea(void){

	double area=0;
	double xyz_list[NUMVERTICES][3];
	double x1,y1,x2,y2,x3,y3;

	/*Get xyz list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	x1=xyz_list[0][0]; y1=xyz_list[0][1];
	x2=xyz_list[1][0]; y2=xyz_list[1][1];
	x3=xyz_list[2][0]; y3=xyz_list[2][1];
 
	return (x2*y3 - y2*x3 + x1*y2 - y1*x2 + x3*y1 - y3*x1)/2;
}
/*}}}*/
/*FUNCTION Tria::GetElementType {{{1*/
int Tria::GetElementType(){

	/*return TriaRef field*/
	return this->element_type;

}
/*}}}*/
/*FUNCTION Tria::GetDofList {{{1*/
void  Tria::GetDofList(int** pdoflist, int approximation_enum,int setenum){

	int i,j;
	int count=0;
	int numberofdofs=0;
	int* doflist=NULL;

	/*First, figure out size of doflist and create it: */
	for(i=0;i<3;i++) numberofdofs+=nodes[i]->GetNumberOfDofs(approximation_enum,setenum);
	doflist=(int*)xmalloc(numberofdofs*sizeof(int));

	/*Populate: */
	count=0;
	for(i=0;i<3;i++){
		nodes[i]->GetDofList(doflist+count,approximation_enum,setenum);
		count+=nodes[i]->GetNumberOfDofs(approximation_enum,setenum);
	}

	/*Assign output pointers:*/
	*pdoflist=doflist;
}
/*}}}*/
/*FUNCTION Tria::GetDofList1 {{{1*/
void  Tria::GetDofList1(int* doflist){

	int i;
	for(i=0;i<3;i++) doflist[i]=nodes[i]->GetDofList1();

}
/*}}}*/
/*FUNCTION Tria::GetParameterListOnVertices(double* pvalue,int enumtype) {{{1*/
void Tria::GetParameterListOnVertices(double* pvalue,int enumtype){

	/*Intermediaries*/
	double     value[NUMVERTICES];
	GaussTria *gauss              = NULL;

	/*Recover input*/
	Input* input=inputs->GetInput(enumtype);
	if (!input) ISSMERROR("Input %s not found in element",EnumToString(enumtype));

	/*Checks in debugging mode*/
	ISSMASSERT(pvalue);

	/* Start looping on the number of vertices: */
	gauss=new GaussTria();
	for (int iv=0;iv<NUMVERTICES;iv++){
		gauss->GaussVertex(iv);
		input->GetParameterValue(&pvalue[iv],gauss);
	}

	/*clean-up*/
	delete gauss;
}
/*}}}*/
/*FUNCTION Tria::GetParameterListOnVertices(double* pvalue,int enumtype,double defaultvalue) {{{1*/
void Tria::GetParameterListOnVertices(double* pvalue,int enumtype,double defaultvalue){

	double     value[NUMVERTICES];
	GaussTria *gauss = NULL;
	Input     *input = inputs->GetInput(enumtype);

	/*Checks in debugging mode*/
	ISSMASSERT(pvalue);

	/* Start looping on the number of vertices: */
	if (input){
		gauss=new GaussTria();
		for (int iv=0;iv<NUMVERTICES;iv++){
			gauss->GaussVertex(iv);
			input->GetParameterValue(&pvalue[iv],gauss);
		}
	}
	else{
		for (int iv=0;iv<NUMVERTICES;iv++) pvalue[iv]=defaultvalue;
	}

	/*clean-up*/
	delete gauss;
}
/*}}}*/
/*FUNCTION Tria::GetParameterValue(double* pvalue,Node* node,int enumtype) {{{1*/
void Tria::GetParameterValue(double* pvalue,Node* node,int enumtype){

	Input* input=inputs->GetInput(enumtype);
	if(!input) ISSMERROR("No input of type %s found in tria",EnumToString(enumtype));

	GaussTria* gauss=new GaussTria();
	gauss->GaussVertex(this->GetNodeIndex(node));

	input->GetParameterValue(pvalue,gauss);
	delete gauss;
}
/*}}}*/
/*FUNCTION Tria::GetSolutionFromInputsDiagnosticHoriz{{{1*/
void  Tria::GetSolutionFromInputsDiagnosticHoriz(Vec solution){

	const int    numdof=NDOF2*NUMVERTICES;

	int          i;
	int*         doflist=NULL;
	double       vx,vy;
	double       values[numdof];
	GaussTria*   gauss=NULL;

	/*Get dof list: */
	GetDofList(&doflist,NoneApproximationEnum,GsetEnum);

	/*Get inputs*/
	Input* vx_input=inputs->GetInput(VxEnum); ISSMASSERT(vx_input);
	Input* vy_input=inputs->GetInput(VyEnum); ISSMASSERT(vy_input);

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	/*P1 element only for now*/
	gauss=new GaussTria();
	for(i=0;i<NUMVERTICES;i++){

		gauss->GaussVertex(i);

		/*Recover vx and vy*/
		vx_input->GetParameterValue(&vx,gauss);
		vy_input->GetParameterValue(&vy,gauss);
		values[i*NDOF2+0]=vx;
		values[i*NDOF2+1]=vy;
	}

	VecSetValues(solution,numdof,doflist,(const double*)values,INSERT_VALUES);

	/*Free ressources:*/
	delete gauss;
	xfree((void**)&doflist);
}
/*}}}*/
/*FUNCTION Tria::GetSolutionFromInputsDiagnosticHutter{{{1*/
void  Tria::GetSolutionFromInputsDiagnosticHutter(Vec solution){

	const int    numdof=NDOF2*NUMVERTICES;

	int i,dummy;
	int*         doflist=NULL;
	double       vx,vy;
	double       values[numdof];
	GaussTria*   gauss=NULL;

	/*Get dof list: */
	GetDofList(&doflist,NoneApproximationEnum,GsetEnum);

	/*Get inputs*/
	Input* vx_input=inputs->GetInput(VxEnum); ISSMASSERT(vx_input);
	Input* vy_input=inputs->GetInput(VyEnum); ISSMASSERT(vy_input);

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	/*P1 element only for now*/
	gauss=new GaussTria();
	for(i=0;i<NUMVERTICES;i++){

		gauss->GaussVertex(i);

		/*Recover vx and vy*/
		vx_input->GetParameterValue(&vx,gauss);
		vy_input->GetParameterValue(&vy,gauss);
		values[i*NDOF2+0]=vx;
		values[i*NDOF2+1]=vy;
	}

	VecSetValues(solution,numdof,doflist,(const double*)values,INSERT_VALUES);

	/*Free ressources:*/
	delete gauss;
	xfree((void**)&doflist);
}
/*}}}*/
/*FUNCTION Tria::GetStrainRate2d(double* epsilon,double* xyz_list, GaussTria* gauss, Input* vx_input, Input* vy_input){{{1*/
void Tria::GetStrainRate2d(double* epsilon,double* xyz_list, GaussTria* gauss, Input* vx_input, Input* vy_input){
	/*Compute the 2d Strain Rate (3 components):
	 * epsilon=[exx eyy exy] */

	int i;
	double epsilonvx[3];
	double epsilonvy[3];

	/*Check that both inputs have been found*/
	if (!vx_input || !vy_input){
		ISSMERROR("Input missing. Here are the input pointers we have for vx: %p, vy: %p\n",vx_input,vy_input);
	}

	/*Get strain rate assuming that epsilon has been allocated*/
	vx_input->GetVxStrainRate2d(epsilonvx,xyz_list,gauss);
	vy_input->GetVyStrainRate2d(epsilonvy,xyz_list,gauss);

	/*Sum all contributions*/
	for(i=0;i<3;i++) epsilon[i]=epsilonvx[i]+epsilonvy[i];
}
/*}}}*/
/*FUNCTION Tria::GradjDragStokes {{{1*/
void  Tria::GradjDragStokes(Vec gradient){

	int        i,ig;
	int        drag_type,analysis_type;
	int        doflist1[NUMVERTICES];
	double     bed,thickness,Neff;
	double     lambda,mu,xi,Jdet,vx,vy,vz;
	double     alpha_complement,drag,cm_noisedmp;
	double     surface_normal[3],bed_normal[3];
	double     xyz_list[NUMVERTICES][3];
	double     dh1dh3[NDOF2][NUMVERTICES];
	double     dk[NDOF2]; 
	double     l1l2l3[3];
	double     epsilon[3]; /* epsilon=[exx,eyy,exy];*/
	double     grade_g[NUMVERTICES]={0.0};
	double     grade_g_gaussian[NUMVERTICES];
	Friction*  friction=NULL;
	GaussTria* gauss=NULL;

	/*retrive parameters: */
	parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	/*retrieve inputs :*/
	inputs->GetParameterValue(&drag_type,DragTypeEnum);
	Input* drag_input    =inputs->GetInput(DragCoefficientEnum); ISSMASSERT(drag_input);
	Input* vx_input      =inputs->GetInput(VxEnum);              ISSMASSERT(vx_input);
	Input* vy_input      =inputs->GetInput(VyEnum);              ISSMASSERT(vy_input);
	Input* vz_input      =inputs->GetInput(VzEnum);              ISSMASSERT(vz_input);
	Input* adjointx_input=inputs->GetInput(AdjointxEnum);        ISSMASSERT(adjointx_input);
	Input* adjointy_input=inputs->GetInput(AdjointyEnum);        ISSMASSERT(adjointy_input);
	Input* adjointz_input=inputs->GetInput(AdjointzEnum);        ISSMASSERT(adjointz_input);

	/*retrieve some parameters: */
	this->parameters->FindParam(&cm_noisedmp,CmNoiseDmpEnum);

	/*Get out if shelf*/
	if(IsOnShelf())return;

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, NUMVERTICES);
	GetDofList1(&doflist1[0]);

	/*Build frictoin element, needed later: */
	inputs->GetParameterValue(&drag_type,DragTypeEnum);
	friction=new Friction("2d",inputs,matpar,analysis_type);

	/* Start  looping on the number of gaussian points: */
	gauss=new GaussTria(4);
	for(ig=gauss->begin();ig<gauss->end();ig++){

		gauss->GaussPoint(ig);

		/*Recover alpha_complement and drag: */
		if (drag_type==2) friction->GetAlphaComplement(&alpha_complement, gauss,VxEnum,VyEnum);
		else alpha_complement=0;
		drag_input->GetParameterValue(&drag,gauss);

		/*recover lambda mu and xi: */
		adjointx_input->GetParameterValue(&lambda,gauss);
		adjointy_input->GetParameterValue(&mu    ,gauss);
		adjointz_input->GetParameterValue(&xi    ,gauss);

		/*recover vx vy and vz: */
		vx_input->GetParameterValue(&vx, gauss);
		vy_input->GetParameterValue(&vy, gauss);
		vz_input->GetParameterValue(&vz, gauss);

		/*Get normal vector to the bed */
		SurfaceNormal(&surface_normal[0],xyz_list);

		bed_normal[0]=-surface_normal[0]; //Program is for surface, so the normal to the bed is the opposite of the result
		bed_normal[1]=-surface_normal[1];
		bed_normal[2]=-surface_normal[2];

		/* Get Jacobian determinant: */
		GetJacobianDeterminant3d(&Jdet, &xyz_list[0][0],gauss);

		/* Get nodal functions value at gaussian point:*/
		GetNodalFunctions(l1l2l3, gauss);

		/*Get nodal functions derivatives*/
		GetNodalFunctionsDerivatives(&dh1dh3[0][0],&xyz_list[0][0],gauss);

		/*Get k derivative: dk/dx */
		drag_input->GetParameterDerivativeValue(&dk[0],&xyz_list[0][0],gauss);

		/*Build gradje_g_gaussian vector (actually -dJ/ddrag): */
		for (i=0;i<NUMVERTICES;i++){
			//standard gradient dJ/dki
			grade_g_gaussian[i]=(
						-lambda*(2*drag*alpha_complement*(vx - vz*bed_normal[0]*bed_normal[2]))
						-mu    *(2*drag*alpha_complement*(vy - vz*bed_normal[1]*bed_normal[2]))
						-xi    *(2*drag*alpha_complement*(-vx*bed_normal[0]*bed_normal[2]-vy*bed_normal[1]*bed_normal[2]))
						)*Jdet*gauss->weight*l1l2l3[i]; 

			//Add regularization term
			grade_g_gaussian[i]+= - cm_noisedmp*Jdet*gauss->weight*(dh1dh3[0][i]*dk[0]+dh1dh3[1][i]*dk[1]);
		}

		/*Add gradje_g_gaussian vector to gradje_g: */
		for( i=0; i<NUMVERTICES; i++)grade_g[i]+=grade_g_gaussian[i];
	}

	VecSetValues(gradient,NUMVERTICES,doflist1,(const double*)grade_g,ADD_VALUES);

	delete friction;
	delete gauss;
}
/*}}}*/
/*FUNCTION Tria::InputUpdateFromSolutionAdjointBalancedthickness {{{1*/
void  Tria::InputUpdateFromSolutionAdjointBalancedthickness(double* solution){

	const int numdof=NDOF1*NUMVERTICES;

	int       i;
	int*      doflist=NULL;
	double    values[numdof];
	double    lambda[NUMVERTICES];

	/*Get dof list: */
	GetDofList(&doflist,NoneApproximationEnum,GsetEnum);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++) values[i]=solution[doflist[i]];

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	for(i=0;i<numdof;i++) lambda[i]=values[i];

	/*Add vx and vy as inputs to the tria element: */
	this->inputs->AddInput(new TriaVertexInput(AdjointEnum,lambda));

	/*Free ressources:*/
	xfree((void**)&doflist);
}
/*}}}*/
/*FUNCTION Tria::InputUpdateFromSolutionAdjointHoriz {{{1*/
void  Tria::InputUpdateFromSolutionAdjointHoriz(double* solution){

	const int numdof=NDOF2*NUMVERTICES;

	int       i;
	int*      doflist=NULL;
	double    values[numdof];
	double    lambdax[NUMVERTICES];
	double    lambday[NUMVERTICES];

	/*Get dof list: */
	GetDofList(&doflist,NoneApproximationEnum,GsetEnum);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++) values[i]=solution[doflist[i]];

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	for(i=0;i<NUMVERTICES;i++){
		lambdax[i]=values[i*NDOF2+0];
		lambday[i]=values[i*NDOF2+1];
	}

	/*Add vx and vy as inputs to the tria element: */
	this->inputs->AddInput(new TriaVertexInput(AdjointxEnum,lambdax));
	this->inputs->AddInput(new TriaVertexInput(AdjointyEnum,lambday));

	/*Free ressources:*/
	xfree((void**)&doflist);
}
/*}}}*/
/*FUNCTION Tria::InputUpdateFromSolutionDiagnosticHoriz {{{1*/
void  Tria::InputUpdateFromSolutionDiagnosticHoriz(double* solution){
	
	const int numdof=NDOF2*NUMVERTICES;

	int       i;
	int       dummy;
	int*      doflist=NULL;
	double    rho_ice,g;
	double    values[numdof];
	double    vx[NUMVERTICES];
	double    vy[NUMVERTICES];
	double    vz[NUMVERTICES];
	double    vel[NUMVERTICES];
	double    pressure[NUMVERTICES];
	double    thickness[NUMVERTICES];
	
	/*Get dof list: */
	GetDofList(&doflist,NoneApproximationEnum,GsetEnum);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++) values[i]=solution[doflist[i]];

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	for(i=0;i<NUMVERTICES;i++){
		vx[i]=values[i*NDOF2+0];
		vy[i]=values[i*NDOF2+1];
	}

	/*Get Vz and compute vel*/
	GetParameterListOnVertices(&vz[0],VzEnum,0);
	for(i=0;i<NUMVERTICES;i++) vel[i]=pow( pow(vx[i],2.0) + pow(vy[i],2.0) + pow(vz[i],2.0) , 0.5);

	/*For pressure: we have not computed pressure in this analysis, for this element. We are in 2D, 
	 *so the pressure is just the pressure at the bedrock: */
	rho_ice=matpar->GetRhoIce();
	g=matpar->GetG();
	GetParameterListOnVertices(&thickness[0],ThicknessEnum);
	for(i=0;i<NUMVERTICES;i++) pressure[i]=rho_ice*g*thickness[i];

	/*Now, we have to move the previous Vx and Vy inputs  to old 
	 * status, otherwise, we'll wipe them off: */
	this->inputs->ChangeEnum(VxEnum,VxOldEnum);
	this->inputs->ChangeEnum(VyEnum,VyOldEnum);
	this->inputs->ChangeEnum(PressureEnum,PressureOldEnum);

	/*Add vx and vy as inputs to the tria element: */
	this->inputs->AddInput(new TriaVertexInput(VxEnum,vx));
	this->inputs->AddInput(new TriaVertexInput(VyEnum,vy));
	this->inputs->AddInput(new TriaVertexInput(VelEnum,vel));
	this->inputs->AddInput(new TriaVertexInput(PressureEnum,pressure));

	/*Free ressources:*/
	xfree((void**)&doflist);

}
/*}}}*/
/*FUNCTION Tria::InputUpdateFromSolutionDiagnosticHutter {{{1*/
void  Tria::InputUpdateFromSolutionDiagnosticHutter(double* solution){
	
	const int numdof=NDOF2*NUMVERTICES;
	
	int       i;
	int       dummy;
	int*      doflist=NULL;
	double    rho_ice,g;
	double    values[numdof];
	double    vx[NUMVERTICES];
	double    vy[NUMVERTICES];
	double    vz[NUMVERTICES];
	double    vel[NUMVERTICES];
	double    pressure[NUMVERTICES];
	double    thickness[NUMVERTICES];
	double*   vz_ptr=NULL;
	Input*    vz_input=NULL;
	
	/*Get dof list: */
	GetDofList(&doflist,NoneApproximationEnum,GsetEnum);

	/*Use the dof list to index into the solution vector: */
	for(i=0;i<numdof;i++) values[i]=solution[doflist[i]];

	/*Ok, we have vx and vy in values, fill in vx and vy arrays: */
	for(i=0;i<NUMVERTICES;i++){
		vx[i]=values[i*NDOF2+0];
		vy[i]=values[i*NDOF2+1];
	}

	/*Get Vz*/
	vz_input=inputs->GetInput(VzEnum);
	if (vz_input){
		if (vz_input->Enum()!=TriaVertexInputEnum){
			ISSMERROR("Cannot compute Vel as Vz is of type %s",EnumToString(vz_input->Enum()));
		}
		vz_input->GetValuesPtr(&vz_ptr,&dummy);
		for(i=0;i<NUMVERTICES;i++) vz[i]=vz_ptr[i];
	}
	else{
		for(i=0;i<NUMVERTICES;i++) vz[i]=0.0;
	}

	/*Now Compute vel*/
	for(i=0;i<NUMVERTICES;i++) vel[i]=pow( pow(vx[i],2.0) + pow(vy[i],2.0) + pow(vz[i],2.0) , 0.5);

	/*For pressure: we have not computed pressure in this analysis, for this element. We are in 2D, 
	 *so the pressure is just the pressure at the bedrock: */
	rho_ice=matpar->GetRhoIce();
	g=matpar->GetG();
	GetParameterListOnVertices(&thickness[0],ThicknessEnum);
	for(i=0;i<NUMVERTICES;i++) pressure[i]=rho_ice*g*thickness[i];

	/*Now, we have to move the previous Vx and Vy inputs  to old 
	 * status, otherwise, we'll wipe them off: */
	this->inputs->ChangeEnum(VxEnum,VxOldEnum);
	this->inputs->ChangeEnum(VyEnum,VyOldEnum);
	this->inputs->ChangeEnum(PressureEnum,PressureOldEnum);

	/*Add vx and vy as inputs to the tria element: */
	this->inputs->AddInput(new TriaVertexInput(VxEnum,vx));
	this->inputs->AddInput(new TriaVertexInput(VyEnum,vy));
	this->inputs->AddInput(new TriaVertexInput(VelEnum,vel));
	this->inputs->AddInput(new TriaVertexInput(PressureEnum,pressure));

	/*Free ressources:*/
	xfree((void**)&doflist);
}
/*}}}*/
/*FUNCTION Tria::InputUpdateFromSolutionOneDof{{{1*/
void  Tria::InputUpdateFromSolutionOneDof(double* solution,int enum_type){

	const int numdof          = NDOF1*NUMVERTICES;

	int*      doflist=NULL;
	double    values[numdof];

	/*Get dof list: */
	GetDofList(&doflist,NoneApproximationEnum,GsetEnum);

	/*Use the dof list to index into the solution vector: */
	for(int i=0;i<numdof;i++) values[i]=solution[doflist[i]];

	/*Add input to the element: */
	this->inputs->AddInput(new TriaVertexInput(enum_type,values));

	/*Free ressources:*/
	xfree((void**)&doflist);
}
/*}}}*/
/*FUNCTION Tria::IsInput{{{1*/
bool Tria::IsInput(int name){
	if (
				name==ThicknessEnum ||
				name==SurfaceEnum ||
				name==BedEnum ||
				name==SurfaceSlopeXEnum ||
				name==SurfaceSlopeYEnum ||
				name==MeltingRateEnum ||
				name==AccumulationRateEnum ||
				name==SurfaceAreaEnum||
				name==ControlParameterEnum ||
				name==VxEnum ||
				name==VyEnum ||
				name==VxObsEnum ||
				name==VyObsEnum ||
				name==CmResponseEnum ||
				name==DragCoefficientEnum ||
				name==GradientEnum ||
				name==OldGradientEnum
		){
		return true;
	}
	else return false;
}
/*}}}*/
/*FUNCTION Tria::SetClone {{{1*/
void  Tria::SetClone(int* minranks){

	ISSMERROR("not implemented yet");
}
/*}}}1*/
/*FUNCTION Tria::SurfaceNormal{{{1*/
void Tria::SurfaceNormal(double* surface_normal, double xyz_list[3][3]){

	int i;
	double v13[3],v23[3];
	double normal[3];
	double normal_norm;

	for (i=0;i<3;i++){
		v13[i]=xyz_list[0][i]-xyz_list[2][i];
		v23[i]=xyz_list[1][i]-xyz_list[2][i];
	}

	normal[0]=v13[1]*v23[2]-v13[2]*v23[1];
	normal[1]=v13[2]*v23[0]-v13[0]*v23[2];
	normal[2]=v13[0]*v23[1]-v13[1]*v23[0];

	normal_norm=sqrt( pow(normal[0],(double)2)+pow(normal[1],(double)2)+pow(normal[2],(double)2) );

	*(surface_normal)=normal[0]/normal_norm;
	*(surface_normal+1)=normal[1]/normal_norm;
	*(surface_normal+2)=normal[2]/normal_norm;
}
/*}}}*/
