%RUNSTEPS class definition
%
%   Usage:
%      rs = runsteps(varargin)
%
%   Examples:
%      rs = runsteps;                             %build an empty runsteps object
%      rs = runsteps('../Models/');               %build an empty runsteps object with a given repository
%      rs = runsteps('../Models/','models.AGU.'); %build an empty runsteps object with a given repository and a prefix
%      rs = runsteps('../Models/','models.AGU.','models.default.Agu.'); %build an empty runsteps object with a given repository, a prefix and a default prefix, in case we are branching

classdef runsteps
    properties (SetAccess=public) 
		 % {{{1
		 repository='./';
		 prefix    ='model.';
		 defaultprefix    ='';
		 steps     =[];
		 %}}}
	 end
	 methods
		 function rs=runsteps(varargin) % {{{1
			 if (nargin==0),
				 %default constructor
			 elseif (nargin==1 & ischar(varargin{1})),

				 %Check repository
				 if exist(varargin{1},'dir')~=7,
					 error(['runsteps constructor error message: repository ' varargin{1} ' is not a directory']),
				 end
				 rs.repository=varargin{1};

			 elseif (nargin==2 & ischar(varargin{1}) & ischar(varargin{2})),

				 %Check repository
				 if exist(varargin{1},'dir')~=7,
					 error(['runsteps constructor error message: repository ' varargin{1} ' is not a directory']),
				 end
				 rs.repository=varargin{1};
				 rs.prefix    =varargin{2};
			 elseif (nargin==3 & ischar(varargin{1}) & ischar(varargin{2}) & ischar(varargin{3})),

				 %Check repository
				 if exist(varargin{1},'dir')~=7,
					 error(['runsteps constructor error message: repository ' varargin{1} ' is not a directory']),
				 end
				 rs.repository=varargin{1};
				 rs.prefix    =varargin{2};
				 rs.defaultprefix    =varargin{3};
			 else
				 help runsteps
				 error('runsteps constructor error message: bad usage')
			 end
		 end
		 %}}}
		 function disp(rs) % {{{1
			 disp(sprintf('\n%s = \n',inputname(1)));
			 disp(sprintf('   Repository: ''%s''',rs.repository));
			 disp(sprintf('   Prefix:     ''%s''',rs.prefix));
			 if isempty(rs.steps)
				 disp('   no step');
			 else
				 for i=1:length(rs.steps),
					 disp(sprintf('   step #%i',i));
					 disp(sprintf('      id:      %i',      rs.steps(i).id));
					 disp(sprintf('      message: ''%s''\n',rs.steps(i).message));
				 end
			 end
		 end
		 %}}}
		 function rs = addstep(rs,message),% {{{1

			 %check message
			 if ~ischar(message),
				 error('addstep error message: message provided should be a string');
			 end

			 rs.steps(end+1).id=length(rs.steps)+1;
			 rs.steps(end).message=message;
		 end% }}}
		 function bool=echo(rs,steps,varargin)% {{{1
			 if length(steps)==0,
				 error('steps is empty'),
			 elseif (length(steps)==1 & steps==0)
				 if isempty(rs.steps)
					 disp('   no step');
				 else
					 for i=1:length(rs.steps),
						 disp(sprintf('   step #%2i : %s',rs.steps(i).id,rs.steps(i).message));
					 end
				 end
				 bool=true;
				 return;
			 else
				 bool=false;
				 return;
			 end
		 end%}}}
		 function id=GetId(rs,message) % {{{1
			 %GetId - get step id from message
			 %
			 %   Usage:
			 %      GetId(rs,num)

			 for i=1:length(rs.steps),
				 if strcmp(rs.steps(i).message,message),
					 id=i;
					 return;
				 end
			 end

			 %If we are here, no step with given message has been found
			 error(['runsteps error messge: no step with message ''' message ''' has been found ']);
		 end%}}}
		 function md=loadmodel(rs,step,varargin),% {{{1
			 %LOADMODEL - save model for a given step
			 %
			 %   Usage:
			 %      md=loadmodel(rs,num,varargin)
			 %
			 %   Examples:
			 %      md=loadmodel(rs,num)
			 %      md=loadmodel(rs,'previous model')
			 %      md=loadmodel(rs,num,'2d')

			 %check inputs
			 if (nargin<2 | nargin>3),
				 help loamodel
				 error('loadmodel error message: bad usage');
			 end

			 %Get id
			 if ischar(step),
				 id=GetId(rs,step);
			 else
				 id=step;
			 end

			 %check rs length
			 if (id>length(rs.steps)),
				 error(['runsteps error message: element with id ' num2str(num) ' not found']);
			 end

			 %get type if provided
			 if nargin~=3
				 path=[rs.repository '/' rs.prefix num2str(id)];
			 elseif ischar(varargin{1}),
				 path=[rs.repository '/' rs.prefix num2str(id) '.' varargin{1} ];
			 else
				 error('loadmodel error message: third argument should be a string');
			 end
			 %figure out if the model is there, otherwise, we have to use the default path supplied by user.
			 if exist(path,'file'),

				 %load model
				 md=loadmodel(path);
			 else
				 if isempty(rs.defaultprefix),
					 error('loadmodel error message: cannot find model, and default prefix was not supplied');
				 else

					 if nargin~=3
						 path=[rs.repository '/' rs.defaultprefix num2str(id)];
					 elseif ischar(varargin{1}),
						 path=[rs.repository '/' rs.defaultprefix num2str(id) '.' varargin{1} ];
					 else
						 error('loadmodel error message: third argument should be a string');
					 end
				 end
				 %load model
				 md=loadmodel(path);
			 end
		 end%}}}
		 function message(rs,num) % {{{1
			 %MESSAGE - display message of rs with id =num
			 %
			 %   Usage:
			 %      message(rs,num)

			 %Some checks
			 if (abs(floor(num))~=num),
				 error(['runsteps error message: the number provided is not a positive integer']);
			 end
			 if (num>length(rs.steps)),
				 error(['runsteps error message: element with id ' num2str(num) ' not found']);
			 end

			 %Print message
			 disp(sprintf('\n   step #%i -> %s\n',rs.steps(num).id,rs.steps(num).message));
		 end%}}}
		 function savemodel(rs,num,varargin) % {{{1
			 %SAVEMODEL - save model for a given step
			 %
			 %   Usage:
			 %      savemodel(rs,num,md)
			 %
			 %   Examples:
			 %      savemodel(rs,num,md)
			 %      savemodel(rs,num,'2d',md)

			 %check inputs
			 if nargin==3 
				 md=varargin{1};
			 elseif nargin==4
				 type=varargin{1};
				 md=varargin{2};
			 else
				 help loamodel
				 error('savemodel error message: bad usage');
			 end

			 %check that md is a model
			 if ~isa(md,'model'),
				 error('savemodel error message: third argument is not a model')
			 end

			 %check rs length
			 if (num>length(rs.steps)),
				 error(['runsteps error message: element with id ' num2str(num) ' not found']);
			 end

			 %before saving model, try and find a runme.m file, and save it.
			 if isnans(md.runmefile),
				 md.runmefile=cell(0,1);
				 md.runmefile{1,1}=char(textread('runme.m','%s','delimiter','\n'));
			 else
				 md.runmefile{end+1,1}=char(textread('runme.m','%s','delimiter','\n'));
			 end

			 %save model
			 if nargin==3,
				 name=[rs.repository '/' rs.prefix num2str(num)];
			 else
				 name=[rs.repository '/' rs.prefix num2str(num) '.' type];
			 end

			 save(name,'md','-v7.3');
			 disp(['model saved as: ' name]);
		 end%}}}
	end
end
