/*!\file ElementVector.cpp
 * \brief: implementation of the ElementVector object, used to plug values from element into global load
 */

/*Headers:*/
/*{{{1*/
#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include <string.h>
#include "../objects.h"
#include "../../shared/shared.h"
#include "../../Container/Container.h"
#include "../../include/include.h"
/*}}}*/

/*ElementVector constructors and destructor*/
/*FUNCTION ElementVector::ElementVector(){{{1*/
ElementVector::ElementVector(){

	this->nrows=0;
	this->values=NULL;
	this->fsize=0;
	this->flocaldoflist=NULL;
	this->fglobaldoflist=NULL;

}
/*}}}*/
/*FUNCTION ElementVector::ElementVector(ElementVector* pe1, ElementVector* pe2){{{1*/
ElementVector::ElementVector(ElementVector* pe1, ElementVector* pe2){

	/*intermediaries*/
	int i,j,counter;
	int gsize,fsize,ssize;
	int* P=NULL;
	bool found;

	/*If one of the two matrix is NULL, we copy the other one*/
	if(!pe1 && !pe2){
		ISSMERROR("Two input element matrices are NULL");
	}
	else if(!pe1){
		this->Init(pe2);
		return;
	}
	else if(!pe2){
		this->Init(pe1);
		return;
	}

	/*Initialize itransformation matrix pe[P[i]] = pe2[i]*/
	P=(int*)xmalloc(pe2->nrows*sizeof(int));

	/*1: Get the new numbering of pe2 and get size of the new matrix*/
	gsize=pe1->nrows;
	for(i=0;i<pe2->nrows;i++){
		found=false;
		for(j=0;j<pe1->nrows;j++){
			if(pe2->gglobaldoflist[i]==pe1->gglobaldoflist[j]){
				found=true; P[i]=j; break;
			}
		}
		if(!found){
			P[i]=gsize; gsize++;
		}
	}

	/*2: Initialize static fields*/
	this->nrows=gsize;
	this->pf=pe1->pf;

	/*Gset and values*/
	this->gglobaldoflist=(int*)xmalloc(this->nrows*sizeof(int));
	this->values=(double*)xcalloc(this->nrows,sizeof(double));
	for(i=0;i<pe1->nrows;i++){
		this->values[i] += pe1->values[i];
		this->gglobaldoflist[i]=pe1->gglobaldoflist[i];
	}
	for(i=0;i<pe2->nrows;i++){
		this->values[P[i]] += pe2->values[i];
		this->gglobaldoflist[P[i]]=pe2->gglobaldoflist[i];
	}

	/*Fset*/
	fsize=pe1->fsize;
	for(i=0;i<pe2->fsize;i++){
		if(P[pe2->flocaldoflist[i]] >= pe1->nrows) fsize++;
	}
	this->fsize=fsize;
	if(fsize){
		this->flocaldoflist =(int*)xmalloc(fsize*sizeof(int));
		this->fglobaldoflist=(int*)xmalloc(fsize*sizeof(int));
		for(i=0;i<pe1->fsize;i++){
			this->flocaldoflist[i] =pe1->flocaldoflist[i];
			this->fglobaldoflist[i]=pe1->fglobaldoflist[i];
		}
		counter=pe1->fsize;
		for(i=0;i<pe2->fsize;i++){
			if(P[pe2->flocaldoflist[i]] >= pe1->nrows){
				this->flocaldoflist[counter] =P[pe2->flocaldoflist[i]];
				this->fglobaldoflist[counter]=pe2->fglobaldoflist[i];
				counter++;
			}
		}
	}
	else{
		this->flocaldoflist=NULL;
		this->fglobaldoflist=NULL;
	}

	/*clean-up*/
	xfree((void**)&P);
}
/*}}}*/
/*FUNCTION ElementVector::ElementVector(ElementVector* pe1, ElementVector* pe2,ElementVector* pe3){{{1*/
ElementVector::ElementVector(ElementVector* pe1, ElementVector* pe2,ElementVector* pe3){

	/*Concatenate all matrices*/
	ElementVector* pe12 =new ElementVector(pe1,pe2);
	ElementVector* pe123=new ElementVector(pe12,pe3);

	/*Initialize current object with this matrix*/
	this->Init(pe123);

	/*clean-up*/
	delete pe12;
	delete pe123;
}
/*}}}*/
/*FUNCTION ElementVector::ElementVector(Node** nodes,int numnodes,Parameters* parameters,int approximation){{{1*/
ElementVector::ElementVector(Node** nodes,int numnodes,Parameters* parameters,int approximation){

	/*retrieve some parameters: */
	parameters->FindParam(&this->pf,KffEnum);

	/*get Vector size and properties*/
	this->nrows=GetNumberOfDofs(nodes,numnodes,GsetEnum,approximation);

	/*fill values with 0: */
	this->values=(double*)xcalloc(this->nrows,sizeof(double));
	
	/*g list*/
	this->gglobaldoflist=GetGlobalDofList(nodes,numnodes,GsetEnum,approximation);

	/*Get fsize*/
	if(pf){
		this->fsize=GetNumberOfDofs(nodes,numnodes,FsetEnum,approximation);
		this->flocaldoflist =GetLocalDofList( nodes,numnodes,FsetEnum,approximation);
		this->fglobaldoflist=GetGlobalDofList(nodes,numnodes,FsetEnum,approximation);
	}
	else{
		this->fsize=0;
		this->flocaldoflist =NULL;
		this->fglobaldoflist=NULL;
	}
}
/*}}}*/
/*FUNCTION ElementVector::~ElementVector(){{{1*/
ElementVector::~ElementVector(){
	
	xfree((void**)&this->values);
	xfree((void**)&this->gglobaldoflist);
	xfree((void**)&this->flocaldoflist);
	xfree((void**)&this->fglobaldoflist);
}
/*}}}*/

/*ElementVector specific routines: */
/*FUNCTION ElementVector::AddToGlobal(Vec pg, Vec pf){{{1*/
void ElementVector::AddToGlobal(Vec pg, Vec pf){

	int i;
	double* localvalues=NULL;

	if(!pf){
		/*add local values into global  vector, using the fglobaldoflist: */
		VecSetValues(pg,this->nrows,this->gglobaldoflist,(const double*)values,ADD_VALUES);
	}
	else{
		if(this->fsize){
			/*first, retrieve values that are in the f-set from the g-set values vector: */
			localvalues=(double*)xmalloc(this->fsize*sizeof(double));
			for(i=0;i<this->fsize;i++){
				localvalues[i]=this->values[this->flocaldoflist[i]];
			}
			/*add local values into global  vector, using the fglobaldoflist: */
			VecSetValues(pf,this->fsize,this->fglobaldoflist,(const double*)localvalues,ADD_VALUES);

			/*Free ressources:*/
			xfree((void**)&localvalues);
		}
	}
}
/*}}}*/
/*FUNCTION ElementVector::Echo{{{1*/
void ElementVector::Echo(void){

	int i,j;
	printf("Element Vector echo: \n");
	printf("   nrows: %i\n",nrows);
	printf("   pf: %s\n",pf?"true":"false");

	printf("   values: \n");
	for(i=0;i<nrows;i++){
		printf("      %i: %10g\n",i,values[i]);
	}

	printf("   gglobaldoflist (%p): ",gglobaldoflist);
	if(gglobaldoflist) for(i=0;i<nrows;i++)printf("%i ",gglobaldoflist[i]); printf("\n");

	printf("   fsize: %i\n",fsize);
	printf("   flocaldoflist (%p): ",flocaldoflist);
	if(flocaldoflist) for(i=0;i<fsize;i++)printf("%i ",flocaldoflist[i]); printf("\n");
	printf("   fglobaldoflist (%p): ",fglobaldoflist);
	if(fglobaldoflist)for(i=0;i<fsize;i++)printf("%i ",fglobaldoflist[i]); printf("\n");
}
/*}}}*/
/*FUNCTION ElementVector::Init{{{1*/
void ElementVector::Init(ElementVector* pe){

	ISSMASSERT(pe);

	this->nrows =pe->nrows;
	this->pf    =pe->pf;

	this->values=(double*)xmalloc(this->nrows*sizeof(double));
	memcpy(this->values,pe->values,this->nrows*sizeof(double));

	this->gglobaldoflist=(int*)xmalloc(this->nrows*sizeof(int));
	memcpy(this->gglobaldoflist,pe->gglobaldoflist,this->nrows*sizeof(int));

	this->fsize=pe->fsize;
	if(this->fsize){
		this->flocaldoflist=(int*)xmalloc(this->fsize*sizeof(int));
		memcpy(this->flocaldoflist,pe->flocaldoflist,this->fsize*sizeof(int));
		this->fglobaldoflist=(int*)xmalloc(this->fsize*sizeof(int));
		memcpy(this->fglobaldoflist,pe->fglobaldoflist,this->fsize*sizeof(int));
	}
	else{
		this->flocaldoflist=NULL;
		this->fglobaldoflist=NULL;
	}
}
/*}}}*/
